﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/vi/sf/vi_ServiceTypes.h>
#include "../visrv_Config.h"
#include "../vic/visrv_VicModule.h"
#include "../vic/visrv_VicMemoryPool.h"
#include "../vic/visrv_VicImageBuffer.h"
#include "../vic/visrv_VicTaskQueue.h"
#include <nvrm_channel.h>
#include <ui/GraphicBuffer.h>

#include "../visrv_ServerManager.h"
#include "visrv_IndirectConsumerBindState.h"

namespace nn{ namespace visrv{ namespace indirect{

    class IndirectLayerBufferStage;
    class IndirectLockCounter;

    struct IndirectMapCopyImageContextParameter
    {
        void*  pBuffer;
        size_t bufferSize;
        int    width;
        int    height;
        float  sourceRectX;
        float  sourceRectY;
        float  sourceRectWidth;
        float  sourceRectHeight;

        nn::vi::IndirectConsumerHandleType consumerHandle;
        IndirectConsumerBoundId            destinationBoundId;

        IndirectLayerBufferStage* pSourceBufferStage;
        IndirectLockCounter*      pSourceBufferLockCounter;
        android::sp<android::GraphicBuffer> pSourceGraphicBuffer;

        IndirectLockCounter*      pDestinationBoundLockCounter;
    };

    class IndirectMapCopyImageContext
        : public vic::IVicTask
    {
    public:
        typedef IndirectMapCopyImageContextParameter Parameter;
    public:
        IndirectMapCopyImageContext() NN_NOEXCEPT;

        nn::Result Initialize(const Parameter& param) NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        bool IsInitialized() const NN_NOEXCEPT;
        Parameter GetParameter() const NN_NOEXCEPT;

        nn::Result GetOutResult() const NN_NOEXCEPT;
        size_t GetOutImageSize() const NN_NOEXCEPT;
        size_t GetOutImageStride() const NN_NOEXCEPT;

        nn::os::EventType* GetCompleteEvent() NN_NOEXCEPT;
        nn::os::MultiWaitHolderType* GetCompleteEventHolder() NN_NOEXCEPT;

        virtual void Execute() NN_NOEXCEPT NN_OVERRIDE;

    private:
        nn::Result ExecuteImpl() NN_NOEXCEPT;

        // ExecuteImpl() から呼ばれる
        nn::Result InitializeImageBuffer(void* pBuffer, size_t bufferSize, int widht, int height) NN_NOEXCEPT;

        // ExecuteImpl() から呼ばれる
        void       FinalizeImageBuffer() NN_NOEXCEPT;

        // ExecuteImpl() から呼ばれる
        nn::Result KickVicCopy(NvRmFence* pOutFence) NN_NOEXCEPT;

        // 指定した ReleaseFence を設定してコピー元バッファのロックを解除する。
        // これ以降、コピー元バッファの所有権を IndirectLayer が持っていることが保証されなくなる。
        // Producer の Flip や Disconnect, Unbind あるいは Layer の Destroy が発生する可能性がある。
        // ExecuteImpl() から呼ばれる
        void UnlockSourceBuffer(const NvRmFence& fence) NN_NOEXCEPT;

        // コピー先バッファのロックを解除する。
        // これ以降、Consumer の Unbind あるいは Layer の Destroy が発生する可能性がある。
        // ExecuteImpl() から呼ばれる
        void UnlockDestinationBuffer() NN_NOEXCEPT;

        // 指定した Fence を待つ
        // ExecuteImpl() から呼ばれる
        void WaitSyncpoint(const NvRmFence& fence) NN_NOEXCEPT;

        // コピー完了を通知する。
        // これ以降、このコンテキストが破棄される可能性があるため、 *this に対する操作を行ってはならない。
        // ExecuteImpl() から呼ばれる
        void SignalCompleteEvent() NN_NOEXCEPT;

    private:
        bool      m_IsInitialized;
        Parameter m_Parameter;

        nn::Result m_OutResult;
        size_t m_OutImageSize;
        size_t m_OutImageStride;

        nn::os::EventType           m_CompleteEvent;
        nn::os::MultiWaitHolderType m_CompleteEventHolder;

        vic::VicMemoryPool  m_MemoryPool;
        vic::VicImageBuffer m_ImageBuffer;
    };

    class IndirectMapCopyImageContextTable
    {
    public:
        static const int Capacity = IndirectLayerMapCopyContextCountMax;
    public:
        struct DeferringInfo
        {
            nn::os::MultiWaitHolderType* pSession;
            ServerManager::DisplayServerManager* pServerManager;
        };

        enum State
        {
            State_Free = 0,
            State_Used,
        };
    public:
        void Initialize() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;

        nn::Result Acquire(IndirectMapCopyImageContext** pOut) NN_NOEXCEPT;
        void Release(IndirectMapCopyImageContext* p) NN_NOEXCEPT;
        IndirectMapCopyImageContext* GetCurrentContext() NN_NOEXCEPT;

        int GetIndexOf(const IndirectMapCopyImageContext* p) const NN_NOEXCEPT;
        void DeferExecution(IndirectMapCopyImageContext* p) NN_NOEXCEPT;
        void ResumeExecution(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

    private:
        static void DeferExecutionHandler(ServerManager::DisplayServerManager* pManager, nn::os::MultiWaitHolderType* pSession, void* pUserPtr) NN_NOEXCEPT;

    public:
        NN_ALIGNAS(NN_ALIGNOF(IndirectMapCopyImageContext)) char m_Memory[sizeof(IndirectMapCopyImageContext) * Capacity];
        DeferringInfo m_DeferringInfo[Capacity];
        State m_StateList[Capacity];
        int m_CurrentContextIndex;
    };

    extern IndirectMapCopyImageContextTable g_IndirectMapCopyImageContextTable;

}}}
