﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/os.h>
#include <nn/sf/sf_NativeHandleFwd.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/vi/vi_CropRegion.h>
#include <nn/vi/vi_ImageTransform.h>
#include <nn/vi/fbshare/vi_SharedLayerHandle.h>
#include <nn/vi/fbshare/vi_SharedLayerTextureIndexList.h>
#include "../../visrv_Config.h"
#include "../../util/visrv_TQueue.h"
#include "../../util/visrv_TIntegerContainer.h"
#include "../../native/visrv_SyncpointWaiter.h"

namespace nn{ namespace visrv{ namespace client{
    class ClientObject;
}}}

namespace nn{ namespace visrv{ namespace fbshare{
    class SharedBuffer;
}}}

namespace nn{ namespace visrv{ namespace fbshare{ namespace detail{
    class SharedLowLevelLayer;
    class SharedFrameBuffer;

    enum SharedClientLayerState
    {
        // 無効なオブジェクト
        SharedClientLayerState_Invalid,
        // 初期化済
        SharedClientLayerState_Initialized,
        // バインド済
        SharedClientLayerState_Bound,
        // 接続済
        SharedClientLayerState_Connected,
    };

    enum SharedClientLayerAttachState
    {
        // LowLevelLayer にアタッチされていない
        SharedClientLayerAttachState_Detached,
        // LowLevelLayer にアタッチ済
        SharedClientLayerAttachState_Attached,
        // LowLevelLayer からデタッチ中
        SharedClientLayerAttachState_Detaching,
    };

    class SharedClientLayer
    {
    public:
        static const int FrameBufferCountMin = SharedAttachedFrameBufferCountMinPerClientLayer;
        static const int FrameBufferCountMax = SharedAttachedFrameBufferCountMaxPerClientLayer;

        enum LocalMultiWaitTag
        {
            LocalMultiWaitTag_DetachAcquireFenceWaitMin = 0,
            LocalMultiWaitTag_DetachAcquireFenceWaitMax = LocalMultiWaitTag_DetachAcquireFenceWaitMin + FrameBufferCountMax - 1,
            LocalMultiWaitTag_Max,
        };

        static const int MultiWaitTagCount = LocalMultiWaitTag_Max;

    private:
        typedef util::TQueue<int, FrameBufferCountMax, false> FrameBufferIndexQueue;
        typedef util::TIntegerContainer<int, FrameBufferCountMax, -1, util::ContainerThreadSafetyOption_SingleThread> FrameBufferIndexContainer;

        struct MultiWaitAdaptor
        {
        public:
            MultiWaitAdaptor() NN_NOEXCEPT;
            uintptr_t GetLocalTag(uintptr_t globalTag) const NN_NOEXCEPT;
            void LinkMultiWait(nn::os::MultiWaitHolderType* pHolder, uintptr_t localTag) NN_NOEXCEPT;
            void UnlinkMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;
        public:
            uintptr_t tagBase;
        };

        struct FenceWait
        {
            public:
                void Activate(int frameBufferIndex, const android::sp<android::Fence>& fence, MultiWaitAdaptor& multiWaitAdaptor, uintptr_t localTag) NN_NOEXCEPT;
                void Deactivate(bool isAlreadyUnlinked) NN_NOEXCEPT;
            public:
                native::SyncpointEntry      entry;
                nn::os::MultiWaitHolderType holder;
                bool                        isActive;
        };

    public:
        SharedClientLayer() NN_NOEXCEPT;

    //--------------------
    // Getter
    //--------------------
    public:
        // @pre true
        bool IsInitialized() const NN_NOEXCEPT;
        // @pre true
        bool IsBound() const NN_NOEXCEPT;
        // @pre true
        bool IsConnected() const NN_NOEXCEPT;
        // @pre true
        bool IsAttached() const NN_NOEXCEPT;
        // @pre true
        bool IsDetaching() const NN_NOEXCEPT;

        // @pre true
        SharedClientLayerState GetState() const NN_NOEXCEPT;
        // @pre true
        SharedClientLayerAttachState GetAttachState() const NN_NOEXCEPT;
        // @pre true
        nn::vi::fbshare::SharedLayerHandle GetHandle() const NN_NOEXCEPT;

        // @pre IsInitialized
        client::ClientObject* GetOwnerClient() NN_NOEXCEPT;
        // @pre IsBound
        client::ClientObject* GetUserClient() NN_NOEXCEPT;
        // @pre IsAttached
        SharedLowLevelLayer* GetAttachedLowLevelLayer() NN_NOEXCEPT;
        // @pre IsInitialized
        nn::sf::NativeHandle GetAcquirableSystemEventHandle() NN_NOEXCEPT;
        // @pre IsInitialized
        nn::sf::NativeHandle GetDetachReadySystemEventHandle() NN_NOEXCEPT;

        // @pre IsInitialized
        nn::vi::LayerStackFlagType GetLayerStackFlags() const NN_NOEXCEPT;

    //--------------------
    // Setter
    //--------------------
    public:
        // @pre IsInitialized
        void SetLayerStackFlags(nn::vi::LayerStackFlagType stacks) NN_NOEXCEPT;

    //--------------------
    // StateOperations
    //--------------------
    public:
        void SetupMultiWaitAdaptor(uintptr_t tagBase) NN_NOEXCEPT;
        void DispatchMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

        // @pre State == Invalid
        // @pre AttachState == Detached
        // @post State == Initialized
        // pOwnerClient の値は変更されない
        nn::Result Initialize(
            nn::vi::fbshare::SharedLayerHandle* pOutHandle,
            client::ClientObject* pOwnerClient,
            nn::applet::AppletResourceUserId userAruid
        ) NN_NOEXCEPT;

        // @pre State == Initialized
        // @pre AttachState == Detached
        // @post State == Invalid
        void Finalize() NN_NOEXCEPT;

        // @pre State == Initailized
        // @post State == Bound
        // pObject の値は変更されない
        nn::Result Bind(client::ClientObject* pObject, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

        // @pre State == Bound
        // @post State == Initialized
        void Unbind() NN_NOEXCEPT;

        // @pre State == Bound
        // @post State == Connected
        nn::Result Connect() NN_NOEXCEPT;

        // @pre State == Connected
        // @post State == Bound
        void Disconnect() NN_NOEXCEPT;

        // @pre State in {Initialized, Bound, Connected}
        // @pre AttachState == Detached
        // @pre LowLayer.State == Connected
        // @post AttachState == Attached
        // @details
        // pLowLayer の値は変更されないため pLower 側のアタッチ処理も別途必要。pLowLayer のアタッチ処理との呼出順は任意。
        //
        // 指定できる frame buffer は以下を満たすこと
        //   - どの SharedClientLayer にもアタッチされていない
        //   - pLowerLayer でない SharedLowLevelLayer にアタッチされていない
        nn::Result Attach(
            SharedLowLevelLayer* pLowLayer,
            const nn::vi::fbshare::SharedLayerTextureIndexList& frameBufferIndexList
        ) NN_NOEXCEPT;

        // @pre AttachState == Attached
        // @post AttachState == Detaching
        // @details
        // 安全なデタッチ処理を開始する。
        //
        // 安全なデタッチ処理では、以下の処理を行う
        // - 1. Client からの AcquireBuffer の停止
        // - 2. 以下の条件が揃うまで待つ
        //     - 2.1. Client→Presented→Queueing のバッファがなくなる
        //       - Queueing から Display に移すとき
        //       - Client から Acquirable に戻すとき
        //       - Disconnect するとき
        //     - 2.2. すべてのバッファの AcquireFence が Expire する（MultiWait のハンドラで判定）
        //
        // (2.2. で待つべき AcquireFence は Client が Present してくるまで確定しないので
        //  先に 2.1. の条件を満たしたあとで 2.2. を判定＆処理する)
        //
        // これによってクライアントの描画処理が完了することが保証される。
        // ディスプレイとの同期は行わないため、別途 LowLevelLayer に対して SynchronizeDisplay する必要がある。
        //
        // デタッチが完了すると DetachReadyEvent がシグナルする。
        // 安全なデタッチ処理中に ForceDetach() が呼ばれると中断して強制デタッチする。
        nn::Result StartDetach() NN_NOEXCEPT;

        // @pre IsDetaching
        // @details
        // 安全なデタッチ処理を完了させる。
        // 安全なデタッチ処理の条件がそろっているか確認してデタッチを行う。
        // 条件がそろっていない場合、 ResultNotReady が返る。
        nn::Result FinishDetach() NN_NOEXCEPT;

        // @pre IsAttached
        // @post AttachState == Detached
        // @details
        // pLowLayer の値は変更されないため pLowLayer 側のデタッチ処理も別途必要。pLowLayer のデタッチ処理との呼出順は任意。
        //
        // すべての FrameBuffer を即座にデタッチする。
        // FrameBuffer の AcquireFence (GPU fence) は待たずに放棄される。
        // この関数の終了後も FrameBuffer への書込みが継続する可能性がある。
        void ForceDetach() NN_NOEXCEPT;

    //--------------------
    // FrameBufferLoopOperations
    //--------------------
    public:
        // @pre IsConnected
        nn::Result Acquire(
            int* pOutFrameBufferIndex,
            native::NativeRmSync* pOutSync,
            nn::vi::fbshare::SharedLayerTextureIndexList* pOutBufferIndexList
        ) NN_NOEXCEPT;

        // @pre IsConnected
        nn::Result Present(
            int frameBufferIndex,
            const native::NativeRmSync& sync,
            const nn::vi::CropRegion& crop,
            nn::vi::ImageTransformType transform,
            int presentInterval
        ) NN_NOEXCEPT;

        // @pre IsConnected
        nn::Result Cancel(
            int frameBufferIndex
        ) NN_NOEXCEPT;

        // @pre IsConnected
        // @pre IsAttached
        // 大体デバッグ用。
        // Acquire 済のバッファの色を塗りつぶします。
        nn::Result FillBufferColor(int frameBufferIndex, int r, int g, int b, int a) NN_NOEXCEPT;

        // @pre IsAttached
        void NotifyBufferIdle() NN_NOEXCEPT;
        // @pre IsAttached
        void NotifyBufferDequeued(int frameBufferIndex) NN_NOEXCEPT;

    private:
        // @pre IsAttached
        // 自動的に ProcessEntranceImpl() が呼ばれる
        void PushBufferToEntranceImpl(int frameBufferIndex) NN_NOEXCEPT;
        // @pre IsAttached
        void ProcessEntranceImpl() NN_NOEXCEPT;

        // @pre IsAttached
        void PushBufferToDisplayImpl(int frameBufferIndex) NN_NOEXCEPT;
        // @pre IsAttached
        // NotifyBufferDequeued の中で呼ばれる
        void ProcessDisplayDequeueingImpl(int frameBufferIndex) NN_NOEXCEPT;

        // @pre IsAttached
        // 自動的に UpdateAcquireEventImpl() が呼ばれる
        void PushBufferToAcquirableImpl(int frameBufferIndex) NN_NOEXCEPT;
        // @pre IsAttached
        // 自動的に UpdateAcquireEventImpl() が呼ばれる
        void CancelBufferToAcquirableImpl(int frameBufferIndex) NN_NOEXCEPT;
        // @pre IsAttached
        // Acquirable なバッファの有無によってシグナル状態を変更する
        void UpdateAcquirableEventImpl() NN_NOEXCEPT;

        // @pre IsAttached
        void PushBufferToClientImpl(int frameBufferIndex) NN_NOEXCEPT;

        // @pre IsAttached
        void PushBufferToPresentedImpl(int frameBufferIndex) NN_NOEXCEPT;

        // @pre IsAttached
        // 自動的に ProcessQueueingImpl() が呼ばれる
        void PushBufferToQueueingImpl(int frameBufferIndex) NN_NOEXCEPT;
        // @pre IsAttached
        // 先頭のバッファをディスプレイに送る
        void ProcessQueueingImpl() NN_NOEXCEPT;

    //--------------------
    // DetachOperations
    //--------------------
    private:
        class DetachOperation
        {
        public:
            enum Progress
            {
                Progress_Inactive,
                Progress_WaitingForAllBuffersQueued,
                Progress_WaitingForAllAcquireFencesExpired,
                Progress_DetachReady,
            };

        public:
            // @pre IsDetaching
            static bool IsDetachReady(const SharedClientLayer* pSelf) NN_NOEXCEPT;

            // @pre AttachState == Attached
            static void StartDetach(SharedClientLayer* pSelf) NN_NOEXCEPT;

            // @pre IsDetaching
            static nn::Result FinishDetach(SharedClientLayer* pSelf) NN_NOEXCEPT;

            static void ForceDetach(SharedClientLayer* pSelf) NN_NOEXCEPT;

            // @pre IsDetaching
            static void ProcessBufferQueued(SharedClientLayer* pSelf, int iBuf) NN_NOEXCEPT;

            // @pre IsDetaching
            static void ProcessBufferCanceled(SharedClientLayer* pSelf, int iBuf) NN_NOEXCEPT;

            // @pre IsDetaching
            static void ProcessAcquireFenceExpired(SharedClientLayer* pSelf, int iWin) NN_NOEXCEPT;

            // @pre IsDetaching
            static void ProcessDisconnected(SharedClientLayer* pSelf) NN_NOEXCEPT;

        private:
            static bool CheckAllBuffersQueuedImpl(const SharedClientLayer* pSelf) NN_NOEXCEPT;
            static bool CheckAllAcquireFencesExpiredImpl(const SharedClientLayer* pSelf) NN_NOEXCEPT;

            static void StartWaitForAllBuffersQueuedImpl(SharedClientLayer* pSelf) NN_NOEXCEPT;
            static void StartWaitForAllAcquireFencesExpiredImpl(SharedClientLayer* pSelf) NN_NOEXCEPT;
            static void NotifySafeDetachReadyImpl(SharedClientLayer* pSelf) NN_NOEXCEPT;

            static void ProcessBufferQueueStateChangedImpl(SharedClientLayer* pSelf) NN_NOEXCEPT;
            static void ProcessAcquireFenceChangedImpl(SharedClientLayer* pSelf) NN_NOEXCEPT;

        };

    private:
        friend class DetachOperation;

        SharedClientLayerState       m_State;
        SharedClientLayerAttachState m_AttachState;
        nn::vi::fbshare::SharedLayerHandle    m_Handle;

        MultiWaitAdaptor m_MultiWaitAdaptor;

        struct
        {
            nn::applet::AppletResourceUserId userAruid;
            client::ClientObject*            pOwnerClient;
            nn::os::SystemEventType acquirableSystemEvent;
            nn::os::SystemEventType detachReadySystemEvent;
        } m_CreationStatus;

        struct
        {
            client::ClientObject*   pUserClient;
        } m_BindingStatus;

        struct
        {
        } m_ConnectionStatus;

        struct AttachmentStatus
        {
            SharedLowLevelLayer* pLowLevelLayer;
            int frameBufferIndexCount;
            nn::vi::fbshare::SharedLayerTextureIndexList frameBufferIndexList;

            DetachOperation::Progress detachProgress;
            FenceWait frameBufferAcquireFenceWaitForDetaching[FrameBufferCountMax];

            // Entrance
            //   投入口。
            //   既にディスプレイに表示中のバッファが先に Acquire されるとデッドロックするため
            //   DisplayReleaseFence がシグナルするのを待ってから次に引き渡す。
            //   次：Dequeued
            FrameBufferIndexContainer entranceIndexContainer;

            // Display
            //   LowLevelLayer に Queue して Dequeue していないもの。
            //   Dequeue したら次に引き渡す。
            //   次：Dequeued
            FrameBufferIndexQueue     displayIndexQueue;

            // Acquirable
            //   UserClient から AcquireTexture で取得できるもの。
            //   空でない場合、 AcquirableSystemEvent がシグナルする。
            //   UserClient からの Acquire() の呼出されたら次へ
            //   次：Client
            FrameBufferIndexQueue     acquirableIndexQueue;

            // Client
            //   UserClient が Acquire したもの。
            //   UserClient が Present したら次へ。
            //   次：Present
            FrameBufferIndexContainer clientIndexContainer;

            // Present
            //   UserClient が Present したもの。
            //   （非同期モード）何もせずに次へ。
            //   （同期モード）GPU フェンスを待って次へ。
            //   次：Queueing
            FrameBufferIndexQueue     presentedIndexQueue;

            // Queueing
            //   ClientLayer としては LowLevelLayer に Queue する準備ができたもの。
            //   LowLevelLayer 側の条件が揃ったら LowLevelLayer に Queue される。
            //   （つまり、前回の DisplayReleaseFence がシグナルする）
            //   次：Display
            FrameBufferIndexQueue     queueingIndexQueue;
        } m_AttachmentStatus;

        struct LayerVariable
        {
            nn::vi::LayerStackFlagType layerStackFlags;
        } m_LayerVariable;

    };

}}}}
