﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SharedClientLayer.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/vi/vi_Result.h>
#include <nn/vi/vi_ResultPrivate.h>
#include "../../visrv_Macro.h"
#include "../../visrv_Log.h"
#include "../../visrv_ResourceIdManagement.h"
#include "../../visrv_ServerManager.h"
#include "../visrv_SharedBuffer.h"
#include "../../client/visrv_ClientObject.h"
#include "visrv_SharedFrameBuffer.h"
#include "visrv_SharedLowLevelLayer.h"

#define NN_VISRV_LOG_FBSHARE_DEV(...) //NN_VISRV_LOG_FBSHARE("[dev]" __VA_ARGS__)
#define NN_VISRV_LOG_FBSHARE_DETACH(...) //NN_VISRV_LOG_FBSHARE("[detach]" __VA_ARGS__)

namespace nn{ namespace visrv{ namespace fbshare{ namespace detail{

    NN_STATIC_ASSERT(SharedClientLayer::FrameBufferCountMax == nn::vi::fbshare::SharedLayerTextureIndexList::IndexCountMax);

    //---------------------------------------------------------------------------

    SharedClientLayer::MultiWaitAdaptor::MultiWaitAdaptor() NN_NOEXCEPT
        : tagBase(-1)
    {
    }

    uintptr_t SharedClientLayer::MultiWaitAdaptor::GetLocalTag(uintptr_t globalTag) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(tagBase, -1);
        return globalTag - this->tagBase;
    }

    void SharedClientLayer::MultiWaitAdaptor::LinkMultiWait(nn::os::MultiWaitHolderType* pHolder, uintptr_t localTag) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(tagBase, -1);
        NN_SDK_REQUIRES_RANGE(localTag, 0, static_cast<uintptr_t>(SharedClientLayer::MultiWaitTagCount));
        pHolder->userData = localTag + tagBase;
        NN_SDK_ASSERT_MINMAX(pHolder->userData, MultiWaitIndex_SharedClientLayerMin, MultiWaitIndex_SharedClientLayerMax);
        g_ServerManager.GetDisplayServerManager()->AddUserWaitHolder(pHolder);
    }

    void SharedClientLayer::MultiWaitAdaptor::UnlinkMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(tagBase, -1);
        nn::os::UnlinkMultiWaitHolder(pHolder);
    }

    //-----------------------------------------------------------

    const int SharedClientLayer::MultiWaitTagCount;

    void SharedClientLayer::FenceWait::Activate(
        int frameBufferIndex,
        const android::sp<android::Fence>& fence,
        MultiWaitAdaptor& multiWaitAdaptor,
        uintptr_t localTag
    ) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!this->isActive);
        this->entry.Reset(fence);
        native::g_SyncpointWaiter.Enqueue(&this->entry);
        this->entry.InitializeWaitHolder(&this->holder);
        multiWaitAdaptor.LinkMultiWait(&this->holder, localTag);
        this->isActive = true;
    }

    void SharedClientLayer::FenceWait::Deactivate(bool isAlreadyUnlinked) NN_NOEXCEPT
    {
        if(!isAlreadyUnlinked)
        {
            nn::os::UnlinkMultiWaitHolder(&this->holder);
        }
        this->entry.FinalizeWaitHolder(&this->holder);
        this->entry.Reset();
        this->isActive = false;
    }

    //----------------------------------------------------------

    SharedClientLayer::SharedClientLayer() NN_NOEXCEPT
        : m_State(SharedClientLayerState_Invalid)
        , m_AttachState(SharedClientLayerAttachState_Detached)
        , m_Handle(nn::vi::fbshare::SharedLayerHandle::GetInvalidValue())
        , m_CreationStatus()
        , m_BindingStatus()
        , m_ConnectionStatus()
        , m_AttachmentStatus()
    {
        NN_UNUSED(m_ConnectionStatus);
    }

    bool SharedClientLayer::IsInitialized() const NN_NOEXCEPT
    {
        return m_State != SharedClientLayerState_Invalid;
    }

    bool SharedClientLayer::IsBound() const NN_NOEXCEPT
    {
        return
            m_State == SharedClientLayerState_Bound ||
            m_State == SharedClientLayerState_Connected;
    }

    bool SharedClientLayer::IsConnected() const NN_NOEXCEPT
    {
        return m_State == SharedClientLayerState_Connected;
    }

    bool SharedClientLayer::IsAttached() const NN_NOEXCEPT
    {
        return
            m_AttachState == SharedClientLayerAttachState_Attached ||
            m_AttachState == SharedClientLayerAttachState_Detaching;
    }

    bool SharedClientLayer::IsDetaching() const NN_NOEXCEPT
    {
        return m_AttachState == SharedClientLayerAttachState_Detaching;
    }

    SharedClientLayerState SharedClientLayer::GetState() const NN_NOEXCEPT
    {
        return m_State;
    }

    SharedClientLayerAttachState SharedClientLayer::GetAttachState() const NN_NOEXCEPT
    {
        return m_AttachState;
    }

    nn::vi::fbshare::SharedLayerHandle SharedClientLayer::GetHandle() const NN_NOEXCEPT
    {
        return m_Handle;
    }

    client::ClientObject* SharedClientLayer::GetOwnerClient() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_CreationStatus.pOwnerClient;
    }

    client::ClientObject* SharedClientLayer::GetUserClient() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBound());
        return m_BindingStatus.pUserClient;
    }

    SharedLowLevelLayer* SharedClientLayer::GetAttachedLowLevelLayer() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        return m_AttachmentStatus.pLowLevelLayer;
    }

    nn::sf::NativeHandle SharedClientLayer::GetAcquirableSystemEventHandle() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        auto h = nn::os::GetReadableHandleOfSystemEvent(&m_CreationStatus.acquirableSystemEvent);
        return nn::sf::NativeHandle(h, false);
    }

    nn::sf::NativeHandle SharedClientLayer::GetDetachReadySystemEventHandle() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        auto h = nn::os::GetReadableHandleOfSystemEvent(&m_CreationStatus.detachReadySystemEvent);
        return nn::sf::NativeHandle(h, false);
    }

    nn::vi::LayerStackFlagType SharedClientLayer::GetLayerStackFlags() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_LayerVariable.layerStackFlags;
    }

    //-------------------------------------------------------------------------------

    void SharedClientLayer::SetLayerStackFlags(nn::vi::LayerStackFlagType flags) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_LayerVariable.layerStackFlags = flags;
    }

    //-------------------------------------------------------------------------------

    void SharedClientLayer::SetupMultiWaitAdaptor(uintptr_t tagBase) NN_NOEXCEPT
    {
        m_MultiWaitAdaptor.tagBase = tagBase;
    }

    void SharedClientLayer::DispatchMultiWait(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pHolder);
        NN_SDK_REQUIRES_MINMAX(pHolder->userData, MultiWaitIndex_SharedClientLayerMin, MultiWaitIndex_SharedClientLayerMax);
        uintptr_t localTag = m_MultiWaitAdaptor.GetLocalTag(pHolder->userData);
        NN_SDK_REQUIRES_RANGE(localTag, 0, MultiWaitTagCount);

        if(localTag >= LocalMultiWaitTag_DetachAcquireFenceWaitMin && localTag <= LocalMultiWaitTag_DetachAcquireFenceWaitMax)
        {
            int iWin = static_cast<int>(localTag - LocalMultiWaitTag_DetachAcquireFenceWaitMin);
            DetachOperation::ProcessAcquireFenceExpired(this, iWin);
        }
        else
        {
            NN_VISRV_LOG_FBSHARE_ERR("[c]unknown local tag %lld\n", localTag);
        }
    }

    nn::Result SharedClientLayer::Initialize(
        nn::vi::fbshare::SharedLayerHandle* pOutHandle,
        client::ClientObject* pOwnerClient,
        nn::applet::AppletResourceUserId userAruid
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_State, SharedClientLayerState_Invalid);
        NN_SDK_REQUIRES_EQUAL(m_AttachState, SharedClientLayerAttachState_Detached);
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
        NN_SDK_REQUIRES_NOT_NULL(pOwnerClient);

        NN_VISRV_PROCESS_START();

        NN_RESULT_DO(nn::os::CreateSystemEvent(&m_CreationStatus.acquirableSystemEvent, nn::os::EventClearMode_ManualClear, true));
        NN_VISRV_PROCESS_ROLLBACK(nn::os::DestroySystemEvent(&m_CreationStatus.acquirableSystemEvent));
        nn::os::ClearSystemEvent(&m_CreationStatus.acquirableSystemEvent);

        NN_RESULT_DO(nn::os::CreateSystemEvent(&m_CreationStatus.detachReadySystemEvent, nn::os::EventClearMode_ManualClear, true));
        NN_VISRV_PROCESS_ROLLBACK(nn::os::DestroySystemEvent(&m_CreationStatus.detachReadySystemEvent));
        nn::os::SignalSystemEvent(&m_CreationStatus.detachReadySystemEvent); // initially signaled

        NN_VISRV_PROCESS_SUCCESS();

        m_CreationStatus.pOwnerClient = pOwnerClient;
        m_CreationStatus.userAruid    = userAruid;
        m_LayerVariable.layerStackFlags = 0;
        m_Handle = { AcquireResourceId() };
        m_State  = SharedClientLayerState_Initialized;
        *pOutHandle = m_Handle;
        NN_RESULT_SUCCESS;
    }

    void SharedClientLayer::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_State, SharedClientLayerState_Initialized);
        NN_SDK_REQUIRES_EQUAL(m_AttachState, SharedClientLayerAttachState_Detached);

        NN_SDK_ASSERT(nn::os::TryWaitSystemEvent(&m_CreationStatus.detachReadySystemEvent));
        nn::os::SignalSystemEvent(&m_CreationStatus.detachReadySystemEvent);
        nn::os::DestroySystemEvent(&m_CreationStatus.detachReadySystemEvent);

        NN_SDK_ASSERT(!nn::os::TryWaitSystemEvent(&m_CreationStatus.acquirableSystemEvent));
        nn::os::SignalSystemEvent(&m_CreationStatus.acquirableSystemEvent); // prevent infinite wait of client
        nn::os::DestroySystemEvent(&m_CreationStatus.acquirableSystemEvent);

        m_CreationStatus.pOwnerClient = nullptr;
        m_CreationStatus.userAruid    = applet::AppletResourceUserId::GetInvalidId();
        m_LayerVariable.layerStackFlags = 0;

        ReleaseResourceId(m_Handle._value);
        m_Handle = nn::vi::fbshare::SharedLayerHandle::GetInvalidValue();
        m_State  = SharedClientLayerState_Invalid;
    }

    nn::Result SharedClientLayer::Bind(client::ClientObject* pObject, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_State, SharedClientLayerState_Initialized);
        NN_SDK_REQUIRES_NOT_NULL(pObject);

        NN_RESULT_THROW_UNLESS(
            pObject->GetConstants().GetPermission() == client::ClientPermission_Manager ||
            aruid != nn::applet::AppletResourceUserId::GetInvalidId(),
            nn::vi::ResultDenied()
        );
        NN_RESULT_THROW_UNLESS(aruid == m_CreationStatus.userAruid, nn::vi::ResultDenied());

        m_BindingStatus.pUserClient = pObject;
        m_State = SharedClientLayerState_Bound;
        NN_RESULT_SUCCESS;
    }

    void SharedClientLayer::Unbind() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_State, SharedClientLayerState_Bound);

        m_BindingStatus.pUserClient = nullptr;

        m_State = SharedClientLayerState_Initialized;
    }

    nn::Result SharedClientLayer::Connect() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_State, SharedClientLayerState_Bound);
        m_State = SharedClientLayerState_Connected;

        if(IsAttached())
        {
            UpdateAcquirableEventImpl();
        }
        NN_RESULT_SUCCESS;
    }

    void SharedClientLayer::Disconnect() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_State, SharedClientLayerState_Connected);
        m_State = SharedClientLayerState_Bound;

        if(IsAttached())
        {
            // Acquire されている FrameBuffer を Entrance に戻す
            m_AttachmentStatus.clientIndexContainer.RemoveIf([&](int index)->bool{
                PushBufferToEntranceImpl(index);
                return true;
            });
            NN_SDK_ASSERT_EQUAL(m_AttachmentStatus.clientIndexContainer.GetCount(), 0);
        }

        if(IsAttached())
        {
            UpdateAcquirableEventImpl();
        }

        if(IsDetaching())
        {
            DetachOperation::ProcessDisconnected(this);
        }
    }

    nn::Result SharedClientLayer::Attach(
        SharedLowLevelLayer* pLowLayer,
        const nn::vi::fbshare::SharedLayerTextureIndexList& frameBufferIndexList
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(
            m_State == SharedClientLayerState_Initialized ||
            m_State == SharedClientLayerState_Bound ||
            m_State == SharedClientLayerState_Connected
        );
        NN_SDK_REQUIRES_EQUAL(m_AttachState, SharedClientLayerAttachState_Detached);
        NN_SDK_REQUIRES_NOT_NULL(pLowLayer);
        NN_SDK_REQUIRES_EQUAL(pLowLayer->GetState(), SharedLowLevelLayerState_Connected);

        int nBuf = pLowLayer->GetFrameBufferCount();

        // アタッチするレイヤをチェック
        int nWin = 0;
        for(int i = 0; i < FrameBufferCountMax; i++)
        {
            int iBuf = frameBufferIndexList[i];
            if(iBuf == nn::vi::fbshare::SharedLayerTextureIndexList::GetInvalidIndex())
            {
                continue;
            }
            NN_RESULT_THROW_UNLESS(iBuf >= 0 && iBuf < nBuf, nn::vi::ResultInvalidRange());

            auto pBuffer = pLowLayer->GetFrameBuffer(iBuf);
            NN_SDK_ASSERT_NOT_NULL(pBuffer);

            // PresentedLowLevelLayer が nullptr または pLowLayer であることの確認
            NN_RESULT_THROW_UNLESS(
                pBuffer->GetPresentedLowLevelLayerId() == 0 || pBuffer->GetPresentedLowLevelLayerId() == pLowLayer->GetLayerResourceId(),
                nn::vi::ResultDenied()
            );

            // SharedClientLayer にアタッチされていないことの確認
            NN_RESULT_THROW_UNLESS(pBuffer->GetAttachedClientLayerHandle() == nn::vi::fbshare::SharedLayerHandle::GetInvalidValue(), nn::vi::ResultDenied());
            nWin++;
        }
        NN_RESULT_THROW_UNLESS(nWin >= FrameBufferCountMin, nn::vi::ResultInvalidRange());

        // アタッチ処理
        // 前詰めに強制
        int iWin = 0;
        m_AttachmentStatus.frameBufferIndexList.Reset();
        for(int i = 0; i < FrameBufferCountMax; i++)
        {
            int iBuf = frameBufferIndexList[i];
            if(iBuf == nn::vi::fbshare::SharedLayerTextureIndexList::GetInvalidIndex())
            {
                continue;
            }
            NN_SDK_ASSERT_RANGE(iBuf, 0, nBuf);
            auto pBuffer = pLowLayer->GetFrameBuffer(iBuf);
            NN_SDK_ASSERT_NOT_NULL(pBuffer);

            // この SharedClientLayer にアタッチ
            pBuffer->SetAttachedClientLayerHandle(m_Handle);
            m_AttachmentStatus.frameBufferIndexList[iWin] = iBuf;
            iWin++;
        }
        NN_SDK_ASSERT_EQUAL(iWin, nWin);
        NN_SDK_ASSERT_EQUAL(nWin, m_AttachmentStatus.frameBufferIndexList.GetCount());

        m_AttachmentStatus.frameBufferIndexCount = nWin;
        m_AttachmentStatus.pLowLevelLayer = pLowLayer;
        m_AttachState = SharedClientLayerAttachState_Attached;
        m_AttachmentStatus.detachProgress = DetachOperation::Progress_Inactive;

        m_AttachmentStatus.entranceIndexContainer.Initialize();
        m_AttachmentStatus.displayIndexQueue.Initialize();
        m_AttachmentStatus.acquirableIndexQueue.Initialize();
        m_AttachmentStatus.clientIndexContainer.Initialize();
        m_AttachmentStatus.presentedIndexQueue.Initialize();
        m_AttachmentStatus.queueingIndexQueue.Initialize();

        // 全バッファをエントランスに投入
        for(int iWin = 0; iWin < nWin; iWin++)
        {
            int iBuf = m_AttachmentStatus.frameBufferIndexList[iWin];
            PushBufferToEntranceImpl(iBuf);
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result SharedClientLayer::StartDetach() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(m_AttachState, SharedClientLayerAttachState_Attached);

        NN_VISRV_LOG_FBSHARE_DEV("[c]starting detaching #%lld\n", GetHandle());
        DetachOperation::StartDetach(this);

        NN_RESULT_SUCCESS;
    }

    nn::Result SharedClientLayer::FinishDetach() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsDetaching());
        NN_VISRV_LOG_FBSHARE_DEV("[c]finishing detaching #%lld\n", GetHandle());
        auto result = DetachOperation::FinishDetach(this);
        if(result.IsSuccess())
        {
            NN_VISRV_LOG_FBSHARE_DEV("[c]  -> success\n");
        }
        else
        {
            NN_VISRV_LOG_FBSHARE_DEV("[c]  -> failed (%d-%d)\n", result.GetModule(), result.GetDescription());
        }
        return result;
    }

    void SharedClientLayer::ForceDetach() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        NN_VISRV_LOG_FBSHARE_DEV("[c]force detaching #%lld\n", GetHandle());
        DetachOperation::ForceDetach(this);
    }


    //-----------------------------------------------

    nn::Result SharedClientLayer::Acquire(
        int* pOutFrameBufferIndex,
        native::NativeRmSync* pOutSync,
        nn::vi::fbshare::SharedLayerTextureIndexList* pOutBufferIndexList
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFrameBufferIndex);
        NN_SDK_REQUIRES_NOT_NULL(pOutSync);
        NN_SDK_REQUIRES_NOT_NULL(pOutBufferIndexList);
        NN_SDK_REQUIRES(IsConnected());

        if(!IsAttached())
        {
            // Detach 時にクリアされているはず
            NN_SDK_ASSERT(!nn::os::TryWaitSystemEvent(&m_CreationStatus.acquirableSystemEvent));
            NN_RESULT_THROW(nn::vi::ResultPrivateSharedLayerNotAcquirable());
        }
        if(IsDetaching())
        {
            // Detach 開始時にクリアされているはず
            NN_SDK_ASSERT(!nn::os::TryWaitSystemEvent(&m_CreationStatus.acquirableSystemEvent));
            NN_RESULT_THROW(nn::vi::ResultPrivateSharedLayerNotAcquirable());
        }

        NN_RESULT_THROW_UNLESS(!m_AttachmentStatus.acquirableIndexQueue.IsEmpty(), nn::vi::ResultPrivateSharedLayerNotAcquirable());

        int index = -1;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AttachmentStatus.acquirableIndexQueue.TryDequeue(&index));

        auto pBuffer = m_AttachmentStatus.pLowLevelLayer->GetFrameBuffer(index);
        NN_SDK_ASSERT_NOT_NULL(pBuffer);

        native::NativeRmSync sync;
        sync.ReadFrom(pBuffer->GetReleaseFence());

        UpdateAcquirableEventImpl();
        PushBufferToClientImpl(index);
        *pOutFrameBufferIndex = index;
        *pOutSync = sync;
        *pOutBufferIndexList = m_AttachmentStatus.frameBufferIndexList;
        NN_RESULT_SUCCESS;
    }

    nn::Result SharedClientLayer::Present(
        int frameBufferIndex,
        const native::NativeRmSync& sync,
        const nn::vi::CropRegion& crop,
        nn::vi::ImageTransformType transform,
        int presentInterval
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsConnected());

        // Attach 中でないか、 Acquire していないバッファを Present されたら失敗。
        // Acquire->Present の間に FroceDetach->Attach が走った場合 IsAttached == true だが Acquire されていないバッファが Present されることがある。
        // 今の実装ではクライアントがインデックスを間違えたのか再アタッチされたのか区別がつかない。
        // このエラーは SharedNativeWindow の shim で無視される。
        NN_RESULT_THROW_UNLESS(IsAttached(), nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(m_AttachmentStatus.clientIndexContainer.Contains(frameBufferIndex), nn::vi::ResultDenied());

        // AcquireFence と画像のパラメータを設定
        android::sp<android::Fence> fence;
        NN_ABORT_UNLESS(sync.WriteTo(&fence));
        SharedFrameBufferContentParameter param = SharedFrameBufferContentParameter::GetDefaultValue();
        param.layerStacks = m_LayerVariable.layerStackFlags;
        param.cropRegion = crop;
        param.transform = transform;
        param.presentInterval = presentInterval;

        auto pBuffer = m_AttachmentStatus.pLowLevelLayer->GetFrameBuffer(frameBufferIndex);
        NN_SDK_ASSERT_NOT_NULL(pBuffer);
        NN_SDK_ASSERT(!pBuffer->GetAcquireFence()->isValid());
        pBuffer->SetAcquireFence(fence);
        pBuffer->SetContentParameter(param);

        // Presented に移動
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AttachmentStatus.clientIndexContainer.Unregister(frameBufferIndex));
        PushBufferToPresentedImpl(frameBufferIndex);

        NN_RESULT_SUCCESS;
    }

    nn::Result SharedClientLayer::Cancel(
        int frameBufferIndex
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsConnected());

        // Attach 中でないか、 Acquire していないバッファを Present されたら失敗。
        // Acquire->Present の間に FroceDetach->Attach が走った場合 IsAttached == true だが Acquire されていないバッファが Present されることがある。
        // 今の実装ではクライアントがインデックスを間違えたのか再アタッチされたのか区別がつかない。
        // このエラーは SharedNativeWindow の shim で無視される。
        NN_RESULT_THROW_UNLESS(IsAttached(), nn::vi::ResultDenied());
        NN_RESULT_THROW_UNLESS(m_AttachmentStatus.clientIndexContainer.Contains(frameBufferIndex), nn::vi::ResultDenied());

        // Acquirable に戻す
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AttachmentStatus.clientIndexContainer.Unregister(frameBufferIndex));
        CancelBufferToAcquirableImpl(frameBufferIndex);

        NN_RESULT_SUCCESS;
    }

    nn::Result SharedClientLayer::FillBufferColor(int frameBufferIndex, int r, int g, int b, int a) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsConnected());
        NN_SDK_REQUIRES(IsAttached());

        NN_RESULT_THROW_UNLESS(m_AttachmentStatus.clientIndexContainer.Contains(frameBufferIndex), nn::vi::ResultDenied());
        NN_SDK_ASSERT_EQUAL(m_AttachmentStatus.pLowLevelLayer->GetFrameBuffer(frameBufferIndex)->GetAttachedClientLayerHandle(), GetHandle());

        nn::util::Color4u8 color = {};
        color.SetR(r < 0 ? 0 : (r > 255 ? 255 : r));
        color.SetG(g < 0 ? 0 : (g > 255 ? 255 : g));
        color.SetB(b < 0 ? 0 : (b > 255 ? 255 : b));
        color.SetA(a < 0 ? 0 : (a > 255 ? 255 : a));

        nn::vi::fbshare::SharedTextureOption option = {};
        option.colorOption = nn::vi::fbshare::SharedTextureColorOption_PreMultipledAlpha;
        option.alphaOption = nn::vi::fbshare::SharedTextureAlphaOption_None;
        option.stacks      = m_LayerVariable.layerStackFlags;
        option.transform   = nn::vi::ImageTransform_None;
        m_AttachmentStatus.pLowLevelLayer->GetConnectedSharedBuffer()->FillColor(frameBufferIndex, color, option);
        NN_RESULT_SUCCESS;
    }

    void SharedClientLayer::NotifyBufferIdle() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        ProcessEntranceImpl();
        ProcessQueueingImpl();
    }

    void SharedClientLayer::NotifyBufferDequeued(int frameBufferIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        ProcessDisplayDequeueingImpl(frameBufferIndex);
    }

    //-----------------------------------------------

    void SharedClientLayer::PushBufferToEntranceImpl(int frameBufferIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        NN_VISRV_LOG_FBSHARE_DEV("[c]->Entrance(%d)\n", frameBufferIndex);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AttachmentStatus.entranceIndexContainer.Register(frameBufferIndex));
        ProcessEntranceImpl();
    }

    void SharedClientLayer::ProcessEntranceImpl() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        m_AttachmentStatus.entranceIndexContainer.RemoveIf([&](int v)->bool{
            if(m_AttachmentStatus.pLowLevelLayer->IsFrameBufferIdle(v))
            {
                PushBufferToAcquirableImpl(v);
                return true;
            }
            else
            {
                return false;
            }
        });
    }

    //-----------------------------------------------

    void SharedClientLayer::PushBufferToDisplayImpl(int frameBufferIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        NN_VISRV_LOG_FBSHARE_DEV("[c]->Display(%d)\n", frameBufferIndex);
        NN_SDK_REQUIRES_RANGE(frameBufferIndex, 0, m_AttachmentStatus.pLowLevelLayer->GetFrameBufferCount());

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AttachmentStatus.displayIndexQueue.TryEnqueue(frameBufferIndex));
    }

    void SharedClientLayer::ProcessDisplayDequeueingImpl(int frameBufferIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        const int* pHead = m_AttachmentStatus.displayIndexQueue.GetHead();
        if(pHead == nullptr || *pHead != frameBufferIndex)
        {
            NN_VISRV_LOG_FBSHARE_DEV("[c]Display->dropping unknown buffer(%d)\n", frameBufferIndex);
            return;
        }

        int index = -1;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AttachmentStatus.displayIndexQueue.TryDequeue(&index));
        NN_SDK_ASSERT_EQUAL(index, frameBufferIndex);

        // Display から Dequeue されたバッファの AcquireFence は削除する
        auto pBuffer = m_AttachmentStatus.pLowLevelLayer->GetFrameBuffer(index);
        NN_SDK_ASSERT_NOT_NULL(pBuffer);
        pBuffer->SetAcquireFence(android::Fence::NO_FENCE);

        // NOTE:
        //   画像データは維持されているので ContentParameter は変更しない。

        PushBufferToAcquirableImpl(frameBufferIndex);
    }

    //-----------------------------------------------

    void SharedClientLayer::PushBufferToAcquirableImpl(int frameBufferIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        NN_VISRV_LOG_FBSHARE_DEV("[c]->Acquirable(%d)\n", frameBufferIndex);
        NN_SDK_REQUIRES_RANGE(frameBufferIndex, 0, m_AttachmentStatus.pLowLevelLayer->GetFrameBufferCount());

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AttachmentStatus.acquirableIndexQueue.TryEnqueue(frameBufferIndex));
        NN_SDK_ASSERT(!m_AttachmentStatus.acquirableIndexQueue.IsEmpty());
        UpdateAcquirableEventImpl();
    }

    void SharedClientLayer::CancelBufferToAcquirableImpl(int frameBufferIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        NN_VISRV_LOG_FBSHARE_DEV("[c]->Acquirable(canceled)(%d)\n", frameBufferIndex);
        NN_SDK_REQUIRES_RANGE(frameBufferIndex, 0, m_AttachmentStatus.pLowLevelLayer->GetFrameBufferCount());

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AttachmentStatus.acquirableIndexQueue.TryCancel(frameBufferIndex));
        NN_SDK_ASSERT(!m_AttachmentStatus.acquirableIndexQueue.IsEmpty());
        UpdateAcquirableEventImpl();
        if(IsDetaching())
        {
            DetachOperation::ProcessBufferCanceled(this, frameBufferIndex);
        }
    }

    void SharedClientLayer::UpdateAcquirableEventImpl() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        if(IsConnected() && !IsDetaching() && !m_AttachmentStatus.acquirableIndexQueue.IsEmpty())
        {
            NN_VISRV_LOG_FBSHARE_DEV("[c]signaled acquirable event\n");
            nn::os::SignalSystemEvent(&m_CreationStatus.acquirableSystemEvent);
        }
        else
        {
            NN_VISRV_LOG_FBSHARE_DEV("[c]cleared acquirable event\n");
            nn::os::ClearSystemEvent(&m_CreationStatus.acquirableSystemEvent);
        }
    }

    //------------------------------------------------

    void SharedClientLayer::PushBufferToClientImpl(int frameBufferIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        NN_VISRV_LOG_FBSHARE_DEV("[c]->Client(%d)\n", frameBufferIndex);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AttachmentStatus.clientIndexContainer.Register(frameBufferIndex));
    }

    //------------------------------------------------

    void SharedClientLayer::PushBufferToPresentedImpl(int frameBufferIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        NN_VISRV_LOG_FBSHARE_DEV("[c]->Presented(%d)\n", frameBufferIndex);

        // スキップして Queueing へ
        PushBufferToQueueingImpl(frameBufferIndex);
    }

    //------------------------------------------------

    void SharedClientLayer::PushBufferToQueueingImpl(int frameBufferIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());
        NN_VISRV_LOG_FBSHARE_DEV("[c]->Queueing(%d)\n", frameBufferIndex);

        NN_ABORT_UNLESS_RESULT_SUCCESS(m_AttachmentStatus.queueingIndexQueue.TryEnqueue(frameBufferIndex));
        ProcessQueueingImpl();
    }

    void SharedClientLayer::ProcessQueueingImpl() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsAttached());

        auto& queue = m_AttachmentStatus.queueingIndexQueue;

        while(!queue.IsEmpty())
        {
            auto pHead = queue.GetHead();
            NN_SDK_ASSERT_NOT_NULL(pHead);
            int headIndex = *pHead;

            // 準備ができていればディスプレイに送る
            if(m_AttachmentStatus.pLowLevelLayer->IsFrameBufferIdle(headIndex))
            {
                int index = -1;
                NN_ABORT_UNLESS_RESULT_SUCCESS(queue.TryDequeue(&index));
                NN_SDK_ASSERT_EQUAL(index, headIndex);
                // このレイヤにアタッチされていることを確認
                NN_SDK_ASSERT_EQUAL(m_AttachmentStatus.pLowLevelLayer->GetFrameBuffer(index)->GetAttachedClientLayerHandle(), GetHandle());
                // Attach したバッファであればどのディスプレイにも出力されていないことが保証できる
                NN_SDK_ASSERT_EQUAL(m_AttachmentStatus.pLowLevelLayer->GetFrameBuffer(index)->GetPresentedLowLevelLayerId(), 0);
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_AttachmentStatus.pLowLevelLayer->Enqueue(headIndex));
                PushBufferToDisplayImpl(headIndex);
                if(IsDetaching())
                {
                    DetachOperation::ProcessBufferQueued(this, index);
                }
            }
            else
            {
                break;
            }
        }
    }

    //------------------------------------------------
    bool SharedClientLayer::DetachOperation::CheckAllBuffersQueuedImpl(const SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSelf->IsDetaching());
        if(pSelf->m_AttachmentStatus.clientIndexContainer.GetCount() > 0)
        {
            return false;
        }
        if(!pSelf->m_AttachmentStatus.presentedIndexQueue.IsEmpty())
        {
            return false;
        }
        if(!pSelf->m_AttachmentStatus.queueingIndexQueue.IsEmpty())
        {
            return false;
        }
        return true;
    }

    bool SharedClientLayer::DetachOperation::CheckAllAcquireFencesExpiredImpl(const SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSelf->IsDetaching());
        for(int i = 0; i < pSelf->m_AttachmentStatus.frameBufferIndexCount; i++)
        {
            auto& wait = pSelf->m_AttachmentStatus.frameBufferAcquireFenceWaitForDetaching[i];
            if(wait.isActive)
            {
                return false;
            }
        }
        return true;
    }

    bool SharedClientLayer::DetachOperation::IsDetachReady(const SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSelf->IsDetaching());
        return pSelf->m_AttachmentStatus.detachProgress == Progress_DetachReady;
    }

    void SharedClientLayer::DetachOperation::StartDetach(SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(pSelf->m_AttachState, SharedClientLayerAttachState_Attached);
        NN_SDK_REQUIRES_EQUAL(pSelf->m_AttachmentStatus.detachProgress, Progress_Inactive);

        // NOTE: Detach 処理中でなければシグナル状態
        NN_SDK_ASSERT(nn::os::TryWaitSystemEvent(&pSelf->m_CreationStatus.detachReadySystemEvent));

        pSelf->m_AttachState = SharedClientLayerAttachState_Detaching;
        nn::os::ClearSystemEvent(&pSelf->m_CreationStatus.detachReadySystemEvent);
        nn::os::ClearSystemEvent(&pSelf->m_CreationStatus.acquirableSystemEvent);

        StartWaitForAllBuffersQueuedImpl(pSelf);
    }

    void SharedClientLayer::DetachOperation::StartWaitForAllBuffersQueuedImpl(SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSelf->IsDetaching());
        NN_UNUSED(pSelf);

        pSelf->m_AttachmentStatus.detachProgress = Progress_WaitingForAllBuffersQueued;
        ProcessBufferQueueStateChangedImpl(pSelf);
    }

    void SharedClientLayer::DetachOperation::ProcessBufferQueued(SharedClientLayer* pSelf, int iBuf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSelf->IsDetaching());
        NN_UNUSED(iBuf);
        NN_VISRV_LOG_FBSHARE_DETACH("queued iBuf=%d\n", iBuf);
        ProcessBufferQueueStateChangedImpl(pSelf);
    }

    void SharedClientLayer::DetachOperation::ProcessBufferCanceled(SharedClientLayer* pSelf, int iBuf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSelf->IsDetaching());
        NN_UNUSED(iBuf);
        NN_VISRV_LOG_FBSHARE_DETACH("canceled iBuf=%d\n", iBuf);
        ProcessBufferQueueStateChangedImpl(pSelf);
    }

    void SharedClientLayer::DetachOperation::ProcessDisconnected(SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSelf->IsDetaching());
        NN_VISRV_LOG_FBSHARE_DETACH("disconnected\n");
        ProcessBufferQueueStateChangedImpl(pSelf);
    }

    void SharedClientLayer::DetachOperation::ProcessBufferQueueStateChangedImpl(SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        // IsDetaching の間は常に呼ばれる可能性がある（特に Disconnect から）ので
        // detachProgress で ASSERT してはいけない。

        if(pSelf->m_AttachmentStatus.detachProgress == Progress_WaitingForAllBuffersQueued)
        {
            if(CheckAllBuffersQueuedImpl(pSelf))
            {
                StartWaitForAllAcquireFencesExpiredImpl(pSelf);
            }
        }
    }

    void SharedClientLayer::DetachOperation::StartWaitForAllAcquireFencesExpiredImpl(SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSelf->IsDetaching());
        NN_SDK_REQUIRES(CheckAllBuffersQueuedImpl(pSelf));

        // NOTE:
        //   バッファに紐づいている AcquireFence は LowLevelLayer からの Dequeue 時に削除されるため
        //   バッファの AcquireFence はデタッチ用の waiter と齟齬が発生する。
        //   デタッチ処理は AcquireFence の待ちを開始する際に全バッファのフェンスを見て waiter を設定し、
        //   以降は waiter がシグナルしたかをチェックしてデタッチ準備ができたか判定する。

        // AcquireFence 待ちを開始
        for(int iWin = 0; iWin < pSelf->m_AttachmentStatus.frameBufferIndexCount; iWin++)
        {
            int iBuf = pSelf->m_AttachmentStatus.frameBufferIndexList[iWin];
            auto& wait = pSelf->m_AttachmentStatus.frameBufferAcquireFenceWaitForDetaching[iWin];

            if(wait.isActive)
            {
                // 本来ここには来てはいけない。
                NN_VISRV_LOG_FBSHARE_ERR("already waiting for acquire fence (iWin=%d)\n", iWin);
                NN_SDK_ASSERT(!wait.isActive);
                wait.Deactivate(false); // Release では待ち解除して続行
            }

            auto pBuf = pSelf->m_AttachmentStatus.pLowLevelLayer->GetFrameBuffer(iBuf);
            NN_SDK_ASSERT_NOT_NULL(pBuf);

            auto fence = pBuf->GetAcquireFence();
            if(fence->isValid())
            {
                NN_VISRV_LOG_FBSHARE_DETACH("waiting for acquire fence of FrameBuffer iWin=%d,iBuf=%d\n", iWin, iBuf);
                wait.Activate(iBuf, fence, pSelf->m_MultiWaitAdaptor, LocalMultiWaitTag_DetachAcquireFenceWaitMin + iWin);
            }
        }

        pSelf->m_AttachmentStatus.detachProgress = Progress_WaitingForAllAcquireFencesExpired;
        ProcessAcquireFenceChangedImpl(pSelf);
    }

    void SharedClientLayer::DetachOperation::ProcessAcquireFenceExpired(SharedClientLayer* pSelf, int iWin) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSelf->IsDetaching());
        NN_SDK_REQUIRES(CheckAllBuffersQueuedImpl(pSelf));
        NN_SDK_REQUIRES_EQUAL(pSelf->m_AttachmentStatus.detachProgress, Progress_WaitingForAllAcquireFencesExpired);

        // expire したバッファの AcquireFence を削除
        {
            int iBuf = pSelf->m_AttachmentStatus.frameBufferIndexList[iWin];
            NN_VISRV_LOG_FBSHARE_DETACH("expired acquire fence of FrameBuffer iWin=%d,iBuf=%d\n", iWin, iBuf);

            auto& wait = pSelf->m_AttachmentStatus.frameBufferAcquireFenceWaitForDetaching[iWin];
            NN_SDK_ASSERT(wait.isActive);
            wait.Deactivate(true);

            auto pBuf = pSelf->m_AttachmentStatus.pLowLevelLayer->GetFrameBuffer(iBuf);
            // NOTE: Dequeue 時に AcquireFence は削除されるので現在値をチェックしない
            pBuf->SetAcquireFence(android::Fence::NO_FENCE);
        }

        ProcessAcquireFenceChangedImpl(pSelf);
    }

    void SharedClientLayer::DetachOperation::ProcessAcquireFenceChangedImpl(SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        // こっちは detachProgress で ASSERT してもいいかも。

        if(pSelf->m_AttachmentStatus.detachProgress == Progress_WaitingForAllAcquireFencesExpired)
        {
            if(CheckAllAcquireFencesExpiredImpl(pSelf))
            {
                NotifySafeDetachReadyImpl(pSelf);
            }
        }
    }

    void SharedClientLayer::DetachOperation::NotifySafeDetachReadyImpl(SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSelf->IsDetaching());
        NN_SDK_REQUIRES(CheckAllBuffersQueuedImpl(pSelf));
        NN_SDK_REQUIRES(CheckAllAcquireFencesExpiredImpl(pSelf));

        pSelf->m_AttachmentStatus.detachProgress = Progress_DetachReady;
        nn::os::SignalSystemEvent(&pSelf->m_CreationStatus.detachReadySystemEvent);
    }

    nn::Result SharedClientLayer::DetachOperation::FinishDetach(SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(pSelf->IsDetaching());

        NN_RESULT_THROW_UNLESS(IsDetachReady(pSelf), nn::vi::ResultNotReady());
        NN_SDK_ASSERT(nn::os::TryWaitSystemEvent(&pSelf->m_CreationStatus.detachReadySystemEvent));

        // ASSERT: 全 wait が解除されているはず
        for(int iWin = 0; iWin < pSelf->m_AttachmentStatus.frameBufferIndexCount; iWin++)
        {
            auto& wait = pSelf->m_AttachmentStatus.frameBufferAcquireFenceWaitForDetaching[iWin];
            NN_SDK_ASSERT(!wait.isActive);
            NN_UNUSED(wait);
        }

        // 既に安全なので強制デタッチで OK
        ForceDetach(pSelf);
        // NOTE: Detach するとシグナル状態になる
        NN_SDK_ASSERT(nn::os::TryWaitSystemEvent(&pSelf->m_CreationStatus.detachReadySystemEvent));
        NN_RESULT_SUCCESS;
    }

    void SharedClientLayer::DetachOperation::ForceDetach(SharedClientLayer* pSelf) NN_NOEXCEPT
    {
        nn::os::SignalSystemEvent(&pSelf->m_CreationStatus.detachReadySystemEvent);
        nn::os::ClearSystemEvent(&pSelf->m_CreationStatus.acquirableSystemEvent);

        // AcquireFence の待ちを強制解除
        for(int iWin = 0; iWin < pSelf->m_AttachmentStatus.frameBufferIndexCount; iWin++)
        {
            auto& wait = pSelf->m_AttachmentStatus.frameBufferAcquireFenceWaitForDetaching[iWin];
            if(wait.isActive)
            {
                NN_VISRV_LOG_FBSHARE_DETACH("deactivate waiting for acquire fence iWin=%d\n", iWin);
                wait.Deactivate(false);
            }
        }

        // この SharedClientLayer からフレームバッファをデタッチ
        for(int iWin = 0; iWin < pSelf->m_AttachmentStatus.frameBufferIndexCount; iWin++)
        {
            auto iBuf = pSelf->m_AttachmentStatus.frameBufferIndexList[iWin];
            auto pBuf = pSelf->m_AttachmentStatus.pLowLevelLayer->GetFrameBuffer(iBuf);
            NN_SDK_ASSERT_NOT_NULL(pBuf);
            NN_SDK_ASSERT_EQUAL(pBuf->GetAttachedClientLayerHandle(), pSelf->m_Handle);
            pBuf->SetAttachedClientLayerHandle(nn::vi::fbshare::SharedLayerHandle::GetInvalidValue());

            // FORCED: GPU フェンスを削除
            if(pBuf->GetAcquireFence()->isValid())
            {
                NN_VISRV_LOG_FBSHARE_DETACH("removing acquire fence on buffer iWin=%d,iBuf=%d\n", iWin, iBuf);
            }
            pBuf->SetAcquireFence(android::Fence::NO_FENCE);

            // NOTE:
            //   画像データは維持されているので ContentParameter は変更しない
        }

        pSelf->m_AttachmentStatus.frameBufferIndexCount = 0;
        pSelf->m_AttachmentStatus.frameBufferIndexList.Reset();
        pSelf->m_AttachmentStatus.entranceIndexContainer.Finalize();
        pSelf->m_AttachmentStatus.displayIndexQueue.Finalize();
        pSelf->m_AttachmentStatus.acquirableIndexQueue.Finalize();
        pSelf->m_AttachmentStatus.clientIndexContainer.Finalize();
        pSelf->m_AttachmentStatus.presentedIndexQueue.Finalize();
        pSelf->m_AttachmentStatus.queueingIndexQueue.Finalize();

        pSelf->m_AttachmentStatus.pLowLevelLayer = nullptr;
        pSelf->m_AttachmentStatus.detachProgress = Progress_Inactive;
        pSelf->m_AttachState = SharedClientLayerAttachState_Detached;
    }


}}}}
