﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SystemEventSplitter.h"

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <nn/os/os_SdkSystemEventApi.h>
#include <nn/vi/vi_Result.h>
#include "../visrv_Log.h"
#include "../master/visrv_Display.h"

namespace nn{ namespace visrv{ namespace client{

    HotplugSplitter g_HotplugEventSplitter;
    VsyncSplitter g_VsyncEventSplitter;
    ErrorReportSplitter g_ErrorReportEventSplitter;
    ModeChangedSplitter g_ExternalModeChangedSplitter;

    SystemEventSplitter::SystemEventSplitter(
    ) NN_NOEXCEPT
    {
    }

    SystemEventSplitter::~SystemEventSplitter() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
    }

    nn::Result SystemEventSplitter::Initialize(
        int destinationCountMax,
        nn::os::SystemEventType* pDestinationSystemEventList,
        nn::vi::DisplayId* pDestinationDisplayIdList,
        nn::os::NativeHandle sourceEventHandle,
        bool isManaged
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pDestinationSystemEventList);
        NN_SDK_REQUIRES_NOT_NULL(pDestinationDisplayIdList);
        std::memset(this, 0, sizeof(*this));
        m_pDestinationSystemEventList = pDestinationSystemEventList;
        m_pDestinationDisplayIdList   = pDestinationDisplayIdList;
        m_DestinationCountMax         = destinationCountMax;

        nn::os::AttachReadableHandleToSystemEvent(&m_SourceSystemEvent, sourceEventHandle, isManaged, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(&m_SourceWaitHolder, &m_SourceSystemEvent);

        for(int i = 0; i < m_DestinationCountMax; i++)
        {
            m_pDestinationDisplayIdList[i] = 0;
        }

        m_IsInitialized = true;
        NN_RESULT_SUCCESS;
    }

    void SystemEventSplitter::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_EQUAL(GetSplittedEventCount(), 0);

        nn::os::FinalizeMultiWaitHolder(&m_SourceWaitHolder);
        nn::os::DestroySystemEvent(&m_SourceSystemEvent);
    }

    bool SystemEventSplitter::IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    nn::Result SystemEventSplitter::CreateSplittedEvent(nn::os::NativeHandle* pOutHandle, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutHandle);
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_EQUAL(displayId, 0);

        // find empty slot
        int index = -1;
        for(int i = 0; i < m_DestinationCountMax; i++)
        {
            if(m_pDestinationDisplayIdList[i] == 0)
            {
                index = i;
                break;
            }
        }
        NN_RESULT_THROW_UNLESS(index >= 0, nn::vi::ResultOperationFailed());

        NN_RESULT_DO(nn::os::CreateSystemEvent(&m_pDestinationSystemEventList[index], nn::os::EventClearMode_ManualClear, true));
        *pOutHandle = nn::os::GetReadableHandleOfSystemEvent(&m_pDestinationSystemEventList[index]);
        m_pDestinationDisplayIdList[index] = displayId;

        NN_VISRV_LOG_EVENT("Added SplittedEvent %d/%d\n", GetSplittedEventCount(), m_DestinationCountMax);

        NN_RESULT_SUCCESS;
    }

    void SystemEventSplitter::RemoveSplittedEvents(nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_EQUAL(displayId, 0);

        for(int i = 0; i < m_DestinationCountMax; i++)
        {
            if(m_pDestinationDisplayIdList[i] == displayId)
            {
                nn::os::DestroySystemEvent(&m_pDestinationSystemEventList[i]);
                m_pDestinationDisplayIdList[i] = 0;
            }
        }
        NN_VISRV_LOG_EVENT("Removed SplittedEvent %d/%d\n", GetSplittedEventCount(), m_DestinationCountMax);
    }

    nn::os::MultiWaitHolderType* SystemEventSplitter::GetSourceMultiWaitHolder() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return &m_SourceWaitHolder;
    }

    nn::os::SystemEventType* SystemEventSplitter::GetSourceEvent() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return &m_SourceSystemEvent;
    }

    void SystemEventSplitter::SignalSplittedEvents() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        for(int i = 0; i < m_DestinationCountMax; i++)
        {
            if(m_pDestinationDisplayIdList[i] != 0)
            {
                nn::os::SignalSystemEvent(&m_pDestinationSystemEventList[i]);
            }
        }
    }

    int SystemEventSplitter::GetSplittedEventCount() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        int count = 0;
        for(int i = 0; i < m_DestinationCountMax; i++)
        {
            if(m_pDestinationDisplayIdList[i] != 0)
            {
                count++;
            }
        }
        return count;
    }

    void SystemEventSplitter::OnSourceEventSignaled(void* pArg) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pArg);
        auto pSplitter = reinterpret_cast<SystemEventSplitter*>(pArg);
        nn::os::ClearSystemEvent(pSplitter->GetSourceEvent());
        pSplitter->SignalSplittedEvents();
    }

}}}

