﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ClientObject.h"

#include <algorithm>

#include <nn/nn_SdkAssert.h>
#include <nn/settings/fwdbg/settings_SettingsCommon.h>

#include <nn/vi/vi_Result.h>
#include <nn/vi/vi_ResultPrivate.h>
#include "../visrv_Log.h"
#include "../visrv_ProcessHeapBlockManager.h"
#include "../master/detail/visrv_MemoryMap.h"
#include "../master/detail/visrv_Android.h"
#include "../master/detail/visrv_Policies.h"
#include "../master/visrv_ActiveDisplay.h"
#include "visrv_ClientUtility.h"
#include "visrv_LayerPool.h"
#include "visrv_PresentationTracerPool.h"
#include "visrv_SystemEventSplitter.h"

namespace
{
    nn::Result EnableDisplay(nn::visrv::master::Display* pDisplay) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pDisplay);

        nn::vi::HotplugStateType hotplug;
        NN_RESULT_DO(pDisplay->GetHotplugState(&hotplug));

        if( hotplug == nn::vi::HotplugState_Connected )
        {
            NN_RESULT_DO(pDisplay->SetPowerState(nn::vi::PowerState_On));
        }
        else
        {
            NN_RESULT_DO(pDisplay->SetPowerState(nn::vi::PowerState_Blank));
        }

        NN_RESULT_SUCCESS;
    }
}

namespace nn{ namespace visrv{ namespace client{

    ClientObject::ClientObject(const ClientConstants& constants) NN_NOEXCEPT
        : m_Constants(constants)
    {
        NN_VISRV_LOG_LIFETIME("ClientObject ctor\n");
    }

    ClientObject::~ClientObject() NN_NOEXCEPT
    {
        NN_VISRV_LOG_LIFETIME("ClientObject dtor\n");
    }

    bool ClientObject::IsAlive() const NN_NOEXCEPT
    {
        return ClientObjectRefcount::IsAlive();
    }

    void ClientObject::Initialize() NN_NOEXCEPT
    {
        InitializeSharing(this);
        m_OwnProcessHeapBlockIdList.Initialize();
    }

    void ClientObject::Finalize() NN_NOEXCEPT
    {
        FinalizeSharing();

        // close all display and destroy all stray layers
        // (if this is the manager, all managed layers will be destroyed)
        if(m_DisplayHolderList.size() > 0)
        {
            NN_VISRV_LOG_OPENCLOSE("Closing all displays: %d displays\n", static_cast<int>(m_DisplayHolderList.size()));
        }
        while(m_DisplayHolderList.size() > 0)
        {
            CloseDisplay(m_DisplayHolderList.front().displayId);
        }

        // unbind all managed layers
        if(m_LayerHolderList.size() > 0)
        {
            NN_VISRV_LOG_OPENCLOSE("Unbinding all managed layers: %d layers\n", static_cast<int>(m_LayerHolderList.size()));
        }
        while(m_LayerHolderList.size() > 0)
        {
            auto& layerHolder = m_LayerHolderList.front();
            NN_SDK_ASSERT_EQUAL(layerHolder.state, ClientLayerHolderState_Bound);
            UnbindManagedLayer(layerHolder.layerId);
        }

        // close all indirect layers
        if(m_OwnIndirectLayerList.size() > 0)
        {
            NN_VISRV_LOG_OPENCLOSE("Closing all indirect layers: %d layers\n", static_cast<int>(m_OwnIndirectLayerList.size()));
        }
        while(m_OwnIndirectLayerList.size() > 0)
        {
            auto& layer = m_OwnIndirectLayerList.front();
            NN_SDK_ASSERT_EQUAL(layer.GetOwnerClient(), this);
            DestroyIndirectLayer(layer.GetHandle());
        }

        // unbind all indirect producer end points
        if(m_ProducerBoundIndirectLayerList.size() > 0)
        {
            NN_VISRV_LOG_OPENCLOSE("Unbinding all indirect producer endpoints: %d endpoints\n", static_cast<int>(m_ProducerBoundIndirectLayerList.size()));
        }
        while(m_ProducerBoundIndirectLayerList.size() > 0)
        {
            auto& layer = m_ProducerBoundIndirectLayerList.front();
            NN_SDK_ASSERT_EQUAL(layer.GetProducerClient(), this);
            UnbindIndirectProducerEndPoint(layer.GetProducerEndPointHandle());
        }

        // unbind all indirect consumer end points
        if(m_ConsumerBoundIndirectLayerList.size() > 0)
        {
            NN_VISRV_LOG_OPENCLOSE("Unbinding all indirect consumer endpoints: %d endpoints\n", static_cast<int>(m_ConsumerBoundIndirectLayerList.size()));
        }
        while(m_ConsumerBoundIndirectLayerList.size() > 0)
        {
            auto& layer = m_ConsumerBoundIndirectLayerList.front();
            NN_SDK_ASSERT_EQUAL(layer.GetConsumerClient(), this);
            UnbindIndirectConsumerEndPoint(layer.GetConsumerEndPointHandle());
        }

        // deallocate all process heap memory blocks
        if(m_OwnProcessHeapBlockIdList.GetCount() > 0)
        {
            NN_VISRV_LOG_OPENCLOSE("Deallocating all process heap memory blocks: %d blocks\n", m_OwnProcessHeapBlockIdList.GetCount());
        }
        while(m_OwnProcessHeapBlockIdList.GetCount() > 0)
        {
            auto blockId = m_OwnProcessHeapBlockIdList.GetHead();
            FreeProcessHeapBlock(blockId);
        }

        NN_SDK_ASSERT_EQUAL(m_DisplayHolderList.size(), 0);
        NN_SDK_ASSERT_EQUAL(m_LayerHolderList.size(), 0);
        NN_SDK_ASSERT(m_BinderTable.IsEmpty());
        NN_SDK_ASSERT_EQUAL(m_OwnIndirectLayerList.size(), 0);
        NN_SDK_ASSERT_EQUAL(m_ProducerBoundIndirectLayerList.size(), 0);
        NN_SDK_ASSERT_EQUAL(m_ConsumerBoundIndirectLayerList.size(), 0);
        NN_SDK_ASSERT_EQUAL(m_OwnProcessHeapBlockIdList.GetCount(), 0);

        m_OwnProcessHeapBlockIdList.Finalize();
    }

    const ClientConstants& ClientObject::GetConstants() const NN_NOEXCEPT
    {
        return m_Constants;
    }

//-------------------------
// ProcessHeap
//-------------------------
    nn::Result ClientObject::AllocateProcessHeapBlock(ResourceId* pOutBlockId, size_t size) NN_NOEXCEPT
    {
        ResourceId blockId = 0;
        NN_RESULT_DO(g_ProcessHeapBlockManager.AllocateMemoryBlock(&blockId, size));
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_OwnProcessHeapBlockIdList.Register(blockId));
        *pOutBlockId = blockId;
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::FreeProcessHeapBlock(ResourceId blockId) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(blockId != 0, nn::vi::ResultInvalidRange());
        NN_RESULT_THROW_UNLESS(m_OwnProcessHeapBlockIdList.Contains(blockId), nn::vi::ResultNotFound());
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_OwnProcessHeapBlockIdList.Unregister(blockId));
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ProcessHeapBlockManager.FreeMemoryBlock(blockId));
        NN_RESULT_SUCCESS;
    }

//-------------------------
// Display
//-------------------------

    nn::Result ClientObject::ListDisplays(
        int64_t* pOutCount,
        nn::vi::DisplayInfo* pOutInfoList,
        int infoListCapacity
        ) NN_NOEXCEPT
    {
        return ListDisplays(pOutCount, pOutInfoList, infoListCapacity, m_Constants.GetPolicyLevel());
    }

    nn::Result ClientObject::ListDisplays(
        int64_t* pOutCount,
        nn::vi::DisplayInfo* pOutInfoList,
        int infoListCapacity,
        nn::vi::PolicyLevelType policyLevel
        ) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutCount);
        NN_VISRV_CHECK_NOT_NULL(pOutInfoList);
        NN_VISRV_CHECK_ALIVE();

        int64_t written = 0;

        int displayCount = master::detail::g_DisplayInfoSegmentCount[policyLevel];
        for(int i = 0; i < infoListCapacity && i < displayCount; ++i)
        {
            (reinterpret_cast<master::detail::PlatformDisplayInfo*>(
                nn::visrv::master::detail::g_DisplayInfoSegment[policyLevel]) + i)->WriteInfo(&pOutInfoList[i]);
            ++written;
        }

        *pOutCount = written;
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::GetDisplayResolution(int64_t* pOutWidth, int64_t* pOutHeight, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutWidth);
        NN_VISRV_CHECK_NOT_NULL(pOutHeight);
        NN_VISRV_CHECK_ALIVE();

        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        nn::vi::DisplayModeInfo mode;
        NN_RESULT_TRY(pDisplayHolder->pDisplay->GetMode(&mode))
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(nn::vi::ResultOperationFailed());
            }
        NN_RESULT_END_TRY;

        *pOutWidth = static_cast<int64_t>(mode.width);
        *pOutHeight = static_cast<int64_t>(mode.height);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::GetDisplayLogicalResolution(int* pOutWidth, int* pOutHeight, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutWidth);
        NN_VISRV_CHECK_NOT_NULL(pOutHeight);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        return pDisplayHolder->pDisplay->GetLogicalResolution(pOutWidth, pOutHeight);
    }

    nn::Result ClientObject::GetZOrderCountMin(int64_t* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutValue);
        NN_VISRV_CHECK_ALIVE();

        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        *pOutValue = pDisplayHolder->GetDisplayInfo().GetPolicy().GetZOrderCountMin();
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::GetZOrderCountMax(int64_t* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutValue);
        NN_VISRV_CHECK_ALIVE();

        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        *pOutValue = pDisplayHolder->GetDisplayInfo().GetPolicy().GetZOrderCountMax();
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::SetDisplayMagnification(nn::vi::DisplayId displayId, int x, int y, int width, int height) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_RESULT_THROW_UNLESS(x >= 0, nn::vi::ResultInvalidDimensions());
        NN_RESULT_THROW_UNLESS(y >= 0, nn::vi::ResultInvalidDimensions());
        NN_RESULT_THROW_UNLESS(width > 0, nn::vi::ResultInvalidDimensions());
        NN_RESULT_THROW_UNLESS(height > 0, nn::vi::ResultInvalidDimensions());

        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        NN_RESULT_DO(pDisplayHolder->pDisplay->SetViewport(x, y, width, height));
        NN_RESULT_SUCCESS;
    }

//-------------------------
// Layer
//-------------------------

    nn::Result ClientObject::ConvertScalingMode(std::int64_t* pOutMode, nn::vi::ScalingModeType viMode) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutMode);
        NN_RESULT_THROW_UNLESS(viMode <= nn::vi::ScalingMode_PreserveAspectRatio, nn::vi::ResultOperationFailed());
        NN_VISRV_CHECK_ALIVE();

        // Note: short-circuiting the Display/Layer infrastructure since this is more of a compositor feature.
        nn::vi::ScalingMode mode = static_cast<nn::vi::ScalingMode>(viMode);
        NN_RESULT_THROW_UNLESS(master::detail::ScalingModeSupport(mode), nn::vi::ResultNotSupported());
        *pOutMode = master::detail::ConvertScalingMode(mode);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::SetLayerVisibility(nn::vi::LayerId layerId, bool isVisible) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(pLayerHolder, layerId);
        return pLayerHolder->pLayer->SetVisibility(isVisible);
    }

    nn::Result ClientObject::SetLayerAlpha(nn::vi::LayerId layerId, float alpha) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_MINMAX(alpha, 0, 1);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(pLayerHolder, layerId)
        return pLayerHolder->pLayer->SetAlpha(alpha);
    }

    nn::Result ClientObject::SetLayerPosition(nn::vi::LayerId layerId, float x, float y) NN_NOEXCEPT
    {
        // valid ranges for x, y will depend on resolution and platform
        // TRM recommends having the entire window within the active area of the display for
        // performance reasons, but may not be true for all DCs.
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(pLayerHolder, layerId)
        return pLayerHolder->pLayer->SetPosition(x, y);

    }

    nn::Result ClientObject::SetLayerSize(nn::vi::LayerId layerId, int64_t width, int64_t height) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(width >= 0 && height >= 0,
                               nn::vi::ResultInvalidDimensions());
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(pLayerHolder, layerId)
        return pLayerHolder->pLayer->SetSize(static_cast<int>(width), static_cast<int>(height));
    }

    nn::Result ClientObject::GetLayerZ(int64_t* pOutZ, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutZ);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(pLayerHolder, layerId)
        *pOutZ = static_cast<int64_t>(pLayerHolder->pLayer->GetZ());
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::SetLayerZ(nn::vi::LayerId layerId, int64_t z) NN_NOEXCEPT
    {
        // z-order range will vary based on platform
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(pLayerHolder, layerId)
        return pLayerHolder->pLayer->SetZ(static_cast<int>(z));
    }

//-------------------------
// Layer
//-------------------------
    nn::Result ClientObject::AttachLayerPresentationTracer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);

        NN_RESULT_THROW_UNLESS(it->pPresentationTracer == nullptr, nn::vi::ResultAlreadyOpened());

        PresentationTracer* pTracer = nullptr;
        NN_RESULT_DO(g_PresentationTracerPool.Acquire(&pTracer));

        it->pPresentationTracer = pTracer;

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::DetachLayerPresentationTracer(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);

        NN_RESULT_THROW_UNLESS(it->pPresentationTracer != nullptr, nn::ResultSuccess()); // ok. No tracer is attached.

        g_PresentationTracerPool.Release(it->pPresentationTracer);
        it->pPresentationTracer = nullptr;

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::StartLayerPresentationRecording(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);

        NN_RESULT_THROW_UNLESS(it->pPresentationTracer != nullptr, nn::vi::ResultDenied());

        it->pPresentationTracer->SetMode(PresentationTracer::Mode_Recording);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::StopLayerPresentationRecording(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);

        NN_RESULT_THROW_UNLESS(it->pPresentationTracer != nullptr, nn::vi::ResultDenied());

        it->pPresentationTracer->SetMode(PresentationTracer::Mode_Idle);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::StartLayerPresentationFenceWait(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);

        NN_RESULT_THROW_UNLESS(it->pPresentationTracer != nullptr, nn::vi::ResultDenied());

        it->pPresentationTracer->SetMode(PresentationTracer::Mode_FenceWait);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::StopLayerPresentationFenceWait(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);

        NN_RESULT_THROW_UNLESS(it->pPresentationTracer != nullptr, nn::vi::ResultDenied());

        it->pPresentationTracer->SetMode(PresentationTracer::Mode_Idle);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::GetLayerPresentationAllFencesExpiredEvent(nn::sf::NativeHandle& outHandle, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);

        NN_RESULT_THROW_UNLESS(it->pPresentationTracer != nullptr, nn::vi::ResultDenied());

        it->pPresentationTracer->GetAllFencesExpiredEvent(outHandle);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::AddToLayerStack(nn::vi::LayerId layerId, nn::vi::LayerStackType stackId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);
        return it->pLayer->AddToStack(static_cast<nn::vi::LayerStack>(stackId));
    }

    nn::Result ClientObject::RemoveFromLayerStack(nn::vi::LayerId layerId, nn::vi::LayerStackType stackId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);
        return it->pLayer->RemoveFromStack(static_cast<nn::vi::LayerStack>(stackId));
    }

    nn::Result ClientObject::SetLayerStackFlags(nn::vi::LayerId layerId, nn::vi::LayerStackFlagType stacks) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);
        return it->pLayer->SetStacks(stacks);
    }

    nn::Result ClientObject::GetLayerStackFlags(nn::vi::LayerStackFlagType* pOutStacks, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutStacks);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);
        *pOutStacks = it->pLayer->GetStacks();
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::SetLayerConfig(nn::vi::LayerId layerId, nn::vi::LayerConfig config) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_RESULT_THROW_UNLESS(master::Layer::IsValidStackSet(config.stacks), nn::vi::ResultInvalidRange());
        NN_VISRV_FIND_CLIENT_LAYER(holder, layerId);
        NN_RESULT_THROW_UNLESS(holder->pLayer->IsValidZOrder(config.zOrder), nn::vi::ResultInvalidRange());

        {
            android::SurfaceComposerClient::openGlobalTransaction();
            NN_UTIL_SCOPE_EXIT
            {
                android::SurfaceComposerClient::closeGlobalTransaction();
            };

            NN_RESULT_DO(holder->pLayer->SetStackUnsafe(config.stacks, config.mask));
            NN_RESULT_DO(holder->pLayer->SetZUnsafe(config.zOrder));
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::SetConductorLayer(nn::vi::LayerId layerId, bool isConductor) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(it, layerId);

        auto pLayerHandle = it->pLayer->m_Handle;
        it->pLayer->m_Client.setConductorLayer(pLayerHandle, isConductor);
        //{
        //    NN_VISRV_LOG_WARN("setConductorLayer(#%d, %s)\n", static_cast<int>(layerId), (isConductor ? "true" : "false"));
        //}

        NN_RESULT_SUCCESS;
    }

//-------------------------
// DisplayMode
//-------------------------
    nn::Result ClientObject::ListDisplayModes(int64_t* pOutCount, nn::vi::DisplayModeInfo* pOutModes, int capacity, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutCount);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        if( pOutModes != nullptr && capacity > 0 )
        {
            int n = pDisplayHolder->pDisplay->ListModes(pOutModes, capacity, pDisplayHolder->policyLevel);
            *pOutCount = static_cast<int64_t>(n);
        }
        else
        {
            *pOutCount = 0;
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::ListDisplayRgbRanges(int64_t* pOutCount, nn::vi::RgbRangeType* pOutRanges, int capacity, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutCount);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        if( pOutRanges != nullptr && capacity > 0 )
        {
            int n = pDisplayHolder->pDisplay->ListRgbRanges(pOutRanges, capacity);
            *pOutCount = static_cast<int64_t>(n);
        }
        else
        {
            *pOutCount = 0;
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::ListDisplayContentTypes(int64_t* pOutCount, nn::vi::ContentTypeType* pOutTypes, int capacity, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutCount);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        NN_UNUSED(pOutTypes);
        NN_UNUSED(capacity);
        NN_UNUSED(pDisplayHolder);

        *pOutCount = 0;
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::GetDisplayMode(nn::vi::DisplayModeInfo* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutValue);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->GetMode(pOutValue);
    }

    nn::Result ClientObject::SetDisplayMode(nn::vi::DisplayId displayId, const nn::vi::DisplayModeInfo& value) NN_NOEXCEPT
    {
        // modes are filtered at lower layers since it's platform dependent
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->SetMode(value, pDisplayHolder->policyLevel);
    }

    nn::Result ClientObject::GetDisplayUnderscan(int64_t* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutValue);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        int value = 0;
        NN_RESULT_DO(pDisplayHolder->pDisplay->GetUnderscan(&value));
        *pOutValue = static_cast<int64_t>(value);
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::SetDisplayUnderscan(nn::vi::DisplayId displayId, int64_t value) NN_NOEXCEPT
    {
        // underscan range will vary by DC, so check at lower level
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->SetUnderscan(static_cast<int>(value));
    }

    nn::Result ClientObject::GetDisplayContentType(nn::vi::ContentTypeType* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutValue);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        *pOutValue = 0;
        NN_UNUSED(pDisplayHolder);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::SetDisplayContentType(nn::vi::DisplayId displayId, nn::vi::ContentTypeType value) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(value < nn::vi::ContentType_Max, nn::vi::ResultOperationFailed());
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        NN_UNUSED(pDisplayHolder);
        NN_UNUSED(value);

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::GetDisplayRgbRange(nn::vi::RgbRangeType* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutValue);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->GetRgbRange(pOutValue);
    }

    nn::Result ClientObject::SetDisplayRgbRange(nn::vi::DisplayId displayId, nn::vi::RgbRangeType value) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(value < nn::vi::RgbRange_Max, nn::vi::ResultOperationFailed());
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->SetRgbRange(static_cast<nn::vi::RgbRange>(value));
    }

    nn::Result ClientObject::GetDisplayCmuMode(nn::vi::CmuModeType* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutValue);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->GetCmuMode(pOutValue);
    }

    nn::Result ClientObject::SetDisplayCmuMode(nn::vi::DisplayId displayId, nn::vi::CmuModeType value) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(value <= nn::vi::CmuMode_Grayscale, nn::vi::ResultOperationFailed());
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->SetCmuMode(static_cast<nn::vi::CmuMode>(value));
    }

    nn::Result ClientObject::GetDisplayContrastRatio(float* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutValue);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->GetContrastRatio(pOutValue);
    }

    nn::Result ClientObject::SetDisplayContrastRatio(nn::vi::DisplayId displayId, float value) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_MINMAX(value, 0.f, 1.f);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->SetContrastRatio(value);
    }

    nn::Result ClientObject::GetDisplayHotplugState(nn::vi::HotplugStateType* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutValue);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->GetHotplugState(pOutValue);
    }

    nn::Result ClientObject::GetDisplayGamma(float* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutValue);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->GetGamma(pOutValue);
    }

    nn::Result ClientObject::SetDisplayGamma(nn::vi::DisplayId displayId, float value) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_MINMAX(value, 1.f, 3.f);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->SetGamma(value);
    }

    nn::Result ClientObject::GetDisplayCmuLuma(float* pOutValue, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->GetCmuLuma(pOutValue);
    }

    nn::Result ClientObject::SetDisplayCmuLuma(nn::vi::DisplayId displayId, float value) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->SetCmuLuma(value);
    }


//-------------------------
// DisplayConfig
//-------------------------
    nn::Result ClientObject::SetDisplayAlpha(nn::vi::DisplayId displayId, float alpha) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_MINMAX(alpha, 0.f, 1.f);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        return pDisplayHolder->pDisplay->SetAlpha(alpha);
    }

    nn::Result ClientObject::SetDisplayLayerStack(nn::vi::DisplayId displayId, nn::vi::LayerStackType id) NN_NOEXCEPT
    {
        // valid layer stacks will vary by platform
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        return pDisplayHolder->pDisplay->SetLayerStack(id);
    }

    nn::Result ClientObject::SetDisplayPowerState(nn::vi::DisplayId displayId, nn::vi::PowerStateType state) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(state <= nn::vi::PowerState_On, nn::vi::ResultOperationFailed());
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        return pDisplayHolder->pDisplay->SetPowerState(static_cast<nn::vi::PowerState>(state));
    }

    nn::Result ClientObject::SetDefaultDisplay(nn::vi::DisplayId id) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, id);

        nn::visrv::master::Display* pCurrent = nn::visrv::master::GetActiveDisplay();

        if( pCurrent != nullptr && pCurrent != pDisplayHolder->pDisplay )
        {
            nn::vi::LayerStackType oldLayerStack;
            NN_RESULT_DO(pCurrent->GetLayerStack(&oldLayerStack));
            oldLayerStack = 1 << oldLayerStack;

            nn::vi::LayerStackType newLayerStack;
            NN_RESULT_DO(pDisplayHolder->pDisplay->GetLayerStack(&newLayerStack));
            newLayerStack = 1 << newLayerStack;

            android::SurfaceComposerClient::openGlobalTransaction();
            for( LayerPool::reference layer : g_LayerPool )
            {
                if( layer.state == ClientLayerHolderState_Bound ||
                    layer.state == ClientLayerHolderState_Stray )
                {
                    NN_SDK_ASSERT_NOT_NULL(layer.pLayer);
                    auto stacks = layer.pLayer->GetStacks();

                    if( (stacks & oldLayerStack) != 0 )
                    {
                        stacks &= ~oldLayerStack;
                        stacks |= (1 << nn::vi::LayerStack_Null);
                        layer.pLayer->m_Control->setLayerStackMask(stacks);
                    }
                }
            }
            android::SurfaceComposerClient::closeGlobalTransaction();

            NN_UTIL_SCOPE_EXIT
            {
                android::SurfaceComposerClient::openGlobalTransaction();
                for( LayerPool::reference layer : g_LayerPool )
                {
                    if( layer.state == ClientLayerHolderState_Bound ||
                        layer.state == ClientLayerHolderState_Stray )
                    {
                        NN_SDK_ASSERT_NOT_NULL(layer.pLayer);
                        auto stacks = layer.pLayer->GetStacks();

                        if( (stacks & oldLayerStack) != 0 )
                        {
                            stacks &= ~oldLayerStack;
                            stacks |= newLayerStack;
                            layer.pLayer->m_Control->setLayerStackMask(stacks);
                        }
                    }
                }
                android::SurfaceComposerClient::closeGlobalTransaction();
            };

            // Vsync may have already signalled during setup
            nn::os::ClearSystemEvent(g_VsyncEventSplitter.Get()->GetSourceEvent());
            if( !nn::os::TimedWaitSystemEvent(g_VsyncEventSplitter.Get()->GetSourceEvent(), nn::TimeSpan::FromMilliSeconds(32)))
            {
                NN_VISRV_LOG_ERR("Vsync timeout!\n");
            }

            NN_RESULT_DO(pDisplayHolder->pDisplay->SetPowerState(nn::vi::PowerState_Blank));
            NN_RESULT_DO(pCurrent->SetPowerState(nn::vi::PowerState_Off));
        }

        // Transition to the new DC is complete, but fully enabling the interface could fail
        // Set current display so transition steps are not performed again
        nn::visrv::master::SetActiveDisplay(pDisplayHolder->pDisplay);
        NN_RESULT_DO(EnableDisplay(pDisplayHolder->pDisplay));

        NN_RESULT_SUCCESS;
    }

//-------------------------
// Display Events
//-------------------------
    nn::Result ClientObject::GetDisplayHotplugEvent(nn::os::NativeHandle* pOutHandle, bool* pOutIsManaged, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutHandle);
        NN_SDK_ASSERT_NOT_NULL(pOutIsManaged);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        NN_RESULT_THROW_UNLESS(
            pDisplayHolder->pDisplay->IsHotplugEventSupported(),
            nn::vi::ResultNotSupported()
        );

        // check event count limit
        NN_RESULT_THROW_UNLESS(
            pDisplayHolder->displayEventInfo.hotplugEventCount < DisplayHotplugEventCountPerClientMax,
            nn::vi::ResultDenied()
        );

        NN_RESULT_DO(g_HotplugEventSplitter.Get()->CreateSplittedEvent(pOutHandle, displayId));
        *pOutIsManaged = false;
        pDisplayHolder->displayEventInfo.hotplugEventCount++;

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::GetDisplayModeChangedEvent(nn::os::NativeHandle* pOutHandle, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutHandle);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        NN_RESULT_THROW_UNLESS(
            pDisplayHolder->pDisplay->IsModeChangedEventSupported(),
            nn::vi::ResultNotSupported()
        );

        // check event count limit
        NN_RESULT_THROW_UNLESS(
            pDisplayHolder->displayEventInfo.modeChangeEventCount < DisplayModeChangedEventCountPerClientMax,
            nn::vi::ResultDenied()
        );

        // TODO: Need some way to differentiate splitters per named display.
        //       Since this is only enabled on the External display currently, this should be ok for now.
        NN_RESULT_DO(g_ExternalModeChangedSplitter.Get()->CreateSplittedEvent(pOutHandle, displayId));
        pDisplayHolder->displayEventInfo.modeChangeEventCount++;

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::GetDisplayVsyncEvent(nn::os::NativeHandle* pOutHandle, bool* pOutIsManaged, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutHandle);
        NN_SDK_ASSERT_NOT_NULL(pOutIsManaged);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        NN_RESULT_THROW_UNLESS(
            pDisplayHolder->pDisplay->IsVsyncEventSupported(),
            nn::vi::ResultNotSupported()
        );

        // check event count limit
        NN_RESULT_THROW_UNLESS(
            pDisplayHolder->displayEventInfo.vsyncEventCount < DisplayVsyncEventCountPerClientMax,
            nn::vi::ResultDenied()
        );

        NN_RESULT_DO(g_VsyncEventSplitter.Get()->CreateSplittedEvent(pOutHandle, displayId));
        *pOutIsManaged = false;
        pDisplayHolder->displayEventInfo.vsyncEventCount++;

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::GetDisplayVsyncEventForDebug(nn::os::NativeHandle* pOutHandle, bool* pOutIsManaged, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutHandle);
        NN_SDK_ASSERT_NOT_NULL(pOutIsManaged);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        NN_RESULT_THROW_UNLESS(
            pDisplayHolder->pDisplay->IsVsyncEventSupported(),
            nn::vi::ResultNotSupported()
        );

        // do not allow in production
        NN_RESULT_THROW_UNLESS(
            nn::settings::fwdbg::IsDebugModeEnabled(),
            nn::vi::ResultDenied()
        );

        NN_RESULT_THROW_UNLESS(
            pDisplayHolder->displayEventInfo.vsyncEventDebugCount < DisplayVsyncEventDebugCountPerClientMax,
            nn::vi::ResultDenied()
        );

        NN_RESULT_DO(g_VsyncEventSplitter.Get()->CreateSplittedEvent(pOutHandle, displayId));
        *pOutIsManaged = false;
        pDisplayHolder->displayEventInfo.vsyncEventDebugCount++;

        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::GetDisplayErrorEvent(nn::os::NativeHandle* pOutHandle, bool* pOutIsManaged, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutHandle);
        NN_SDK_ASSERT_NOT_NULL(pOutIsManaged);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);

        // check event count limit
        NN_RESULT_THROW_UNLESS(
            pDisplayHolder->displayEventInfo.errorEventCount < DisplayErrorReportEventCountPerClientMax,
            nn::vi::ResultDenied()
        );

        NN_RESULT_DO(g_ErrorReportEventSplitter.Get()->CreateSplittedEvent(pOutHandle, displayId));
        *pOutIsManaged = false;
        pDisplayHolder->displayEventInfo.errorEventCount++;

        NN_RESULT_SUCCESS;
    }

//---------
// Error Report
//---------
    nn::Result ClientObject::GetCompositorErrorInfo(nn::vi::CompositorError* pOutErrorInfo, int* pOutLength, int errorID, nn::vi::DisplayId displayId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_NOT_NULL(pOutErrorInfo);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_DISPLAY(pDisplayHolder, displayId);
        return pDisplayHolder->pDisplay->GetCompositorErrorInfo(pOutErrorInfo, pOutLength, errorID);
    }

//-------------------------
// Layer Events
//-------------------------
    nn::Result ClientObject::AcquireLayerTexturePresentingEvent(nn::os::NativeHandle* pOutHandle, bool* pOutIsManaged, nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutHandle);
        NN_SDK_ASSERT_NOT_NULL(pOutIsManaged);
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(pLayerHolder, layerId);

        NN_RESULT_THROW_UNLESS(
            !pLayerHolder->layerTexturePresentingEvent.isInitialized,
            nn::vi::ResultEventAlreadyAcquired()
        );

        bool isSuccess = false;
        {
            NN_RESULT_DO(nn::os::CreateSystemEvent(&pLayerHolder->layerTexturePresentingEvent.event, nn::os::EventClearMode_ManualClear, true));
            pLayerHolder->layerTexturePresentingEvent.isInitialized = true;
            NN_UTIL_SCOPE_EXIT
            {
                if(!isSuccess)
                {
                    nn::os::DestroySystemEvent(&pLayerHolder->layerTexturePresentingEvent.event);
                    pLayerHolder->layerTexturePresentingEvent.isInitialized = false;
                }
            };

            *pOutHandle = nn::os::GetReadableHandleOfSystemEvent(&pLayerHolder->layerTexturePresentingEvent.event);
            *pOutIsManaged = false;

            isSuccess = true;
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result ClientObject::ReleaseLayerTexturePresentingEvent(nn::vi::LayerId layerId) NN_NOEXCEPT
    {
        NN_VISRV_CHECK_ALIVE();
        NN_VISRV_FIND_CLIENT_LAYER(pLayerHolder, layerId);

        NN_RESULT_THROW_UNLESS(
            pLayerHolder->layerTexturePresentingEvent.isInitialized,
            nn::vi::ResultEventNotFound()
        );

        nn::os::DestroySystemEvent(&pLayerHolder->layerTexturePresentingEvent.event);
        pLayerHolder->layerTexturePresentingEvent.isInitialized = false;
        NN_RESULT_SUCCESS;
    }

}}}
