﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/vi_Result.h>
#include "vi_Layer.h"
#include "detail/vi_WindowManager.h"
#include "vi_Display.h"
#include "detail/vi_Globals.h"
#include "detail/vi_Common.h"

nn::vi::Layer::Layer(Display* pDisplay) NN_NOEXCEPT
    : m_pDisplay(pDisplay)
    , m_Window(nullptr)
{
}

nn::vi::Layer::~Layer() NN_NOEXCEPT
{
    NN_SDK_ASSERT(detail::GetInitializerLock().IsLockedByCurrentThread());

    if( m_Window != nullptr )
    {
        detail::g_WindowManager->DestroyWindow(m_Window);
    }
}

nn::Result nn::vi::Layer::Initialize(int width, int height, PixelFormat format, LayerSettings settings) NN_NOEXCEPT
{
    NN_SDK_ASSERT(detail::GetInitializerLock().IsLockedByCurrentThread());

    NN_UNUSED(format);
    NN_UNUSED(settings);
    NN_UNUSED(width);
    NN_UNUSED(height);

    m_Window = detail::g_WindowManager->CreateWindow(1280, 720, m_pDisplay->GetInfo().IsTouchscreen());

    if( m_Window == nullptr )
    {
        return nn::vi::ResultOperationFailed();
    }

    PIXELFORMATDESCRIPTOR descriptor = { 0 };
    descriptor.nSize = sizeof(PIXELFORMATDESCRIPTOR);
    descriptor.nVersion = 1;
    descriptor.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
    descriptor.iPixelType = PFD_TYPE_RGBA;
    descriptor.cColorBits = 32;

    // Note: Since the window has CS_OWNDC style, calling ReleaseDC is not
    //       necessary since this DC is unique to the window.
    HDC context = GetDC(m_Window);
    int selectedPixelFormat = ChoosePixelFormat(context, &descriptor);

    if ( SetPixelFormat(context, selectedPixelFormat, &descriptor) == FALSE )
    {
        return nn::vi::ResultOperationFailed();
    }

    if ( SetVisibility(true).IsFailure() )
    {
        return nn::vi::ResultOperationFailed();
    }

    return nn::ResultSuccess();
}

nn::Result nn::vi::Layer::SetPosition(float x, float y) NN_NOEXCEPT
{
    NN_UNUSED(x);
    NN_UNUSED(y);

    return nn::ResultSuccess();
}

nn::Result nn::vi::Layer::SetZ(int z) NN_NOEXCEPT
{
    NN_UNUSED(z);

    return nn::ResultSuccess();
}

nn::Result nn::vi::Layer::SetScalingMode(ScalingMode mode) NN_NOEXCEPT
{
    if( m_pDisplay->GetInfo().ValidateScalingMode(mode) )
    {
        return nn::ResultSuccess();
    }

    return nn::vi::ResultNotSupported();
}

nn::Result nn::vi::Layer::SetSize(int width, int height) NN_NOEXCEPT
{
    NN_UNUSED(width);
    NN_UNUSED(height);

    return nn::ResultSuccess();
}

nn::Result nn::vi::Layer::SetVisibility(bool isVisible) NN_NOEXCEPT
{
    int command;

    if( isVisible )
    {
        command = SW_SHOW;
    }
    else
    {
        command = SW_HIDE;
    }

    // return value only has to do with previous state
    ShowWindow(m_Window, command);

    return nn::ResultSuccess();
}

nn::Result nn::vi::Layer::SetAlpha(float alpha) NN_NOEXCEPT
{
    NN_UNUSED(alpha);

    return nn::ResultSuccess();
}

float nn::vi::Layer::GetX() const NN_NOEXCEPT
{
    return 0.f;
}

float nn::vi::Layer::GetY() const NN_NOEXCEPT
{
    return 0.f;
}

int nn::vi::Layer::GetZ() const NN_NOEXCEPT
{
    return 0;
}

int nn::vi::Layer::GetCurrentWidth() const NN_NOEXCEPT
{
    return 0;
}

int nn::vi::Layer::GetCurrentHeight() const NN_NOEXCEPT
{
    return 0;
}

nn::vi::NativeWindowHandle nn::vi::Layer::GetNativeWindow() const NN_NOEXCEPT
{
    return m_Window;
}

nn::vi::Display* nn::vi::Layer::GetDisplay() const NN_NOEXCEPT
{
    return m_pDisplay;
}
