﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/vi_Layer.h>
#include <nn/vi/vi_LayerConfig.h>
#include <nn/vi/vi_Result.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/am/am_Shim.h>

#include "vi_Config.h"
#include "vi_MemoryManagement.h"
#include "vi_CommonUtility.h"
#include "vi_DisplayUtility.h"
#include "vi_LayerUtility.h"

#include "native/vi_DeserializeNativeWindow.h"
#include "dev/vi_ExternalLayer.h"

namespace nn{ namespace vi{

    namespace {

        vi::LayerId CreateManagedDisplayLayerByAppletManager() NN_NOEXCEPT
        {
            vi::LayerId ret;
            NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetSelfController()->CreateManagedDisplayLayer(&ret));
            return ret;
        }

        template <typename ServiceType>
        nn::Result CreateLayerInternal(Layer** pOutLayer,
                                       Display* pDisplay,
                                       LayerSettings settings,
                                       nn::sf::SharedPointer<ServiceType> pService,
                                       int width,
                                       int height
                                       ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutLayer);

            *pOutLayer = nullptr;
            NN_RESULT_THROW_UNLESS(
                width > 0 &&
                height > 0,
                nn::vi::ResultInvalidDimensions()
            );

            std::lock_guard<nn::os::Mutex> displayLock(ObjectManager::GetDisplayLock());
            NN_VI_REQUIRES_DISPLAY_OPENED(pDisplay);
            std::lock_guard<nn::os::Mutex> layerLock(ObjectManager::GetLayerLock());
            NN_SDK_REQUIRES_LESS(GetDisplayHolder(pDisplay)->GetLayerCount(), LayerCountPerDisplayMax);
            NN_SDK_REQUIRES_LESS(ObjectManager::GetValidLayerHolderList()->size(), LayerCountPerDisplayMax * DisplayCountMax);

            char nativeWindowDataBuffer[256];
            int64_t nativeWindowDataSize = 0;

            LayerId id = {};

            nn::applet::AppletResourceUserId aruid = nn::applet::GetAppletResourceUserId();
            if(dev::g_ExternalLayerId != 0)
            {
                LayerId layerId = dev::g_ExternalLayerId;
                NN_RESULT_DO(
                    GetService()->OpenLayer(
                        nn::sf::Out<int64_t>(&nativeWindowDataSize),
                        nn::sf::OutBuffer(nativeWindowDataBuffer, sizeof(nativeWindowDataBuffer)),
                        layerId,
                        GetDisplayHolder(pDisplay)->GetDisplayName(),
                        aruid
                    )
                );
                id = layerId;
            }
            else if(aruid == nn::applet::AppletResourceUserId::GetInvalidId())
            {
                NN_RESULT_DO(
                    pService->CreateStrayLayer(
                        nn::sf::Out<LayerId>(&id),
                        nn::sf::Out<int64_t>(&nativeWindowDataSize),
                        nn::sf::OutBuffer(nativeWindowDataBuffer, sizeof(nativeWindowDataBuffer)),
                        GetDisplayId(pDisplay),
                        settings._storage[0]
                    )
                );
            }
            else
            {
                LayerId layerId = CreateManagedDisplayLayerByAppletManager();
                NN_RESULT_DO(
                    GetService()->OpenLayer(
                        nn::sf::Out<int64_t>(&nativeWindowDataSize),
                        nn::sf::OutBuffer(nativeWindowDataBuffer, sizeof(nativeWindowDataBuffer)),
                        layerId,
                        GetDisplayHolder(pDisplay)->GetDisplayName(),
                        aruid
                    )
                );
                id = layerId;
            }

            auto it = ObjectManager::CreateValidateLayerHolder(id, GetDisplayHolder(pDisplay));

            NN_ABORT_UNLESS_RESULT_SUCCESS(
                DeserializeNativeWindow(&it->m_NativeWindowHolder, nativeWindowDataBuffer, nativeWindowDataSize)
            );

            native_window_set_buffers_user_dimensions(it->GetNativeWindow(), width, height);

            *pOutLayer = reinterpret_cast<Layer*>(&(*it));
            NN_RESULT_SUCCESS;
        }
    }

//--------------------------
// Public API
//--------------------------

    nn::Result CreateLayer(
        Layer** pOutLayer,
        Display* pDisplay
        ) NN_NOEXCEPT
    {
        return CreateLayer(pOutLayer, pDisplay, LayerDefaultWidth, LayerDefaultHeight);
    }

    nn::Result CreateLayer(
        Layer** pOutLayer,
        Display* pDisplay,
        int width,
        int height
    ) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED();

        LayerSettings settings;
        SetLayerSettingsDefaults(&settings);

        return CreateLayerInternal(pOutLayer, pDisplay, settings, GetService(), width, height);
    }

    void DestroyLayer(Layer* pLayer) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED();
        std::lock_guard<nn::os::Mutex> lock(ObjectManager::GetLayerLock());
        NN_VI_REQUIRES_LAYER_VALID(pLayer);

        GetLayerHolder(pLayer)->ReleaseNativeWindow();

        nn::applet::AppletResourceUserId aruid = nn::applet::GetAppletResourceUserId();
        if(dev::g_ExternalLayerId != 0)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                GetService()->CloseLayer(GetLayerId(pLayer))
            );
        }
        else if(aruid == nn::applet::AppletResourceUserId::GetInvalidId())
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                GetService()->DestroyStrayLayer(GetLayerId(pLayer))
            );
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                GetService()->CloseLayer(GetLayerId(pLayer))
            );
        }

        auto it = ObjectManager::FindValidLayerHolder(pLayer);
        ObjectManager::InvalidateDestroyLayerHolder(it);
    }

    nn::Result SetLayerScalingMode(Layer* pLayer, ScalingMode mode) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED();
        NN_SDK_REQUIRES_NOT_NULL(pLayer);
        std::lock_guard<nn::os::Mutex> lock(ObjectManager::GetLayerLock());
        NN_VI_REQUIRES_LAYER_VALID(pLayer);

        auto holder = GetLayerHolder(pLayer);

        std::int64_t nativeMode;
        NN_RESULT_DO(GetService()->ConvertScalingMode(&nativeMode, mode));
        NN_RESULT_THROW_UNLESS(native_window_set_scaling_mode(holder->GetNativeWindow(), nativeMode) == 0,
                               nn::vi::ResultOperationFailed());

        NN_RESULT_SUCCESS;
    }

    nn::Result GetNativeWindow(NativeWindowHandle* pOutWindow, Layer* pLayer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutWindow);
        NN_SDK_REQUIRES_NOT_NULL(pLayer);

        std::lock_guard<nn::os::Mutex> lock(ObjectManager::GetLayerLock());

        // check if pLayer is external native window
        for(int i = 0; i < dev::ExternalNativeWindowCount; i++)
        {
            if(pLayer == reinterpret_cast<Layer*>(&dev::g_ExternalNativeWindowList[i]))
            {
                *pOutWindow = dev::g_ExternalNativeWindowList[i];
                NN_RESULT_SUCCESS;
            }
        }

        NN_VI_REQUIRES_INITIALIZED();
        NN_VI_REQUIRES_LAYER_VALID(pLayer);

        auto pHolder = GetLayerHolder(pLayer);
        *pOutWindow = pHolder->GetNativeWindow();
        NN_RESULT_SUCCESS;
    }

//--------------------------
// System API
//--------------------------

    nn::Result CreateLayer(
        Layer** pOutLayer,
        Display* pDisplay,
        LayerSettings settings
        ) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();

        return CreateLayerInternal(pOutLayer,
                                   pDisplay,
                                   settings,
                                   GetSystemService(),
                                   LayerDefaultWidth,
                                   LayerDefaultHeight
        );
    }

    Result SetLayerPosition(Layer* pLayer, float x, float y) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        return GetSystemService()->SetLayerPosition(GetLayerIdWithValidation(pLayer), x, y);
    }

    Result SetLayerSize(Layer* pLayer, int width, int height) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        return GetSystemService()->SetLayerSize(GetLayerIdWithValidation(pLayer), static_cast<int64_t>(width), static_cast<int64_t>(height));
    }

    Result SetLayerZ(Layer* pLayer, int z) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        return GetSystemService()->SetLayerZ(GetLayerIdWithValidation(pLayer), static_cast<int64_t>(z));
    }

    Result GetLayerZ(int* pOutZ, const Layer* pLayer) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        NN_SDK_REQUIRES_NOT_NULL(pOutZ);
        int64_t value = 0;
        NN_RESULT_DO(GetSystemService()->GetLayerZ(&value, GetLayerIdWithValidation(pLayer)));
        *pOutZ = static_cast<int>(value);
        NN_RESULT_SUCCESS;
    }

    nn::Result SetLayerVisibility(Layer* pLayer, bool isVisible) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_MANAGER();

        return GetManagerService()->SetLayerVisibility(GetLayerIdWithValidation(pLayer), isVisible);
    }

    nn::Result SetLayerAlpha(Layer* pLayer, float alpha) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();

        return GetSystemService()->SetLayerAlpha(GetLayerIdWithValidation(pLayer), alpha);
    }

//--------------------------
// Manager API
//--------------------------

    nn::Result AddToLayerStack(Layer* pLayer, LayerStack id) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_MANAGER();

        return GetManagerService()->AddToLayerStack(GetLayerIdWithValidation(pLayer), id);
    }

    nn::Result RemoveFromLayerStack(Layer* pLayer, LayerStack id) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_MANAGER();

        return GetManagerService()->RemoveFromLayerStack(GetLayerIdWithValidation(pLayer), id);
    }

}}
