﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/vi_DisplayMode.private.h>

#include <algorithm>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include "vi_Config.h"
#include "vi_MemoryManagement.h"
#include "vi_CommonUtility.h"
#include "vi_DisplayUtility.h"
#include "vi_LayerUtility.h"

namespace nn{ namespace vi{

//--------------------------
// System API
//--------------------------
    int ListDisplayModes(DisplayModeInfo* pOutModes, int modeCountMax, const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        // TODO: not a precondition!
        NN_SDK_REQUIRES_NOT_NULL(pOutModes);
        NN_SDK_REQUIRES_GREATER_EQUAL(modeCountMax, 0);
        int64_t count = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetSystemService()->ListDisplayModes(
                &count,
                nn::sf::OutArray<DisplayModeInfo>(pOutModes, static_cast<size_t>(modeCountMax)),
                GetDisplayIdWithValidation(pDisplay)
            )
        );
        return static_cast<int>(count);
    }

    int ListDisplayRgbRanges(RgbRange* pOutRanges, int rgbCountMax, const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        NN_SDK_REQUIRES_NOT_NULL(pOutRanges);
        NN_SDK_REQUIRES_GREATER_EQUAL(rgbCountMax, 0);
        int64_t count = 0;
        RgbRangeType list[RgbRange_Max];
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetSystemService()->ListDisplayRgbRanges(
                &count,
                nn::sf::OutArray<RgbRangeType>(list, std::min(static_cast<size_t>(rgbCountMax), static_cast<size_t>(RgbRange_Max))),
                GetDisplayIdWithValidation(pDisplay)
            )
        );
        NN_SDK_ASSERT_LESS_EQUAL(count, static_cast<int64_t>(rgbCountMax));
        for(int64_t i = 0; i < count; i++)
        {
            pOutRanges[i] = static_cast<RgbRange>(list[i]);
        }
        return static_cast<int>(count);
    }

    int ListDisplayContentTypes(ContentType* pOutTypes, int contentTypeCountMax, const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        NN_SDK_REQUIRES_NOT_NULL(pOutTypes);
        NN_SDK_REQUIRES_GREATER_EQUAL(contentTypeCountMax, 0);
        int64_t count = 0;
        ContentTypeType list[ContentType_Max];
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetSystemService()->ListDisplayContentTypes(
                &count,
                nn::sf::OutArray<ContentTypeType>(list, std::min(static_cast<size_t>(contentTypeCountMax), static_cast<size_t>(ContentType_Max))),
                GetDisplayIdWithValidation(pDisplay)
            )
        );
        NN_SDK_ASSERT_LESS_EQUAL(count, static_cast<int64_t>(contentTypeCountMax));
        for(int64_t i = 0; i < count; i++)
        {
            pOutTypes[i] = static_cast<ContentType>(list[i]);
        }
        return static_cast<int>(count);
    }

    Result GetDisplayMode(DisplayModeInfo* pOutMode, const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        NN_SDK_REQUIRES_NOT_NULL(pOutMode);
        return GetSystemService()->GetDisplayMode(pOutMode, GetDisplayIdWithValidation(pDisplay));
    }

    Result SetDisplayMode(Display* pDisplay, const DisplayModeInfo* pMode) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        NN_SDK_REQUIRES_NOT_NULL(pMode);
        return GetSystemService()->SetDisplayMode(GetDisplayIdWithValidation(pDisplay), *pMode);
    }

    Result GetDisplayUnderscan(int* pOutUnderscan, const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        NN_SDK_REQUIRES_NOT_NULL(pOutUnderscan);
        int64_t value = {};
        NN_RESULT_DO(GetSystemService()->GetDisplayUnderscan(&value, GetDisplayIdWithValidation(pDisplay)));
        *pOutUnderscan = static_cast<int>(value);
        NN_RESULT_SUCCESS;
    }

    Result SetDisplayUnderscan(Display* pDisplay, int underscan) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        return GetSystemService()->SetDisplayUnderscan(GetDisplayIdWithValidation(pDisplay), static_cast<int64_t>(underscan));
    }

    Result GetDisplayContentType(ContentType* pOutType, const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        NN_SDK_REQUIRES_NOT_NULL(pOutType);
        ContentTypeType value = {};
        NN_RESULT_DO(GetSystemService()->GetDisplayContentType(&value, GetDisplayIdWithValidation(pDisplay)));
        *pOutType = static_cast<ContentType>(value);
        NN_RESULT_SUCCESS;
    }

    Result SetDisplayContentType(Display* pDisplay, ContentType type) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        return GetSystemService()->SetDisplayContentType(GetDisplayIdWithValidation(pDisplay), static_cast<ContentTypeType>(type));
    }

    Result GetDisplayRgbRange(RgbRange* pOutRange, const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        NN_SDK_REQUIRES_NOT_NULL(pOutRange);
        RgbRangeType value = {};
        NN_RESULT_DO(GetSystemService()->GetDisplayRgbRange(&value, GetDisplayIdWithValidation(pDisplay)));
        *pOutRange = static_cast<RgbRange>(value);
        NN_RESULT_SUCCESS;
    }

    Result SetDisplayRgbRange(Display* pDisplay, RgbRange range) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        return GetSystemService()->SetDisplayRgbRange(GetDisplayIdWithValidation(pDisplay), static_cast<RgbRangeType>(range));
    }

}}
