﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/vi_Display.h>
#include <nn/vi/vi_Display.private.h>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_StringUtil.h>

#include <nn/vi/vi_Result.h>
#include "vi_Config.h"
#include "vi_MemoryManagement.h"
#include "vi_CommonUtility.h"
#include "vi_DisplayUtility.h"
#include "vi_LayerUtility.h"

namespace nn{ namespace vi{

//--------------------------
// Public API
//--------------------------

    int ListDisplays(DisplayInfo* pOutDisplays, int count) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED();
        if(pOutDisplays == nullptr || count == 0)
        {
            return 0;
        }
        int64_t outCount = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetService()->ListDisplays(&outCount, nn::sf::OutArray<DisplayInfo>(pOutDisplays, static_cast<size_t>(count)))
        );
        return static_cast<int>(outCount);
    }

    namespace {
        nn::Result ReopenOpenedDisplay(Display** pOutDisplay, const char* name) NN_NOEXCEPT
        {
            auto it = ObjectManager::FindValidDisplayHolderByName(name);
            NN_RESULT_THROW_UNLESS(
                it != ObjectManager::GetValidDisplayHolderList()->end(),
                ResultOperationFailed()
            );
            *pOutDisplay = reinterpret_cast<Display*>(&(*it));
            it->IncrementOpenCount();
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result OpenDefaultDisplay(Display** pOutDisplay) NN_NOEXCEPT
    {
        return OpenDisplay(pOutDisplay, NN_VI_DEFAULT_DISPLAY_NAME);
    }

    nn::Result OpenDisplay(Display** pOutDisplay, const char* name) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED();
        NN_SDK_REQUIRES_NOT_NULL(pOutDisplay);
        // nn::util::Strnlen requires name != nullptr
        // check for non-null here to improve error messages
        NN_SDK_REQUIRES_NOT_NULL(name);
        NN_SDK_REQUIRES_LESS(nn::util::Strnlen(name, sizeof(DisplayName)), sizeof(DisplayName));

        *pOutDisplay = nullptr;

        NN_RESULT_THROW_UNLESS(
            name != nullptr && name[0] != '\0',
            ResultNotFound()
        );
        NN_RESULT_THROW_UNLESS(
            nn::util::Strnlen(name, sizeof(DisplayName)) < static_cast<int>(sizeof(DisplayName)),
            ResultNotFound()
        );

        DisplayName displayName = {};
        nn::util::Strlcpy(displayName.value, name, sizeof(DisplayName));

        // take lock here since size could change on concurrent close
        std::lock_guard<nn::os::Mutex> lock(ObjectManager::GetDisplayLock());
        NN_SDK_REQUIRES_LESS(ObjectManager::GetValidDisplayHolderList()->size(), DisplayCountMax);

        DisplayId id = {};
        NN_RESULT_TRY(GetService()->OpenDisplay(&id, displayName))
            NN_RESULT_CATCH(ResultAlreadyOpened)
            {
                return ReopenOpenedDisplay(pOutDisplay, name);
            }
        NN_RESULT_END_TRY;

        auto it = ObjectManager::CreateValidateDisplayHolder(id, name);
        it->IncrementOpenCount();

        *pOutDisplay = reinterpret_cast<Display*>(&(*it));
        NN_RESULT_SUCCESS;
    }

    void CloseDisplay(Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED();
        std::lock_guard<nn::os::Mutex> displayLock(ObjectManager::GetDisplayLock());
        NN_VI_REQUIRES_DISPLAY_OPENED(pDisplay);

        auto it = ObjectManager::FindValidDisplayHolder(pDisplay);
        it->DecrementOpenCount();

        if(it->GetOpenCount() > 0)
        {
            return;
        }

        std::lock_guard<nn::os::Mutex> layerLock(ObjectManager::GetLayerLock());
        // NOTE:
        //   We should release native windows on this display before destroy holders
        for(auto& layerHolder : *ObjectManager::GetValidLayerHolderList())
        {
            if(layerHolder.GetDisplayHolder() == GetDisplayHolder(pDisplay))
            {
                layerHolder.ReleaseNativeWindow();
            }
        }

        // NOTE:
        //   Layers on server side will be destroyed automatically.
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetService()->CloseDisplay(GetDisplayId(pDisplay))
        );


        // NOTE:
        //   LayerHolders will be invalidated and deallocated.
        ObjectManager::InvalidateDestroyDisplayHolder(it);
    }

    nn::Result GetDisplayVsyncEvent(nn::os::SystemEventType* pOutEvent, Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED();
        NN_SDK_REQUIRES_NOT_NULL(pOutEvent);

        nn::sf::NativeHandle handle;

        nn::Result result = GetService()->GetDisplayVsyncEvent(&handle, GetDisplayIdWithValidation(pDisplay));

        if( result.IsSuccess() )
        {
            nn::os::AttachReadableHandleToSystemEvent(pOutEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
            handle.Detach();
        }

        return result;
    }

//--------------------------
// System API
//--------------------------

    int GetZOrderCountMin(const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();

        DisplayId id = GetDisplayIdWithValidation(pDisplay);

        int64_t value = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetSystemService()->GetZOrderCountMin(&value, id)
        );
        return static_cast<int>(value);
    }

    int GetZOrderCountMax(const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();

        int64_t value = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            GetSystemService()->GetZOrderCountMax(&value, GetDisplayIdWithValidation(pDisplay))
        );
        return static_cast<int>(value);
    }

    Result GetDisplayLogicalResolution(int* pOutWidth, int* pOutHeight, const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        NN_SDK_REQUIRES_NOT_NULL(pOutWidth);
        NN_SDK_REQUIRES_NOT_NULL(pOutHeight);

        return GetSystemService()->GetDisplayLogicalResolution(pOutWidth, pOutHeight, GetDisplayIdWithValidation(pDisplay));
    }

    nn::Result GetDisplayResolution(int* pOutWidth, int* pOutHeight, const Display* pDisplay) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        NN_SDK_REQUIRES_NOT_NULL(pOutWidth);
        NN_SDK_REQUIRES_NOT_NULL(pOutHeight);
        NN_VI_REQUIRES_DISPLAY_OPENED(pDisplay);

        nn::vi::DisplayModeInfo mode;

        NN_RESULT_DO(
            GetSystemService()->GetDisplayMode(
                &mode,
                GetDisplayIdWithValidation(pDisplay)
            )
        );

        *pOutWidth  = static_cast<int>(mode.width);
        *pOutHeight = static_cast<int>(mode.height);

        NN_RESULT_SUCCESS;
    }


    nn::Result SetDisplayMagnification(Display* pDisplay, int x, int y, int width, int height) NN_NOEXCEPT
    {
        NN_VI_REQUIRES_INITIALIZED_SYSTEM();
        NN_RESULT_DO(GetSystemService()->SetDisplayMagnification(GetDisplayIdWithValidation(pDisplay), x, y, width, height));
        NN_RESULT_SUCCESS;
    }

}}
