﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/manager/vi_Manager.h>
#include "vi_ManagerServiceData.h"

#include <nn/vi/sf/vi_ServiceName.h>
#include "../vi_Config.h"

// relay
#include <binder/HOSServiceManager.h>
#include <utils/String8.h>

namespace nn{ namespace vi{ namespace manager{

    nn::vi::ProxyName DefaultProxyName = { NN_VI_DEFAULT_DISPLAY_NAME };

    DisplayManagerService::DisplayManagerService() NN_NOEXCEPT
        : m_ClientManager(NN_SF_SIMPLE_ALL_IN_ONE_HIPC_CLIENT_MANAGER_INITIALIZER)
        , m_RootServiceHolder(NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER)
    {
        std::memset(m_Storage, 0, sizeof(m_Storage));
    }

    void DisplayManagerService::Initialize(
        PolicyLevelType policyLevel,
        const nn::vi::ProxyName* pProxyName
    ) NN_NOEXCEPT
    {
        InitializeMinimum(policyLevel, pProxyName);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_pApplicationDisplayService->GetRelayService(&m_pRelayService)
        );

        {
            char proxyNameSafe[sizeof(nn::vi::ProxyName) + 1] = {};
            std::memcpy(&proxyNameSafe, &m_ProxyName, sizeof(nn::vi::ProxyName));
            NN_ABORT_UNLESS_EQUAL(android::NO_ERROR, android::defaultHOSServiceManager()->addServiceProxy(android::String8(proxyNameSafe), m_pRelayService));
        }

        // initialize gralloc
        NN_STATIC_ASSERT(sizeof(m_Storage) >= sizeof(ManagerServiceData));
        NN_STATIC_ASSERT(NN_ALIGNOF(m_Storage) >= NN_ALIGNOF(ManagerServiceData));
        auto pData = new(&m_Storage[0]) ManagerServiceData;
        NN_ABORT_UNLESS_EQUAL(0, hw_get_module(GRALLOC_HARDWARE_MODULE_ID, &pData->pModule));
        NN_ABORT_UNLESS_EQUAL(0, gralloc_open(pData->pModule, &pData->pGralloc));
    }

    void DisplayManagerService::Finalize() NN_NOEXCEPT
    {
        char proxyNameSafe[sizeof(nn::vi::ProxyName) + 1] = {};
        std::memcpy(&proxyNameSafe, &m_ProxyName, sizeof(nn::vi::ProxyName));

        NN_ABORT_UNLESS_GREATER_EQUAL(android::defaultHOSServiceManager()->removeServiceProxy(android::String8(proxyNameSafe)), 0);
        m_pRelayService.Reset();

        // finalize gralloc
        auto pData = GetData();
        gralloc_close(pData->pGralloc);
        pData->pGralloc = nullptr;
        pData->pModule = nullptr;
        pData->~ManagerServiceData();

        FinalizeMinimum();
    }

    void DisplayManagerService::InitializeMinimum(
        PolicyLevelType policyLevel,
        const nn::vi::ProxyName* pProxyName
    ) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_ClientManager.InitializeShimLibraryHolder(&m_RootServiceHolder, nn::vi::sf::ManagerDisplayServiceName)
        );

        if(pProxyName == nullptr)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                m_RootServiceHolder.GetObject()->GetDisplayService(&m_pApplicationDisplayService, policyLevel)
            );
            m_ProxyName = DefaultProxyName;
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                m_RootServiceHolder.GetObject()->GetDisplayServiceWithProxyNameExchange(&m_pApplicationDisplayService, policyLevel, *pProxyName)
            );
            m_ProxyName = *pProxyName;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_pApplicationDisplayService->GetSystemDisplayService(&m_pSystemDisplayService)
        );
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_pApplicationDisplayService->GetManagerDisplayService(&m_pManagerDisplayService)
        );
    }

    void DisplayManagerService::FinalizeMinimum() NN_NOEXCEPT
    {
        m_pManagerDisplayService.Reset();
        m_pSystemDisplayService.Reset();
        m_pApplicationDisplayService.Reset();
        m_RootServiceHolder.FinalizeHolder();
    }


}}}
