﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/fbshare/vi_SharedTexturePool.h>

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>
#include "../vi_Log.h"

#include <nvn/nvn_FuncPtrInline.h>

namespace nn{ namespace vi{ namespace fbshare{

    SharedTexturePool::SharedTexturePool() NN_NOEXCEPT
        : m_State(SharedTexturePoolState_Invalid)
        , m_pDevice()
        , m_Format(NVN_FORMAT_NONE)
        , m_pMemoryPool()
        , m_TextureStateList()
        , m_TextureList()
    {
    }

    bool SharedTexturePool::IsInitialized() const NN_NOEXCEPT
    {
        return m_State != SharedTexturePoolState_Invalid;
    }

    nn::Result SharedTexturePool::Initialize(NVNdevice* pDevice, SharedTextureMemoryPool* pMemoryPool, const SharedMemoryPoolLayout& layout, NVNformat format) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDevice);
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pMemoryPool);
        NN_SDK_REQUIRES(pMemoryPool->IsInitialized());

        int count = layout.count;
        NN_RESULT_THROW_UNLESS(count > 0 && count <= TextureCountMax, nn::vi::ResultInvalidRange());

        size_t requiredSize = 0;
        for(int i = 0; i < count; i++)
        {
            auto& e = layout.entries[i];
            requiredSize = std::max<size_t>(requiredSize, e.offset + e.size);
        }
        NN_RESULT_THROW_UNLESS(pMemoryPool->GetSize() >= requiredSize, nn::vi::ResultOperationFailed());

        for(int i = 0; i < count; i++)
        {
            m_TextureStateList[i] = SharedTextureState_Invalid;
        }

        m_pDevice = pDevice;
        m_Format = format;
        m_pMemoryPool = pMemoryPool;
        m_Layout      = layout;
        m_State = SharedTexturePoolState_Initialized;
        NN_RESULT_SUCCESS;
    }

    void SharedTexturePool::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        int count = m_Layout.count;
        for(int i = 0; i < count; i++)
        {
            if(m_TextureStateList[i] != SharedTextureState_Invalid)
            {
                DestroyTextureImpl(i);
            }
        }

        m_pMemoryPool = nullptr;
        m_Layout = {};
        m_Format = NVN_FORMAT_NONE;
        m_pDevice = nullptr;
        m_State = SharedTexturePoolState_Invalid;
    }

    void SharedTexturePool::SetupTextureBuilder(NVNtextureBuilder& texBuilder, NVNdevice* pDevice, int width, int height, NVNformat format, bool isCompressible) NN_NOEXCEPT
    {
        int flags =
            NVN_TEXTURE_FLAGS_DISPLAY_BIT |
            NVN_TEXTURE_FLAGS_IMAGE_BIT |
            (isCompressible ? NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT : 0);

        nvnTextureBuilderSetDevice(&texBuilder, pDevice);
        nvnTextureBuilderSetDefaults(&texBuilder);
        nvnTextureBuilderSetFlags(&texBuilder, flags);
        nvnTextureBuilderSetFormat(&texBuilder, format);
        nvnTextureBuilderSetWidth(&texBuilder, width);
        nvnTextureBuilderSetHeight(&texBuilder, height);
        nvnTextureBuilderSetDepth(&texBuilder, 1);
        nvnTextureBuilderSetTarget(&texBuilder, NVN_TEXTURE_TARGET_2D);
    }

    nn::Result SharedTexturePool::CreateTextureImpl(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_RANGE(index, 0, m_Layout.count);
        NN_SDK_REQUIRES_EQUAL(m_TextureStateList[index], SharedTextureState_Invalid);

        NN_VI_LOG_DEV("[share]Creating texture %d\n", index);

        auto& layout = m_Layout.entries[index];
        NVNtexture& texture = m_TextureList[index];
        {
            NVNtextureBuilder texBuilder;
            SetupTextureBuilder(texBuilder, m_pDevice, layout.width, layout.height, m_Format, m_pMemoryPool->IsCompressible());
            NN_ABORT_UNLESS_GREATER_EQUAL(layout.size, nvnTextureBuilderGetStorageSize(&texBuilder));
            nvnTextureBuilderSetStorage(&texBuilder, m_pMemoryPool->GetNvnImportedMemoryPool(), layout.offset);
            NN_ABORT_UNLESS(nvnTextureInitialize(&texture, &texBuilder));
        }

        m_TextureStateList[index] = SharedTextureState_Initialized;
        NN_RESULT_SUCCESS;
    }

    void SharedTexturePool::DestroyTextureImpl(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_RANGE(index, 0, m_Layout.count);
        NN_SDK_REQUIRES_EQUAL(m_TextureStateList[index], SharedTextureState_Initialized);

        NN_VI_LOG_DEV("[share]Destroying texture %d\n", index);

        NVNtexture& texture = m_TextureList[index];
        nvnTextureFinalize(&texture);

        m_TextureStateList[index] = SharedTextureState_Invalid;
    }

    nn::Result SharedTexturePool::AcquireTexture(NVNtexture** pOutTexture, int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_RANGE(index, 0, m_Layout.count);

        if(m_TextureStateList[index] == SharedTextureState_Invalid)
        {
            NN_RESULT_DO(CreateTextureImpl(index));
        }
        NN_SDK_ASSERT_EQUAL(m_TextureStateList[index], SharedTextureState_Initialized);

        *pOutTexture = &m_TextureList[index];
        NN_RESULT_SUCCESS;
    }

    void SharedTexturePool::ReleaseTexture(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_UNUSED(index);
    }

    nn::Result SharedTexturePool::GetTexutureIndex(int* pOutValue, const NVNtexture* pTexture) const NN_NOEXCEPT
    {
        int index = -1;

        // TORIAEZU:
        for(int i = 0; i < TextureCountMax; i++)
        {
            if(pTexture == &m_TextureList[i])
            {
                index = i;
                break;
            }
        }

        NN_RESULT_THROW_UNLESS(index >= 0 && index < TextureCountMax, nn::vi::ResultNotFound());

        *pOutValue = index;
        NN_RESULT_SUCCESS;
    }

}}}

