﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/vi/fbshare/vi_SharedLayerWindow.h>
#include <nn/vi/fbshare/vi_SharedLayerWindowImpl.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi/vi_Result.h>

#include <atomic>
#include <nn/am/am_Shim.h>
#include <nn/am/am_Result.h>
#include "../vi_CommonUtility.h"
#include "../vi_Log.h"

namespace nn{ namespace vi{ namespace fbshare{

    static const int SharedLayerWindowSize = sizeof(SharedLayerWindow);
    static const int SharedLayerWindowImplSize = sizeof(SharedLayerWindowImpl);
    static const int SharedLayerWindowAlignment = NN_ALIGNOF(SharedLayerWindow);
    static const int SharedLayerWindowImplAlignment = NN_ALIGNOF(SharedLayerWindowImpl);

    NN_STATIC_ASSERT(SharedLayerWindowSize >= SharedLayerWindowImplSize);
    NN_STATIC_ASSERT(SharedLayerWindowAlignment <= SharedLayerWindowImplAlignment);

    namespace {
        SharedLayerWindowImpl* GetImpl(SharedLayerWindow* p)
        {
            return reinterpret_cast<SharedLayerWindowImpl*>(p);
        }

        const SharedLayerWindowImpl* GetImpl(const SharedLayerWindow* p)
        {
            return reinterpret_cast<const SharedLayerWindowImpl*>(p);
        }

        nn::Result GetSharedLayerHandle(SharedLayerHandle* pOutLayerHandle, SharedBufferHandle* pOutBufferHandle) NN_NOEXCEPT
        {
            SharedLayerHandle hLayer = {};
            SharedBufferHandle hBuffer = {};
            NN_RESULT_DO(am::GetSelfController()->GetSystemSharedLayerHandle(&hBuffer, &hLayer));
            *pOutLayerHandle = hLayer;
            *pOutBufferHandle = hBuffer;
            NN_RESULT_SUCCESS;
        }

        enum SupportLevel : int
        {
            SupportLevel_Unknown = 0,
            SupportLevel_NoSupport,
            SupportLevel_Supported,
        };

    }

    bool SharedLayerWindow::IsSupported() NN_NOEXCEPT
    {
        // 1 回取得したら Shim レイヤでキャッシュしておく。
        static std::atomic_int s_SupportLevel = {};

        bool isSupported = false;
        auto supportLevel = s_SupportLevel.load();
        if(supportLevel == SupportLevel_Unknown)
        {
            isSupported = am::GetSelfController()->IsSystemBufferSharingEnabled().IsSuccess();
            if(isSupported)
            {
                s_SupportLevel.store(SupportLevel_Supported);
            }
            else
            {
                s_SupportLevel.store(SupportLevel_NoSupport);
            }
        }
        else
        {
            switch(supportLevel)
            {
            case SupportLevel_Supported:
                isSupported = true;
                break;
            default:
                isSupported = false;
            }
        }

        return isSupported;
    }

    SharedLayerWindow::SharedLayerWindow() NN_NOEXCEPT
    {
        std::memset(m_Data, 0, sizeof(m_Data));
        new(m_Data) SharedLayerWindowImpl();
    }

    SharedLayerWindow::~SharedLayerWindow() NN_NOEXCEPT
    {
        GetImpl(this)->~SharedLayerWindowImpl();
        std::memset(m_Data, 0, sizeof(m_Data));
    }

    bool SharedLayerWindow::IsInitialized() const NN_NOEXCEPT
    {
        return GetImpl(this)->IsInitialized();
    }

    nn::Result SharedLayerWindow::Initialize(NVNdevice* pDevice, int bufferCount, NVNformat textureFormat) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES(textureFormat == NVN_FORMAT_RGBA8 || textureFormat == NVN_FORMAT_RGBA8_SRGB);
        auto pService = GetSystemService();
        NN_ABORT_UNLESS(pService != nullptr, "nn::vi is not initialized for system");

        // レイヤハンドルを取得
        SharedLayerHandle hLayer = {};
        SharedBufferHandle hBuffer = {};
        NN_RESULT_DO(GetSharedLayerHandle(&hLayer, &hBuffer));
        NN_VI_LOG_DEV("[share]hLayer=%llu,hBuffer=%llu\n", hLayer, hBuffer);

        NN_RESULT_DO(GetImpl(this)->Initialize(pService, pDevice, hBuffer, hLayer, bufferCount, textureFormat));

        NN_SDK_ASSERT(IsInitialized());
        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindow::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        GetImpl(this)->Finalize();

        NN_SDK_ASSERT(!IsInitialized());
    }

    int SharedLayerWindow::GetTextureWidth(NVNtexture* pTexture) const NN_NOEXCEPT
    {
        return GetImpl(this)->GetTextureWidth(pTexture);
    }

    int SharedLayerWindow::GetTextureHeight(NVNtexture* pTexture) const NN_NOEXCEPT
    {
        return GetImpl(this)->GetTextureHeight(pTexture);
    }

    //---------------------------------------------

    bool SharedLayerWindow::IsTexturePreAcquired() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return GetImpl(this)->IsTexturePreAcquired();
    }

    bool SharedLayerWindow::IsTextureAcquired() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return GetImpl(this)->IsTextureAcquired();
    }

    nn::Result SharedLayerWindow::PreAcquireTexture(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        NN_RESULT_DO(GetImpl(this)->PreAcquireTexture(timeout));

        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindow::CancelPreAcquiredTexture() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        GetImpl(this)->CancelPreAcquiredTexture();
    }

    nn::Result SharedLayerWindow::AcquireTexture(NVNsync* pOutTextureAvailableSync, int* pOutTextureIndex, NVNtexture** pOutTexture) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        auto timeout = nn::TimeSpan::FromSeconds(1);
        for(;;)
        {
            auto result = GetImpl(this)->PreAcquireTexture(timeout);
            if(result.IsSuccess())
            {
                break;
            }
            else if(nn::vi::ResultNotReady::Includes(result))
            {
                NN_VI_LOG_WARN("SharedLayerWindow::AcquireTexture() is waiting for texture acqirable. Use PreAcquireTexutre() to avoid infinite wait.\n");
            }
            else
            {
                return result;
            }
        }

        GetImpl(this)->AcquirePreAcquiredTexture(pOutTextureAvailableSync, pOutTextureIndex, pOutTexture);
        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindow::AcquirePreAcquiredTexture(NVNsync* pOutTextureAvailableSync, int* pOutTextureIndex, NVNtexture** pOutTexture) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        GetImpl(this)->AcquirePreAcquiredTexture(pOutTextureAvailableSync, pOutTextureIndex, pOutTexture);
    }

    void SharedLayerWindow::PresentTexture(NVNqueue* pQueue, int textureIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        GetImpl(this)->PresentTexture(pQueue, textureIndex);
    }

    void SharedLayerWindow::CancelTexture(int textureIndex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        GetImpl(this)->CancelTexture(textureIndex);
    }

    void SharedLayerWindow::SetSwapInterval(int value) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        GetImpl(this)->SetSwapInterval(value);
    }

    void SharedLayerWindow::SetCrop(int x, int y, int width, int height) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        GetImpl(this)->SetCrop(x, y, width, height);
    }

    //------------------------------

    nn::Result SharedLayerWindow::AcquireLastApplicationTexture(bool* pOutIsScreenShotPermitted, NVNtexture** pOutTexture) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_VI_LOG_DEV("[share]AcquireLastApplicationTexture\n");

        bool screenshotPermission = false;
        int32_t iBuf = -1;
        NN_RESULT_DO(am::GetDisplayController()->AcquireLastApplicationCaptureSharedBuffer(&screenshotPermission, &iBuf));

        if(iBuf < 0 || iBuf >= SharedBufferTextureCountMax)
        {
            *pOutIsScreenShotPermitted = false;
            *pOutTexture = nullptr;
            NN_RESULT_THROW(nn::am::ResultNotSupported());
        }

        NVNtexture* pTexture = nullptr;
        GetImpl(this)->GetSharedTexture(&pTexture, iBuf);

        *pOutIsScreenShotPermitted = screenshotPermission;
        *pOutTexture = pTexture;
        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindow::ReleaseLastApplicationTexture() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_VI_LOG_DEV("[share]ReleaseLastApplicationTexture\n");

        (void)am::GetDisplayController()->ReleaseLastApplicationCaptureSharedBuffer();
    }

    nn::Result SharedLayerWindow::AcquireLastForegroundTexture(bool* pOutIsScreenShotPermitted, NVNtexture** pOutTexture) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_VI_LOG_DEV("[share]AcquireLastForegroundTexture\n");

        bool screenshotPermission = false;
        int32_t iBuf = -1;
        NN_RESULT_DO(am::GetDisplayController()->AcquireLastForegroundCaptureSharedBuffer(&screenshotPermission, &iBuf));

        if(iBuf < 0 || iBuf >= SharedBufferTextureCountMax)
        {
            *pOutIsScreenShotPermitted = false;
            *pOutTexture = nullptr;
            NN_RESULT_THROW(nn::am::ResultNotSupported());
        }

        NVNtexture* pTexture = nullptr;
        GetImpl(this)->GetSharedTexture(&pTexture, iBuf);

        *pOutIsScreenShotPermitted = screenshotPermission;
        *pOutTexture = pTexture;
        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindow::ReleaseLastForegroundTexture() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_VI_LOG_DEV("[share]ReleaseLastForegroundTexture\n");

        (void)am::GetDisplayController()->ReleaseLastForegroundCaptureSharedBuffer();
    }

    nn::Result SharedLayerWindow::AcquireCallerAppletTexture(bool* pOutIsScreenShotPermitted, NVNtexture** pOutTexture) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_VI_LOG_DEV("[share]AcquireCallerAppletTexture\n");

        bool screenshotPermission = false;
        int32_t iBuf = -1;
        NN_RESULT_DO(am::GetDisplayController()->AcquireCallerAppletCaptureSharedBuffer(&screenshotPermission, &iBuf));

        if(iBuf < 0 || iBuf >= SharedBufferTextureCountMax)
        {
            *pOutIsScreenShotPermitted = false;
            *pOutTexture = nullptr;
            NN_RESULT_THROW(nn::am::ResultNotSupported());
        }

        NVNtexture* pTexture = nullptr;
        GetImpl(this)->GetSharedTexture(&pTexture, iBuf);

        *pOutIsScreenShotPermitted = screenshotPermission;
        *pOutTexture = pTexture;
        NN_RESULT_SUCCESS;
    }

    void SharedLayerWindow::ReleaseCallerAppletTexture() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_VI_LOG_DEV("[share]ReleaseCallerAppletTexture\n");

        (void)am::GetDisplayController()->ReleaseCallerAppletCaptureSharedBuffer();
    }

    nn::Result SharedLayerWindow::GetInternalTextureIndex(int* pOutValue, const NVNtexture* pTexture) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return GetImpl(this)->GetInternalTextureIndex(pOutValue, pTexture);
    }

}}}

