﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/os/os_Cache.h>
#include <nn/nn_Log.h>
#include <nn/vi/buffer/vi_Buffer.h>
#include <nn/vi/vi_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include "../vi_LayerUtility.h"
#include "../vi_CommonUtility.h"
#include "vi_BufferResource.h"
#include <nvrm_surface.h>
#include <nvgr.h>
#include <gui/SurfaceComposerClient.h>
#include "../vi_LayerHolder.h"

namespace nn{ namespace vi{ namespace buffer{

    void PrepareSurface(NvRmSurface* pOutSurface, const BufferInfo& info, bool isShadow) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(info.width, 0);
        NN_SDK_REQUIRES_GREATER(info.height, 0);

        NvColorFormat surfaceFormat;
        switch( info.format )
        {
        case PixelFormat_Rgba8888:
            surfaceFormat = NvColorFormat_A8B8G8R8;
            break;
        case PixelFormat_Rgbx8888:
            surfaceFormat = NvColorFormat_X8B8G8R8;
            break;
        case PixelFormat_Rgb888:
           surfaceFormat = NvColorFormat_R8_G8_B8;
           break;
        case PixelFormat_Rgb565:
            surfaceFormat = NvColorFormat_R5G6B5;
            break;
        case PixelFormat_Bgra8888:
            surfaceFormat = NvColorFormat_A8R8G8B8;
            break;
        case PixelFormat_Rgba5551:
            surfaceFormat = NvColorFormat_R5G5B5A1;
            break;
        case PixelFormat_Rgba4444:
            surfaceFormat = NvColorFormat_A4B4G4R4;
            break;
        //case PixelFormat_StandardRgba8888:
        //    surfaceFormat = NvColorFormat_R8G8B8A8_sRGB;
        //    break;
        //case PixelFormat_StandardRgbx8888:
        //    surfaceFormat = NvColorFormat_R8G8B8X8_sRGB;
        //    break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        NvRmSurfaceLayout layout = isShadow ?  NvRmSurfaceLayout_Pitch : NvRmSurfaceLayout_Blocklinear;
        NvU32 attrs[] = {
        NvRmSurfaceAttribute_Layout, layout,
        NvRmSurfaceAttribute_OptimizeForDisplay, 0, // Avoid large tile size (worse for GPU, better for display)
        NvRmSurfaceAttribute_None }; // marker for end of attributes

        NvRmSurfaceSetup(
                pOutSurface,
                static_cast<NvU32>(info.width),
                static_cast<NvU32>(info.height),
                surfaceFormat,
                attrs
                );
    }

    BufferQueue::BufferQueue() NN_NOEXCEPT
        : m_IsInitialized(false)
        , m_Storage{}
    {
        NN_STATIC_ASSERT(sizeof(m_Storage) >= sizeof(BufferResource));
        NN_STATIC_ASSERT(NN_ALIGNOF(m_Storage) >= NN_ALIGNOF(BufferResource));
    }

    BufferQueue::~BufferQueue() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
    }

    nn::Result BufferQueue::Initialize(Layer* pLayer, const BufferInfo& info) NN_NOEXCEPT
    {
        {
            std::lock_guard<nn::os::Mutex> lock(ObjectManager::GetLayerLock());
            NN_VI_REQUIRES_LAYER_VALID(pLayer);
        }
        NN_SDK_REQUIRES(!m_IsInitialized);

        ANativeWindow* window = GetLayerHolder(pLayer)->GetNativeWindow();

        native_window_api_connect(window, NATIVE_WINDOW_API_CPU);
        native_window_set_usage(window, GRALLOC_USAGE_HW_COMPOSER);
        native_window_set_buffers_user_dimensions(window, info.width, info.height);

        auto pData = new(&m_Storage[0]) BufferResource;
        int result;
        result = hw_get_module(GRALLOC_HARDWARE_MODULE_ID, &pData->pModule);
        NN_RESULT_THROW_UNLESS(0 == result, ResultOperationFailed());

        gralloc_open(pData->pModule, &pData->pAlloc);
        pData->pGrmodule = (const gralloc_module_t *) pData->pModule;

        NvRmOpenNew(&pData->rmDeviceHandle);

        for(int i=0; i<MaxBuffers; ++i)
        {
            pData->bufferHandle[i] = nullptr;
        }
        pData->pBuffer = nullptr;
        pData->pLayer = pLayer;
        pData->info   = info;
        m_IsInitialized = true;

        NN_RESULT_SUCCESS;
    }

    void BufferQueue::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);

        auto pData = GetData();
        gralloc_close(pData->pAlloc);
        NvRmClose(pData->rmDeviceHandle);


        for(int i=0; i<MaxBuffers; ++i)
        {
            pData->bufferHandle[i] = nullptr;
        }
        pData->pBuffer = nullptr;

        native_window_api_disconnect(GetLayerHolder(pData->pLayer)->GetNativeWindow(), NATIVE_WINDOW_API_CPU);
        m_IsInitialized = false;

    }


    size_t BufferQueue::GetRequiredMemorySize(const BufferInfo& info) NN_NOEXCEPT
    {

        NN_SDK_REQUIRES(m_IsInitialized);

        NvRmSurface surface;
        memset(&surface, 0, sizeof(surface));
        PrepareSurface(&surface, info, false);

        return NvRmSurfaceComputeSize(&surface);
    }

    size_t BufferQueue::GetRequiredMemoryAlignment(const BufferInfo& info) NN_NOEXCEPT
    {

        NN_SDK_REQUIRES(m_IsInitialized);

        NvRmSurface surface;
        memset(&surface, 0, sizeof(surface));
        PrepareSurface(&surface, info, false);
        return NvRmSurfaceComputeAlignment(NULL, &surface);
    }

    size_t BufferQueue::GetStride(const BufferInfo& info) NN_NOEXCEPT
    {

        NN_SDK_REQUIRES(m_IsInitialized);

        int byte;

        switch( info.format )
        {
            case PixelFormat_Rgba8888:
            case PixelFormat_Rgbx8888:
            case PixelFormat_Bgra8888:
                byte = 4;
                break;
            case PixelFormat_Rgb565:
            case PixelFormat_Rgba5551:
            case PixelFormat_Rgba4444:
                byte = 2;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
        }

        NvRmSurface surface;
        memset(&surface, 0, sizeof(surface));
        PrepareSurface(&surface, info, true);
        return (surface.Pitch) / byte;

    }

    nn::Result BufferQueue::SetScanBuffer(int bufferIndex, void* pBuffer, size_t size)  NN_NOEXCEPT
    {
            NN_SDK_REQUIRES(m_IsInitialized);

            int result;
            auto pData = GetData();
            NN_RESULT_THROW_UNLESS(bufferIndex <= (pData->info.bufferCount - 1), ResultOperationFailed());
            NN_RESULT_THROW_UNLESS(size >= GetRequiredMemorySize(pData->info), ResultOperationFailed());
            auto pHolder = GetLayerHolder(pData->pLayer);
            ANativeWindow *window = pHolder->GetNativeWindow();


            NvRmSurface surface;
            memset(&surface, 0, sizeof(surface));
            PrepareSurface(&surface, pData->info, false);
            size_t alignment = NvRmSurfaceComputeAlignment(NULL, &surface);

            NVRM_DEFINE_MEM_HANDLE_ATTR(attr);
            NVRM_MEM_HANDLE_SET_ATTR(attr,
                            alignment,
                            NvOsMemAttribute_WriteBack,
                            size,
                            NVRM_MEM_TAG_GRALLOC_MISC);
            NVRM_MEM_HANDLE_SET_KIND_ATTR(attr, surface.Kind);
            NVRM_MEM_HANDLE_SET_VA_ATTR(attr, (NvU64) pBuffer);

            result = NvRmMemHandleAllocAttr(pData->rmDeviceHandle, &attr, &pData->memHandle[bufferIndex]);
            NN_RESULT_THROW_UNLESS(0 == result, ResultOperationFailed());

            surface.hMem = pData->memHandle[bufferIndex];

            nvgr_import_nvrmsurface(&surface, 1, &pData->bufferHandle[bufferIndex]);

            native_window_set_preallocated_buffer(window,
                            bufferIndex,
                            pData->bufferHandle[bufferIndex],
                            pData->info.width,
                            pData->info.height,
                            nvgr_get_format(pData->bufferHandle[bufferIndex]),
                            nvgr_get_usage(pData->bufferHandle[bufferIndex]),
                            nvgr_get_stride(pData->bufferHandle[bufferIndex]));

            NN_RESULT_SUCCESS;
    }

    nn::Result BufferQueue::DequeueBuffer(BufferQueueHandle** pOutHandle) NN_NOEXCEPT
    {

        NN_SDK_REQUIRES(m_IsInitialized);

        ANativeWindowBuffer* buf;
        auto pData = GetData();
        auto pHolder = GetLayerHolder(pData->pLayer);
        ANativeWindow *window = pHolder->GetNativeWindow();
        int result;

        result = window->dequeueBuffer(window, &buf, &pData->syncFd);
        NN_RESULT_THROW_UNLESS(0 == result, ResultOperationFailed());

        *pOutHandle = reinterpret_cast<BufferQueueHandle*>(buf);

        result = pData->pGrmodule->lockAsync(pData->pGrmodule,
                buf->handle,
                GRALLOC_USAGE_SW_WRITE_OFTEN,
                0, 0,
                buf->width,
                buf->height,
                &pData->pBuffer,
                pData->syncFd);
        NN_RESULT_THROW_UNLESS(0 == result, ResultOperationFailed());


        NN_RESULT_SUCCESS;
    }


    nn::Result BufferQueue::QueueBuffer(BufferQueueHandle* pHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);

        int result;
        auto pData = GetData();
        auto pHolder = GetLayerHolder(pData->pLayer);

        ANativeWindow *window = pHolder->GetNativeWindow();
        ANativeWindowBuffer *buffer = reinterpret_cast<ANativeWindowBuffer*>(pHandle);

        result = pData->pGrmodule->unlockAsync(pData->pGrmodule,
                buffer->handle,
                &pData->syncFd);
        NN_RESULT_THROW_UNLESS(0 == result, ResultOperationFailed());

        result = window->queueBuffer(window, reinterpret_cast<ANativeWindowBuffer*>(pHandle), pData->syncFd);
        NN_RESULT_THROW_UNLESS(0 == result, ResultOperationFailed());

        NN_RESULT_SUCCESS;
    }

    nn::Result BufferQueue::GetScanBufferAddress(void** pOutAddr, BufferQueueHandle* pHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        auto pData = GetData();
        *pOutAddr = pData->pBuffer;
        NN_RESULT_SUCCESS;
    }


}}}

