﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/util/util_BitPack.h>

#include "usb_PdDriver.h"

namespace nn{
namespace usb{
namespace pd{
namespace driver {
namespace detail{

Result Driver::SleepAndWaitForMinimumAwake() NN_NOEXCEPT
{
    Result result;

    m_SleepRequestEvent.Clear();
    Sleep();
    // Hard Reset 発生時の PMIC の /ACOK で起床しないように待つ
    WaitForAcOkFromPdcResetBeforeSleep();

    // AlertThread 停止（ReceiveVdmThread も連動）
    m_SleepAlertReplyEvent.Clear();
    m_SleepAlertRequestEvent.Signal();
    m_SleepAlertReplyEvent.Wait();
    m_SleepAlertReplyEvent.Clear();
    m_SleepReplyEvent.Signal();

    // MinimumAwakeRequest が来るまで ExecuteThread 停止
    m_MinimumAwakeRequestEvent.Wait();
    m_MinimumAwakeRequestEvent.Clear();
    // 終了要求があったらすぐに抜ける
    if ( !m_FinalizeExecuteEvent.TryWait() )
    {
        // AlertThread 再開（ReceiveVdmThread も連動）
        m_AwakeAlertReplyEvent.Clear();
        m_AwakeAlertRequestEvent.Signal();
        m_AwakeAlertReplyEvent.Wait();
        m_AwakeAlertReplyEvent.Clear();
        // 抜去時は出来るだけ早くステータス更新
        m_AwakeExecuteRequest = false;
        if ( m_pAlertEvent[AlertStatus::PlugInsertRemove::Pos]->TryWait() )
        {
            if ( m_IsActive )
            {
                // 挿入されていても挿抜アラート発生時は一旦抜去ステータスに
                ClearStatusBase();
                UpdateStatus( false, Notice::ActiveNotice::Mask );
            }
            // ExecuteThread 処理要求
            m_AwakeExecuteRequest = true;
        }
        MinimumAwake();
    }
    m_MinimumAwakeReplyEvent.Signal();

    return ResultSuccess();
}

Result Driver::WaitForAcOkFromPdcResetBeforeSleep() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    // PlugRemove 前に Hard Reset が発生する場合の誤検出防止
    os::SleepThread( PdDebounceTimeSpan );
    // ExecuteThread 停止中の Hard Reset 発生へ対応
    result = ReceiveCommand1( &m_Status1, L1CommandSize_Status1, L1Command_Status1 );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    UpdateAlert();
    if ( m_Status1.Get<Status1::PlugInsertion>() && m_pAlertEvent[AlertStatus::HardReset::Pos]->TryWait() )
    {
        StartPdcResetToAcOkTimer( HardResetToAcOkTimeSpan );
    }
    // Hard Reset 発生時の PMIC の /ACOK で起床しないように待つ
    WaitForAcOkFromPdcReset();

    return result;
}

Result Driver::Sleep() NN_NOEXCEPT
{
    Result result;

    if ( m_IsSleeping )
    {
        return ResultSuccess();
    }
    m_IsSleeping = true;

    USBPD_DBG_LOG("Sleep\n");

    // スリープ時は強制的に SpdSrc OFF にする
    // Source 時の SpdSrc ON → OFF で PlugRemoved アラート発生（Status1::PlugInsertion==false）して
    // OTG モードであれば 5V 出力を止める処理が走る
    SetSpdSrc( ControllerConfiguration1SpdSrc_Off );

    // DisplayPort アラート無効化
    DisplayPortAlertEnable dpAlertEnable;
    dpAlertEnable.Clear();
    result = SendCommand1( &dpAlertEnable, L1CommandSize_DisplayPortAlertEnable, L1Command_DisplayPortAlertEnable );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("DisplayPortAlertEnable = %04x\n", dpAlertEnable);

    usb::pd::Status status;
    GetStatusCore( &status );

    if ( IsCradleFamily( status.GetDeviceType() ) )
    {
        if ( m_CradleMcuFwVersion < CradleMcuFwRevisionFinalDp1 )
        {
            // クレードル DP1 以前は非対応
            USBPD_WARNING("This cradle doesn't support to sleep!\n");
            return ResultSuccess();
        }
        // クレードルをミニホストへ遷移
        USBPD_DBG_LOG("USB HUB Disable VDM\n");
        result = SendAndReceiveNxVdmWithRequest( &m_CradleUsbHubState, sizeof(m_CradleUsbHubState), nullptr, VdmSet_UsbHubDisableRequest, VdmCommand_UsbHubVBusDetect );
        if ( !(result <= ResultVdmError()) )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
        // クレードルのスリープ
        if ( result.IsSuccess() )
        {
            USBPD_DBG_LOG("Sleep Cradle VDM\n");
            result = SendAndReceiveNxVdmWithRequest( nullptr, 0, nullptr, VdmSet_SleepRequest, VdmCommand_McuSleepEnable );
            if ( !(result <= ResultVdmError()) )
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
    }

    // DisplayPortStatus チェック
    result = ReceiveCommand1( &m_DisplayPortStatus, L1CommandSize_DisplayPortStatus, L1Command_DisplayPortStatus );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("DisplayPortStatus = %04x\n", m_DisplayPortStatus);

#ifdef ENABLE_PDC_HARD_RESET_IN_SLEEP_TEST
    m_pAlertEvent[AlertStatus::HardReset::Pos]->Signal();
#endif

    return ResultSuccess();
}

Result Driver::MinimumAwake() NN_NOEXCEPT
{
    Result result;

    if ( !m_IsSleeping )
    {
        return ResultSuccess();
    }
    m_IsSleeping = false;

    USBPD_DBG_LOG("MinimumAwake\n");

    // スリープ時の SpdSrc OFF から ON への復帰は psm からの EnablePowerRequest 後に行う

    // DisplayPortStatus チェック
    result = ReceiveCommand1( &m_DisplayPortStatus, L1CommandSize_DisplayPortStatus, L1Command_DisplayPortStatus );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("DisplayPortStatus = %04x\n", m_DisplayPortStatus);

    // DisplayPort アラート有効化
    result = SendCommand1( &m_DisplayPortAlertEnable, L1CommandSize_DisplayPortAlertEnable, L1Command_DisplayPortAlertEnable );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    USBPD_DBG_LOG("DisplayPortAlertEnable = %04x\n", m_DisplayPortAlertEnable);

    usb::pd::Status status;
    GetStatusCore( &status );
    if ( IsCradleFamily( status.GetDeviceType() ) )
    {
        if ( m_CradleMcuFwVersion < CradleMcuFwRevisionFinalDp1 )
        {
            // クレードル DP1 以前は非対応
            return ResultSuccess();
        }
        // クレードルのミニホスト解除タイマースタート
        StartCradleUsbHubTimer();
    }

    return ResultSuccess();
}

Result Driver::FullAwake() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    USBPD_DBG_LOG("FullAwake\n");

    m_FullAwakeRequestEvent.Clear();

    usb::pd::Status status;
    GetStatusCore( &status );
    if ( IsCradleFamily( status.GetDeviceType() ) )
    {
        if ( m_CradleMcuFwVersion < CradleMcuFwRevisionFinalDp1 )
        {
            // クレードル DP1 以前は非対応
            result = ResultSuccess();
        }
        else if ( m_CradleAwakeSuspended )
        {
            USBPD_WARNING("Cradle awake was canceled!\n");
        }
        else
        {
            // クレードルのスリープ解除（SIGLO-50877）
            // （ドックイン中に Hard Reset 発生時はスリープ解除しない）
            USBPD_DBG_LOG("Awake Cradle VDM\n");
            result = SendAndReceiveNxVdmWithRequest( nullptr, 0, nullptr, VdmSet_AwakeRequest, VdmCommand_McuSleepEnable );
            if ( result <= ResultVdmError() )
            {
                result = ResultSuccess();
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }
    }

    m_FullAwakeReplyEvent.Signal();

    return ResultSuccess();
}

void Driver::SignalFullAwakeReadyEvent( int alertIndex ) NN_NOEXCEPT
{
    if ( alertIndex == AlertStatus::PlugInsertRemove::Pos )
    {
        if ( m_AwakeExecuteRequest )
        {
            // スリープ中のプラグ挿抜に対する処理の完了通知
            USBPD_DBG_LOG("Awake execution is completed.\n");
            m_AwakeExecuteCompleteEvent.Signal();
        }
    }
}

Result Driver::SleepRequest() NN_NOEXCEPT
{
    USBPD_DBG_LOG("SleepRequest\n");

    if ( m_ResetResult.IsFailure() )
    {
        return ResultSuccess();
    }

    return PmStateRequestCore( psc::PmState_SleepReady );
}

Result Driver::MinimumAwakeRequest() NN_NOEXCEPT
{
    USBPD_DBG_LOG("MinimumAwakeRequest\n");

    if ( m_ResetResult.IsFailure() )
    {
        return ResultSuccess();
    }

    m_AwakeExecuteCompleteEvent.Clear();
    return PmStateRequestCore( psc::PmState_MinimumAwake );
}

Result Driver::FullAwakeRequest() NN_NOEXCEPT
{
    USBPD_DBG_LOG("FullAwakeRequest\n");

    if ( m_ResetResult.IsFailure() )
    {
        return ResultSuccess();
    }

    if ( m_AwakeExecuteRequest )
    {
        // スリープ中のプラグ挿抜に対する処理の完了待ち
        m_AwakeExecuteCompleteEvent.Wait();
        m_AwakeExecuteRequest = false;
    }
    else
    {
        // スリープ中のプラグ挿抜が無ければクレードルをスリープ解除
        return PmStateRequestCore( psc::PmState_FullAwake );
    }
    return ResultSuccess();
}

Result Driver::PmStateRequestCore( nn::psc::PmState pmState ) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_SleepRequestMutex);

    if ( m_PmState == pmState )
    {
        return ResultSuccess();
    }

    os::Event* pRequest = nullptr;
    os::Event* pReply = nullptr;
    switch ( pmState )
    {
    case psc::PmState_FullAwake:
        pRequest = &m_FullAwakeRequestEvent;
        pReply = &m_FullAwakeReplyEvent;
#ifdef ENABLE_PDC_HARD_RESET_INTO_FULL_AWAKE_TEST
    m_pAlertEvent[AlertStatus::HardReset::Pos]->Signal();
#endif
        break;

    case psc::PmState_MinimumAwake:
        if ( m_PmState != psc::PmState_SleepReady )
        {
            m_PmState = pmState;
#ifdef ENABLE_PDC_HARD_RESET_INTO_MINIMUM_AWAKE_TEST
    m_pAlertEvent[AlertStatus::HardReset::Pos]->Signal();
#endif
            return ResultSuccess();
        }
        pRequest = &m_MinimumAwakeRequestEvent;
        pReply = &m_MinimumAwakeReplyEvent;
#ifdef ENABLE_PDC_HARD_RESET_OUT_OF_MINIMUM_AWAKE_TEST
    m_pAlertEvent[AlertStatus::HardReset::Pos]->Signal();
#endif
        break;

    case psc::PmState_SleepReady:
#ifdef ENABLE_PDC_HARD_RESET_INTO_SLEEP_READY_TEST
    m_pAlertEvent[AlertStatus::HardReset::Pos]->Signal();
#endif
        pRequest = &m_SleepRequestEvent;
        pReply = &m_SleepReplyEvent;
        break;

    default:
        USBPD_ABORT("Unexpected PmState!");
        break;
    }
    m_PmState = pmState;

    pRequest->Signal();
    pReply->Wait();
    pReply->Clear();
    return ResultSuccess();
}

} // detail
} // driver
} // pd
} // usb
} // nn
