﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstring>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/util/util_BitPack.h>
#include <nn/gpio/gpio.h>
#include <nn/gpio/gpio_PadAccessorDev.h>
#include <nn/i2c/i2c.h>
#include <nn/i2c/i2c_BusDev.h>

#include "usb_PdDriver.h"

//#define DISABLE_COMMAND2_TIMEOUT

namespace nn{
namespace usb{
namespace pd{
namespace driver {
namespace detail{

const VdmCommandType g_VdmCommandTypeList[VdmCommand_Max + 1] =
{
    VdmCommandType_None,
    VdmCommandType_Led,
    VdmCommandType_LedReply,
    VdmCommandType_None,
    VdmCommandType_None,
    VdmCommandType_None,
    VdmCommandType_Dp2HdmiFwVersion,
    VdmCommandType_Dp2HdmiFwVersionReply,
    VdmCommandType_None,
    VdmCommandType_None,
    VdmCommandType_None,
    VdmCommandType_PdcHFwVersion,
    VdmCommandType_PdcHFwVersionReply,
    VdmCommandType_None,
    VdmCommandType_None,
    VdmCommandType_None,
    VdmCommandType_PdcAFwVersion,
    VdmCommandType_PdcAFwVersionReply,
    VdmCommandType_None,
    VdmCommandType_None,
    VdmCommandType_None,
    VdmCommandType_DeviceErrorNotice,
    VdmCommandType_DeviceState,
    VdmCommandType_DeviceStateReply,
    VdmCommandType_McuFwVersion,
    VdmCommandType_McuFwVersionReply,
    VdmCommandType_McuFwUpdate,
    VdmCommandType_McuFwUpdateReply,
    VdmCommandType_McuSleepEnable,
    VdmCommandType_McuSleepEnableReply,
    VdmCommandType_UsbHubReset,
    VdmCommandType_UsbHubResetReply,
    VdmCommandType_UsbHubVBusDetect,
    VdmCommandType_UsbHubVBusDetectReply
};
const VdmCommandLength g_VdmCommandLength[VdmCommand_Max + 1] =
{
    VdmCommandLength_None,
    VdmCommandLength_Led,
    VdmCommandLength_LedReply,
    VdmCommandLength_None,
    VdmCommandLength_None,
    VdmCommandLength_None,
    VdmCommandLength_Dp2HdmiFwVersion,
    VdmCommandLength_Dp2HdmiFwVersionReply,
    VdmCommandLength_None,
    VdmCommandLength_None,
    VdmCommandLength_None,
    VdmCommandLength_PdcHFwVersion,
    VdmCommandLength_PdcHFwVersionReply,
    VdmCommandLength_None,
    VdmCommandLength_None,
    VdmCommandLength_None,
    VdmCommandLength_PdcAFwVersion,
    VdmCommandLength_PdcAFwVersionReply,
    VdmCommandLength_None,
    VdmCommandLength_None,
    VdmCommandLength_None,
    VdmCommandLength_DeviceErrorNotice,
    VdmCommandLength_DeviceState,
    VdmCommandLength_DeviceStateReply,
    VdmCommandLength_McuFwVersion,
    VdmCommandLength_McuFwVersionReply,
    VdmCommandLength_McuFwUpdate,
    VdmCommandLength_McuFwUpdateReply,
    VdmCommandLength_McuSleepEnable,
    VdmCommandLength_McuSleepEnableReply,
    VdmCommandLength_UsbHubReset,
    VdmCommandLength_UsbHubResetReply,
    VdmCommandLength_UsbHubVBusDetect,
    VdmCommandLength_UsbHubVBusDetectReply
};

Result Driver::SendStandardVdm( Status1* pStatus1, VdmHeader* pHeader, VdmHeaderCommand command ) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    VdmHeader vdm;

    m_IsVdmCanceled = false;
    m_StandardVdmEvent.Clear();
    result = AcceptIncomingVdm( nullptr );
    if ( result <= ResultPdcCommandGiveUp() )
    {
         return ResultVdmReplyGiveUp();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    vdm.Clear();
    vdm.Set<VdmHeader::SvId>( command == VdmHeaderCommand_DiscoverIdentity ? VdmStanderdId : VdmNintendoId );
    vdm.Set<VdmHeader::VdmType>( VdmType_StructuredVdm );
    vdm.Set<VdmHeader::VdmVersion>( VdmVersion_1_0 );
    vdm.Set<VdmHeader::ObjectPosition>( command == VdmHeaderCommand_DiscoverIdentity ? 0 : 1 );
    vdm.Set<VdmHeader::CommandType>( VdmCommandType_Initiator );
    vdm.Set<VdmHeader::HeaderCommand>( command );
    USBPD_DBG_LOG("Send StandardVDM = %08x\n", vdm);
    *pHeader = vdm;
    result = SendCommand1( &vdm, sizeof(vdm), L1Command_OutgoingVdm );
    if ( result <= ResultPdcCommandGiveUp() )
    {
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = SendAndWaitCommand2( pStatus1, AlertStatus::CommandCompleteOrAbort::Mask, L2Command_SendVdmWithSop );
    if ( result <= ResultPdcCommandGiveUp() )
    {
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

Result Driver::ReceiveStadardVdm( Status2* pStatus2, void* pDst, size_t size ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if ( !m_StandardVdmEvent.TimedWait( VdmReplyTimeoutSpan ) || m_IsVdmCanceled )
    {
        USBPD_WARNING("ReceiveStadardVDM is timeout!\n");
        memset( pDst, 0, size );
        result = ResultVdmReplyGiveUp();
    }
    else
    {
        size_t sz = size;
        if ( sz > VdmCommandLength_Max * 4 )
        {
            sz = VdmCommandLength_Max * 4;
        }
        if ( pDst )
        {
            std::lock_guard<nn::os::Mutex> lock( m_ReceiveVdmMutex );
            memcpy( pDst, &m_StandardVdm, sz );
        }
    }
    m_StandardVdmEvent.Clear();

    if ( pStatus2 )
    {
        auto r = ReceiveCommand1( pStatus2, L1CommandSize_Status2, L1Command_Status2 );
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
        USBPD_DBG_LOG("Status2 = %04x\n", *pStatus2);
    }
    {
        auto r = AcceptIncomingVdm( nullptr );
        if ( r <= ResultPdcCommandGiveUp() )
        {
            return ResultVdmReplyGiveUp();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }

    return result;
}

Result Driver::SendNxVdm( const void* pSrc, VdmCommand command ) NN_NOEXCEPT
{
    return SendNxVdmWithRequest( nullptr, nullptr, pSrc, VdmSet_ReadRequest, command );
}

Result Driver::SendNxVdmWithRequest( Status1* pStatus1, VdmHeader* pHeader, const void* pSrc, VdmSet rw, VdmCommand command ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    const VdmUnit* p = static_cast<const VdmUnit*>(pSrc);
    VdmUnit vdm[VdmCommandLength_Max];
    vdm[0].Clear();
    vdm[1].Clear();
    vdm[2].Clear();

    vdm[0].Set<VdmHeader::SvId>( VdmNintendoId );
    vdm[0].Set<VdmHeader::VdmType>( VdmType_UnstructuredVdm );
    vdm[0].Set<VdmHeader::VdmVersion>( VdmVersion_1_0 );
    vdm[0].Set<VdmHeader::ObjectPosition>( 0 );
    vdm[0].Set<VdmHeader::CommandType>( g_VdmCommandTypeList[command] );
    vdm[0].Set<VdmHeader::HeaderCommand>( VdmHeaderCommand_None );
    if ( pHeader )
    {
        pHeader->storage = vdm[0].storage;
    }
    vdm[1].Set<VdmCommonVdo1::Command>( command );
    vdm[1].Set<VdmCommonVdo1::Direction>( VdmDirection_FromSwitch );
    vdm[1].Set<VdmCommonVdo1::Set>( rw ? true : false );
    if ( p )
    {
        vdm[2] = *p;
    }
//    USBPD_DBG_LOG("Send VDM: %08x, %08x, %08x\n", vdm[0], vdm[1], vdm[2]);
    result = SendCommand1( &vdm, g_VdmCommandLength[command] * sizeof(vdm[0]), L1Command_OutgoingVdm );
    if ( result <= ResultPdcCommandGiveUp() )
    {
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = SendAndWaitCommand2( pStatus1, AlertStatus::CommandCompleteOrAbort::Mask, L2Command_SendVdmWithSop );
    if ( result <= ResultPdcCommandGiveUp() )
    {
        return result;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    return result;
}

Result Driver::SendAndReceiveNxVdm( void* pDst, size_t dstSize, const void* pSrc, VdmCommand command ) NN_NOEXCEPT
{
    return SendAndReceiveNxVdmWithRequest( pDst, dstSize, pSrc, VdmSet_ReadRequest, command );
}

Result Driver::SendAndReceiveNxVdmWithRequest( void* pDst, size_t dstSize, const void* pSrc, VdmSet rw, VdmCommand command ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    os::Mutex* m = nullptr;
    int c = command;
    if ( g_VdmCommandTypeList[c + 1] == VdmCommandType_Ack )
    {
        m = &m_ReplyCommonVdmMutex;
    }
    if ( !m )
    {
        return ResultInvalidAddress();
    }
    std::lock_guard<nn::os::Mutex> lock( *m );
    // VDM 処理はここで排他制御される

    m_IsVdmCanceled = false;
    m_ReplyCommonVdmEvent.Clear();
    result = AcceptIncomingVdm( nullptr );
    if ( result <= ResultPdcCommandGiveUp() )
    {
         return ResultVdmReplyGiveUp();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    Status1 status1;
    VdmHeader header;
    result = SendNxVdmWithRequest( &status1, &header, pSrc, rw, command );
    if ( result <= ResultPdcCommandGiveUp() ||
         status1.Get<Status1::LastCommand>() != Status1LastCommand_Completed
       )
    {
        if ( pDst )
        {
            memset( pDst, 0, dstSize );
        }
        return ResultVdmReplyGiveUp();
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    header.Set<VdmHeader::CommandType>( VdmCommandType_Ack );
    result = ReceiveNxVdm( pDst, dstSize, &header, static_cast<VdmCommand>(c + 1) );

    return result;
}

Result Driver::ReceiveNxVdm( void* pDst, size_t size, const VdmHeader* pHeader, VdmCommand command ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    const VdmUnit* v = nullptr;
    os::Event* e = nullptr;
    VdmNoticeType n = SearchVdmNoticeType( command );
    if  ( n != VdmNoticeType_None )
    {
        v = m_NoticeVdm[n];
        e = m_pNoticeVdmEvent[n];
    }
    else if ( g_VdmCommandTypeList[command] == VdmCommandType_Ack )
    {
        v = m_ReplyCommonVdm;
        e = &m_ReplyCommonVdmEvent;
    }
    if ( !v )
    {
        return ResultInvalidAddress();
    }

    if ( !e->TimedWait( VdmReplyTimeoutSpan ) ||
         ( e == &m_ReplyCommonVdmEvent && m_IsVdmCanceled )
       )
    {
        USBPD_WARNING("ReceiveNxVdm is timeout!\n");
        if ( pDst )
        {
            memset( pDst, 0, size );
        }
        result = ResultVdmReplyGiveUp();
    }
    e->Clear();

    if ( result.IsSuccess() )
    {
        size_t sz = size;
        if ( sz > 4 )
        {
            sz = 4;
        }
        if ( pDst )
        {
            std::lock_guard<nn::os::Mutex> lock( m_ReceiveVdmMutex );
            if ( v[0].storage == pHeader->storage &&
                 VdmCommand( v[1].Get<VdmCommonVdo1::Command>() ) == command &&
                 v[1].Get<VdmCommonVdo1::Direction>() == VdmDirection_FromCradle
               )
            {
                int vdoOffset = 2;
                // UsbHubVBusDetect 成功判定のため VDO1 を取得
                if ( command == VdmCommand_UsbHubVBusDetectReply )
                {
                    vdoOffset = 1;
                }
                memcpy( pDst, &v[vdoOffset], sz );
            }
            else
            {
                // 期待した VDM でない場合にエラーにする
                // （ACK フラグと Standard VDM チェック通過後なので VDM データ化け以外では通らないはず）
                if ( pDst )
                {
                    memset( pDst, 0, size );
                }
                result = ResultVdmInvalidReply();
            }
        }
    }

    {
        Status2 status2;
        auto r = ReceiveCommand1( &status2, L1CommandSize_Status2, L1Command_Status2 );
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
        USBPD_DBG_LOG("Status2 = %04x\n", status2);
    }
    {
        auto r = AcceptIncomingVdm( nullptr );
        if ( r <= ResultPdcCommandGiveUp() )
        {
            return ResultVdmReplyGiveUp();
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }

    return result;
}

Result Driver::ReceiveNxVdmCore( void* pDst, size_t size ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if ( !pDst )
    {
        return ResultInvalidAddress();
    }

    VdmUnit vdm[VdmCommandLength_Max];
    memset( &vdm, 0, sizeof(vdm) );
    result = ReceiveCommand1( &vdm, sizeof(vdm), L1Command_IncomingVdm );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
//    USBPD_DBG_LOG("Recv VDM: %08x, %08x, %08x\n", vdm[0], vdm[1], vdm[2]);

    size_t sz = size;
    if ( sz > VdmCommandLength_Max * 4 )
    {
        sz = VdmCommandLength_Max * 4;
    }
    memcpy( pDst, &vdm, sz );

    return result;
}

Result Driver::AcceptIncomingVdm( Status1* pStatus1 ) NN_NOEXCEPT
{
    auto result = SendAndWaitCommand2( pStatus1, AlertStatus::CommandCompleteOrAbort::Mask, L2Command_AcceptIncomingVdm );

    return result;
}

Result Driver::SendCommand1( const void* pSrc, size_t size, L1Command command1 ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if ( m_ResetResult.IsFailure() )
    {
        return ResultSuccess();
    }
    if ( size > L1CommandSize_Max )
    {
        size = L1CommandSize_Max;
    }
    for ( int i = 0; i < CommnadRetryMax; i++ )
    {
        std::lock_guard<nn::os::Mutex> lock( m_I2cMutex );

        result = SendCommand1Core( pSrc, size, command1 );
        if ( result.IsFailure() )
        {
            return result;
        }
        uint8_t receiveBuf[L1CommandSize_Max];
        size_t receivedSize = 0;
        result = ReceiveCommand1( receiveBuf, size, command1, &receivedSize );
        if ( result.IsFailure() )
        {
            return result;
        }
        if ( receivedSize == size && !memcmp( receiveBuf, pSrc, size ) )
        {
            return ResultSuccess();
        }
        USBPD_WARNING("L1Command %04x verify error!\n", command1);
    }

    return ResultPdcCommandVerifyError();
}

Result Driver::SendCommand1Core( const void* pSrc, size_t size, L1Command command1 ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if ( m_ResetResult.IsFailure() )
    {
        return ResultSuccess();
    }
    std::lock_guard<nn::os::Mutex> lock( m_I2cMutex );

    size_t sz = size;
    if ( sz > L1CommandSize_Max )
    {
        sz = L1CommandSize_Max;
    }
    void* p = &m_CommandBuf[1];
    if ( sz >= I2cBlockSwitchSize )
    {
        p = &m_CommandBuf[2];
        m_CommandBuf[1] = sz;
    }
    memcpy( p, pSrc, sz );
    m_CommandBuf[0] = command1;
    sz++;
    if ( p == &m_CommandBuf[2] )
    {
        sz++;
    }
    result = i2c::Send( m_I2cSession, m_CommandBuf, sz, I2cSendConditions );
//    USBPD_DBG_LOG("Send: %02x : Description = %d\n", command1, result.GetDescription());

    return result;
}

Result Driver::SendCommand2( L2Command command2 ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if ( m_ResetResult.IsFailure() )
    {
        return ResultSuccess();
    }
    result = SendCommand2Core( command2 );
    if ( result.IsFailure() )
    {
        return result;
    }
    // Status1 が上書きされないようにローカル変数にする
    Status1 status1;
    result = ReceiveCommand1( &status1, L1CommandSize_Status1, L1Command_Status1 );
    if ( result.IsFailure() )
    {
        return result;
    }
    bool isCommandReceived = false;
    {
        // Alert 処理が完了するまで m_pAlertEvent の参照を待つ
        UpdateAlert();
        // CommandState が Progress でなくなるとほぼ同時に CommandComplete アラート発生
        if ( status1.Get<Status1::CommandState>() == Status1CommandState_Progress ||
             m_pAlertEvent[AlertStatus::CommandCompleteOrAbort::Pos]->TryWait()
          )
        {
            isCommandReceived = true;
        }
    }
    if ( !isCommandReceived )
    {
        USBPD_WARNING("Status1 = %04x\n", status1);
        return ResultPdcCommandNoResponse();
    }
    return ResultSuccess();
}

Result Driver::SendCommand2Core( L2Command command2 ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if ( m_ResetResult.IsFailure() )
    {
        return ResultSuccess();
    }
    std::lock_guard<nn::os::Mutex> lock( m_I2cMutex );

    m_CommandBuf[0] = L1Command_L2Command;
    m_CommandBuf[1] = command2 & 0xFF;
    m_CommandBuf[2] = (command2 >> 8) & 0xFF;
    result = i2c::Send( m_I2cSession, m_CommandBuf, L1CommandSize_L2Command + 1, I2cSendConditions );
//    USBPD_DBG_LOG("Send: %04x : Description = %d\n", command2, result.GetDescription());

    return result;
}

Result Driver::SendAndWaitCommand2( Status1* pStatus1, int alertMask, L2Command command2 ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    std::lock_guard<nn::os::Mutex> lock( m_L2CommandMutex );
    // 現状で必要なケースは m_ReplyCommonVdmMutex で排他されていて不要だが念のため

    if ( pStatus1 == &m_Status1 )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultInvalidAddress());
    }
    if ( pStatus1 )
    {
        pStatus1->Clear();
    }
    if ( m_ResetResult.IsFailure() )
    {
        return ResultSuccess();
    }

    for ( int i = 0; i < CommnadRetryMax; ++i )
    {
        if ( i > 0 )
        {
            USBPD_WARNING("L2Command %04x retry!!\n", command2);
        }
        for ( int ii = 0; ii < AlertNum; ++ii )
        {
            if ( (alertMask >> ii) & 0x1 )
            {
                m_pAlertEvent[ii]->Clear();
            }
        }
        result = SendCommand2( command2 );
        if ( result.IsSuccess() )
        {
            bool isRetryRequest = false;
            for ( int ii = 0; ii < AlertNum; ++ii )
            {
                if ( (alertMask >> ii) & 0x1 )
                {
#ifdef DISABLE_COMMAND2_TIMEOUT
                    m_pAlertEvent[ii]->Wait();
#else
                    bool timeout = !m_pAlertEvent[ii]->TimedWait( Command2TimeoutSpan );
                    if ( timeout )
                    {
                        USBPD_WARNING("L2Command %04x is timeout!\n", command2);
                        return ResultPdcCommandTimeout();
                    }
#endif // DISABLE_COMMAND2_TIMEOUT
                    m_pAlertEvent[ii]->Clear();
                    result = ReceiveCommand1( &m_Status1, L1CommandSize_Status1, L1Command_Status1 );
                    USBPD_DBG_LOG("Status1 = %04x\n", m_Status1);
                    if ( pStatus1 )
                    {
                        *pStatus1 = m_Status1;
                    }
                    if ( !result.IsSuccess() )
                    {
                         return result;
                    }
                    // Command Complete アラート時に Command Invalid だったらリトライ
                    if ( ii == AlertStatus::CommandCompleteOrAbort::Pos &&
                         m_Status1.Get<Status1::LastCommand>() == Status1LastCommand_Invalid
                       )
                    {
                        isRetryRequest = true;
                        break;
                    }
                }
            }
            if ( !isRetryRequest )
            {
                return ResultSuccess();
            }
        }
    }
    return ResultPdcCommandOverRetry();
}

Result Driver::ReceiveCommand1( void* pDst, size_t dstSize, L1Command command1, size_t* pReceivedSize ) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    std::lock_guard<nn::os::Mutex> lock( m_I2cMutex );
    // pDst への書き込みは排他制御される

    void* p = pDst;
    size_t sz = dstSize;
    if ( sz > L1CommandSize_Max )
    {
        sz = L1CommandSize_Max;
    }
    if ( pReceivedSize )
    {
        *pReceivedSize = sz;
    }
    if ( sz >= I2cBlockSwitchSize )
    {
        p = m_CommandBuf;
        sz++;
    }
    if ( m_ResetResult.IsFailure() )
    {
        std::memset( p, 0, sz );
    }
    else
    {
        // コマンドリスト実行
        uint8_t commandList[i2c::CommandListLengthCountMax];
        i2c::CommandListFormatter  commandListFormatter( commandList, sizeof(commandList) );
        std::memset( &commandList, 0, sizeof(commandList) );
        commandListFormatter.EnqueueSendCommand( I2cReceiveCommandConditions, reinterpret_cast<Bit8*>(&command1), 1 );
        commandListFormatter.EnqueueReceiveCommand( I2cReceiveDataConditions, sz );
        result = i2c::ExecuteCommandList( p, sz, m_I2cSession, commandList, commandListFormatter.GetCurrentLength() );
    }
    if ( p == m_CommandBuf )
    {
//        USBPD_DBG_LOG("ReceiveCommand1: %04x\n", *reinterpret_cast<Bit32*>(p));
        sz--;
        memcpy( pDst, &m_CommandBuf[1], sz );
        if ( pReceivedSize )
        {
#ifdef ENABLE_PDC_COMMAND1_SIZE_WORKAROUND_TEST
            *pReceivedSize = 4;
#else
            *pReceivedSize = m_CommandBuf[0];
#endif
        }
    }

    return result;
}

void Driver::WaitDeviceId() NN_NOEXCEPT
{
    if ( m_ResetResult.IsFailure() )
    {
        return;
    }

    // PD コン FW ブート待ち（ブート前の場合 DeviceId=0x0000）
    const TimeSpan interval = TimeSpan::FromMilliSeconds( 10 );
    os::Tick start = os::GetSystemTick();
    int count = 0;
    while (1)
    {
        auto result = ReceiveCommand1( &m_DeviceId, L1CommandSize_DeviceId, L1Command_DeviceId );
#ifdef ENABLE_PDC_FW_BOOT_TIMEOUT_TEST
        m_DeviceId = 0;
#endif
        if ( m_DeviceId || !(count % (1000 / interval.GetMilliSeconds())) )
        {
            USBPD_DBG_LOG("DeviceID = %04x\n", m_DeviceId);
        }
        if ( result.IsFailure() )
        {
            m_ResetResult = result;
            break;
        }
        if ( (nn::os::GetSystemTick() - start).ToTimeSpan() > FirmwareBootTimeoutSpan )
        {
            m_ResetResult = ResultPdcFwBootTimeout();
            break;
        }
        if ( m_DeviceId )
        {
            break;
        }
        os::SleepThread( interval );
        count++;
    };
}

Result Driver::ForceToRemoveDevice() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    USBPD_DBG_LOG("SystemReset\n");

    result = SendAndWaitCommand2( nullptr, AlertStatus::CommandCompleteOrAbort::Mask, L2Command_SystemReset );
    os::SleepThread( SystemResetTimeSpan + PdDebounceTimeSpan );

    return result;
}

void Driver::HardwareReset( bool isPreWait ) NN_NOEXCEPT
{
    if ( m_ResetResult.IsSuccess() )
    {
        std::lock_guard<nn::os::Mutex> vdmLock( m_ReplyCommonVdmMutex );
        std::lock_guard<nn::os::Mutex> commandLock( m_L2CommandMutex );
        std::lock_guard<nn::os::Mutex> alertLock( m_AlertMutex );
        std::lock_guard<nn::os::Mutex> i2cLock( m_I2cMutex );

        USBPD_DBG_LOG("HwReset\n");

        // 通常時の OVP 発生時は VBUS 放電を待ってからリセット
        // 待ち時間が足りないと再度 OVP が発生する
        if ( isPreWait )
        {
            os::SleepThread( VBusDischargeTimeSpan );
        }
        // 直前のアラートイベントクリア
        for ( int i = 0; i < AlertNum; ++i )
        {
            // HardReset アラートは対向デバイスからの異常通知の場合があるので残す
            if ( i != AlertStatus::HardReset::Pos )
            {
                m_pAlertEvent[i]->Clear();
            }
        }
        // HW リセット（回路上で論理反転されている）
        gpio::SetValue( &m_PdResetSession, gpio::GpioValue_High );
        os::SleepThread( HwResetTimeSpan );
        gpio::SetValue( &m_PdResetSession, gpio::GpioValue_Low );
        // 初期化時は 15V 印可時のリセットによって OVP が発生するため
        // リセット回数が変わらないので起動時間を優先してリセット後に
        // VBUS 放電待ち
        // FW ブートがこのウェイト中に完了するはず
        if ( !isPreWait )
        {
            os::SleepThread( VBusDischargeTimeSpan );
        }
        // FW ブート待ち
        while ( gpio::GetValue( &m_AlertSession ) )
        {
            os::SleepThread( TimeSpan::FromMilliSeconds( 10 ) );
        }
    }

    // 遅くともここで Alert スレッドが呼び出される
    UpdateAlert();
    WaitDeviceId();
#ifdef ENABLE_PDC_FW_BOOT_TIMEOUT_TEST
    // PDC から強制的にウェイクさせられないようにするため HW リセット継続
    // ※ 充電不可になり、もし Copper で行ってしまうとシャットダウンすることに注意
    if ( m_ResetResult <= ResultPdcFwBootTimeout() )
    {
        std::lock_guard<nn::os::Mutex> vdmLock( m_ReplyCommonVdmMutex );
        std::lock_guard<nn::os::Mutex> commandLock( m_L2CommandMutex );
        std::lock_guard<nn::os::Mutex> alertLock( m_AlertMutex );
        std::lock_guard<nn::os::Mutex> i2cLock( m_I2cMutex );

        USBPD_WARNING("HwReset forever for FW boot timeout!\n");

        // FW ブートエラー時は HW リセット継続（回路上で論理反転されている）
        gpio::SetValue( &m_PdResetSession, gpio::GpioValue_High );
    }
#endif

    // FW ブート後 System Reset にすぐ応答できない期間を待つ
    os::SleepThread( FirmwareBootToSystemResetTimeSpan );
}

} // detail
} // driver
} // pd
} // usb
} // nn
