﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_HsPrivateIncludes.h"

namespace nn { namespace usb { namespace hs {


//--------------------------------------------------------------------------
//  HostControllerDriverEndpoint class implementation
//--------------------------------------------------------------------------
void* HostControllerDriverEndpoint::operator new(size_t size) NN_NOEXCEPT
{
    return detail::UsbMemoryAllocAligned(size, ObjectMemAlignmentSize,
                                         "HostControllerDriverEndpoint");
}
void HostControllerDriverEndpoint::operator delete(void *p, size_t size) NN_NOEXCEPT
{
    detail::UsbMemoryFree(p, "HostControllerDriverEndpoint");
}

Result HostControllerDriverEndpoint::InitializeAsync(HostEndpoint *pHep,
                                                     HostControllerDriverDeviceContext* pDeviceContext,
                                                     InitializeCallback callback, void* context)
{
    Result result = ResultSuccess();

    m_pHep               = pHep;
    m_InitializeCallback = callback;
    m_InitializeContext  = context;
    m_pPlatform          = m_pHc->GetPlatformController();
    m_pDeviceContext     = pDeviceContext;

    result = m_UrbPool.Initialize(pHep->maxUrbCount);
    if (result.IsFailure())
    {
        return result;
    }

    m_State = State_OpenPending;

    result = m_pHc->OpenEpAsync(this);
    if (result.IsFailure())
    {
        m_State = State_New;
        m_UrbPool.Finalize();
        return result;
    }

    return ResultSuccess();
}

Result HostControllerDriverEndpoint::ResetAsync(ResetCallback callback, void *context)
{
    Result result = ResultSuccess();

    switch (m_State)
    {
    case State_New:
    case State_Closed:
    case State_Error:
        result = ResultEndpointStateInvalid();
        break;

    case State_Open:
        m_ResetCallback = callback;
        m_ResetContext = context;
        m_State = State_ResetPending;
        result = m_pHc->ResetEpAsync(this);
        if (ResultHcCrash::Includes(result))
        {
            /*
             * Falcon is dead, fake the completion, otherwise EP0 would stuck
             * in State_ResetPending, and we are not going to be able to tear
             * down that device (ResultResourceBusy)
             */
            result = ResultSuccess();
            ReportResetComplete(result);
        }
        break;

    case State_OpenPending:
    case State_ClosePending:
    case State_ResetPending:
        result = ResultResourceBusy();
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return result;
}

Result HostControllerDriverEndpoint::TerminateAsync(TerminateCallback callback, void *context)
{
    Result result = ResultSuccess();

    m_TerminateCallback = callback;
    m_TerminateContext  = context;

    switch (m_State)
    {
    case State_New:
    case State_Closed:
        result = ResultEndpointStateInvalid();
        break;

    case State_Open:
        m_State = State_ClosePending;
        result = m_pHc->CloseEpAsync(this);
        break;

    case State_OpenPending:
    case State_ClosePending:
    case State_ResetPending:
        result = ResultResourceBusy();
        break;

    case State_Error:
        // Not in the open state, so we can see it's "closed"
        ReportCloseComplete(ResultSuccess());
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return result;
}

Result HostControllerDriverEndpoint::Finalize()
{
    Result result = ResultSuccess();

    switch (m_State)
    {
    case State_New:
    case State_Open:
    case State_OpenPending:
    case State_ClosePending:
    case State_ResetPending:
        result = ResultInternalStateError();
        break;

    case State_Error:
    case State_Closed:
        result = m_UrbPool.Finalize();
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    return result;
}

bool HostControllerDriverEndpoint::IsTerminationPending()
{
    return (m_State == State_ClosePending);
}

HostEndpoint* HostControllerDriverEndpoint::GetHostEndpoint()
{
    return m_pHep;
}

uint8_t HostControllerDriverEndpoint::GetDeviceAddress()
{
    return m_pDeviceContext->address;
}

UsbDeviceSpeed HostControllerDriverEndpoint::GetDeviceSpeed()
{
    return m_pDeviceContext->speed;
}

PlatformController* HostControllerDriverEndpoint::GetPlatformController()
{
    return m_pHc->m_pPlatform;
}

void HostControllerDriverEndpoint::SetDriverEndpoint(void *pDriverEp)
{
    // sanity check
    if ((m_pDriverEp != nullptr) && (pDriverEp != nullptr))
    {
        NN_USB_ABORT("m_pDriverEp already assigned");
    }
    m_pDriverEp = pDriverEp;
}

void* HostControllerDriverEndpoint::GetDriverEndpoint()
{
    return m_pDriverEp;
}

HostControllerDriverDeviceContext* HostControllerDriverEndpoint::GetDeviceContext()
{
    return m_pDeviceContext;
}

void HostControllerDriverEndpoint::ReportOpenComplete(Result status)
{
    NN_USB_ABORT_UNLESS(m_State == State_OpenPending);

    if(status.IsSuccess())
    {
        m_State = State_Open;
    }
    else
    {
        m_State = State_Error;
    }

    (*m_InitializeCallback)(m_pHs, status, this, m_InitializeContext);
}

void HostControllerDriverEndpoint::ReportResetComplete(Result status)
{
    NN_USB_ABORT_UNLESS(m_State == State_ResetPending);

    m_UrbPool.CompleteAll();

    if (status.IsSuccess())
    {
        m_State = State_Open;
    }
    else
    {
        m_State = State_Error;
    }

    if (m_ResetCallback)
    {
        (*m_ResetCallback)(m_pHs, status, this, m_ResetContext);

        m_ResetCallback = nullptr;
        m_ResetContext  = nullptr;
    }
}

void HostControllerDriverEndpoint::ReportCloseComplete(Result status)
{
    // Now that host controller promises that it won't DMA anymore, we can
    // safely complete any pending transactions
    m_UrbPool.CompleteAll();

    DoTerminateCallback(status);
}

Result HostControllerDriverEndpoint::CreatePrivateUrb(nn::dd::ProcessHandle     procHandle,
                                                      uint64_t                  procVa,
                                                      uint32_t                  size,
                                                      bool                      isSmmuMapDynamic,
                                                      UsbBusToken               token,
                                                      UsbRequestBlock::Callback callback,
                                                      void                     *context,
                                                      uint32_t                  timeoutInMs,
                                                      UsbRequestBlock         **ppCreatedUrb)
{
    return m_UrbPool.AllocateUrb(
        ppCreatedUrb,
        procHandle, procVa, size, isSmmuMapDynamic, token,
        callback, context, timeoutInMs
    );
}

Result HostControllerDriverEndpoint::CreateCtrlUrb(nn::dd::ProcessHandle     procHandle,
                                                   uint64_t                  procVa,
                                                   UsbCtrlRequest&           ctrlRequest,
                                                   bool                      isSmmuMapDynamic,
                                                   UsbRequestBlock::Callback callback,
                                                   void                     *context,
                                                   uint32_t                  timeoutInMs,
                                                   UsbRequestBlock         **ppCreatedUrb)
{
    return m_UrbPool.AllocateUrb(
        ppCreatedUrb,
        procHandle, procVa, ctrlRequest, isSmmuMapDynamic,
        callback, context, timeoutInMs
    );
}

Result HostControllerDriverEndpoint::CreateUrb(ClientEpSession::UrbDeferredRequest *pRequest,
                                               UsbRequestBlock     **ppCreatedUrb)
{
    return m_UrbPool.AllocateUrb(ppCreatedUrb, pRequest);
}

void HostControllerDriverEndpoint::DestroyUrb(UsbRequestBlock *pUrb)
{
    m_UrbPool.FreeUrb(pUrb);
}

void HostControllerDriverEndpoint::CompleteUrb(UsbRequestBlock *pUrb)
{
    if (pUrb->m_Callback != nullptr)
    {
        (*pUrb->m_Callback)(m_pHs, pUrb);
    }
    else if (pUrb->m_pDeferredRequest != nullptr)
    {
        auto& data = pUrb->m_pDeferredRequest->m_Data;

        data.status = XferStatus_Finished;

        for (uint32_t i = 0; i < pUrb->m_XferCount; i++)
        {
            data.xfers[i].xferredSize = pUrb->m_Xfer[i].xferredSize;
            data.xfers[i].result      = pUrb->m_Xfer[i].result;
        }

        pUrb->m_pDeferredRequest->CompleteRequest(ResultSuccess());
    }
}

Result HostControllerDriverEndpoint::SubmitUrb(UsbRequestBlock *pUrb)
{
    if (m_State != State_Open)
    {
        return ResultEndpointClosed();
    }

    return m_pHc->SubmitUrbAsync(pUrb);
}

void HostControllerDriverEndpoint::DoTerminateCallback(Result status)
{
    if ((m_TerminateCallback != nullptr) && (m_State != State_Closed))
    {
        (*m_TerminateCallback)(m_pHs, status, this, m_TerminateContext);
        m_State = State_Closed;

        m_TerminateCallback = nullptr;
        m_TerminateContext  = nullptr;
    }
}

void HostControllerDriverEndpoint::TimerCallback(LocalEventDataType *pData)
{
    // If an URB times out, our only recourse is to begin process
    // of closing the whole endpoint. Interrupting an active transaction
    // requires high level resynchronization (such as that of endpoint data toggle).
    if (m_State == State_Open)
    {
        m_State = State_ResetPending;
        m_pHc->ResetEpAsync(this);
    }
}


//--------------------------------------------------------------------------
//  HostControllerDriver class implementation
//--------------------------------------------------------------------------

HostControllerDriver::HostControllerDriver(Hs *pHs, ControllerType controllerType, StartupMode mode,
                                           PlatformController* pPlatformController)
    : m_pHs(pHs)
    , m_ControllerType(controllerType)
    , m_StartupMode(mode)
    , m_PortCount(0)
    , m_pPlatform(pPlatformController)
{
    NN_UNUSED(m_pHs);
    memset(m_HcName, 0, sizeof(m_HcName));
    memset(m_PortEventHandlers, 0, sizeof(m_PortEventHandlers));
}


HostControllerDriver::~HostControllerDriver()
{

}

Result HostControllerDriver::Initialize()
{
    Result result = ResultSuccess();

    return result;
}


Result HostControllerDriver::Finalize()
{
    Result result = ResultSuccess();

    NN_USB_LOG_INFO("HostControllerDriver::Finalize()\n");

    return result;
}

int HostControllerDriver::GetObjectName(char *pName, uint32_t maxSize)
{
    return strlcpy(pName, m_HcName, maxSize);
}

PlatformController* HostControllerDriver::GetPlatformController()
{
    return m_pPlatform;
}

HostControllerDriver::ControllerType HostControllerDriver::GetControllerType()
{
    return m_ControllerType;
}

void HostControllerDriver::ReportPortEvent()
{
    uint32_t portChangeMask = 0;

    // Collect port events
    for (int32_t portIndex = 0; portIndex < m_PortCount; portIndex++)
    {
        UsbHubPortStatus portStatus;
        HubPortNumber port = portIndex + 1;
        if (GetRootHubPortStatus(port, &portStatus).IsSuccess())
        {
            if (portStatus.wPortChange != 0)
            {
                portChangeMask |= (1 << port);
            }
        }
    }

    // Distribute port events to registered handlers
    for(int32_t handlerIndex=0; handlerIndex < NN_USB_ARRAY_COUNT32(m_PortEventHandlers); handlerIndex++)
    {
        PortEventHandler* pHandler = m_PortEventHandlers + handlerIndex;
        if((pHandler->callback != nullptr) && (pHandler->mask & portChangeMask))
        {
            pHandler->callback(pHandler->context, pHandler->mask & portChangeMask);
        }
    }
}

Result HostControllerDriver::SetPortEventHandler(void *context, void (*callback)(void *, uint32_t), uint32_t portMask)
{
    Result result = ResultResourceBusy();
    for(int32_t handlerIndex=0; handlerIndex < NN_USB_ARRAY_COUNT32(m_PortEventHandlers); handlerIndex++)
    {
        PortEventHandler* pHandler = m_PortEventHandlers + handlerIndex;
        if(pHandler->callback == nullptr)
        {
            pHandler->context  = context;
            pHandler->callback = callback;
            pHandler->mask     = portMask;
            result = ResultSuccess();
            break;
        }
    }
    return result;
}

Result HostControllerDriver::ClearPortEventHandler(void *context)
{
    Result result = ResultInternalStateError();
    for(int32_t handlerIndex=0; handlerIndex < NN_USB_ARRAY_COUNT32(m_PortEventHandlers); handlerIndex++)
    {
        PortEventHandler* pHandler = m_PortEventHandlers + handlerIndex;
        if((pHandler->callback != nullptr) && (pHandler->context == context))
        {
            pHandler->context  = nullptr;
            pHandler->callback = nullptr;
            pHandler->mask     = 0;
            result = ResultSuccess();
            break;
        }
    }
    return result;
}

} // end of namespace hs
} // end of namespace usb
} // end of namespace nn
