﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "usb_DsServiceImpl.h"
#include "usb_DsController.h"

using namespace nn::sf;
using namespace nn::usb;

namespace nn {
namespace usb {
namespace ds {

//////////////////////////////////////////////////////////////////////////////
DsServiceImpl::DsServiceImpl(detail::UsbPlatform *pPlatform) NN_NOEXCEPT
    : m_pPlatform(pPlatform)
    , m_pProtocol(nullptr)
    , m_StateChangeEvent(nn::os::EventClearMode_ManualClear, true)
{
    for (unsigned i = 0; i < NN_USB_ARRAY_SIZE(m_pInterface); i++)
    {
        m_pInterface[i] = nullptr;
    }
}


//////////////////////////////////////////////////////////////////////////////
DsServiceImpl::~DsServiceImpl() NN_NOEXCEPT
{
    UnbindComplex();
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::BindComplex(ComplexId complexId) NN_NOEXCEPT
{
    detail::UsbComplex *complex;
    DsController *controller;

    // prevent double bind
    if (m_pProtocol != nullptr)
    {
        NN_USB_LOG_INFO("BindDevice() double bind\n");
        return ResultOperationDenied();
    }

    complex = m_pPlatform->GetComplex(complexId);
    if (complex == nullptr)
    {
        NN_USB_LOG_INFO("GetComplex(%d) failed\n", complexId);
        return ResultInvalidParameter();
    }

    controller = reinterpret_cast<DsController*>(complex->GetController(UsbCapability_Device));
    if (controller == nullptr)
    {
        NN_USB_LOG_INFO("GetController(%d) failed\n", UsbCapability_Device);
        return ResultInvalidParameter();
    }

    m_pProtocol = &(controller->protocol);
    m_pProtocol->BindStateChangeEvent(&m_StateChangeEvent);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
void DsServiceImpl::UnbindComplex() NN_NOEXCEPT
{
    if (m_pProtocol != nullptr)
    {
        m_pProtocol->UnbindStateChangeEvent(&m_StateChangeEvent);
        m_pProtocol = nullptr;
    }
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::BindClientProcess(NativeHandle processHandle) NN_NOEXCEPT
{
    m_ProcessHandle = std::move(processHandle);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
void DsServiceImpl::UnbindClientProcess() NN_NOEXCEPT
{
    // nothing to do, m_ProcessHandle destructor handles everything
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::RegisterInterface(nn::sf::Out<nn::sf::SharedPointer<IDsInterface>> pInterface, uint8_t bInterfaceNumber) NN_NOEXCEPT
{
    Result  result;

    // must bind to a complex first
    if (m_pProtocol == nullptr)
    {
        return ResultOperationDenied();
    }

    result = m_pProtocol->RegisterInterface(bInterfaceNumber);

    if (result.IsSuccess())
    {
        *pInterface = detail::Factory::CreateSharedEmplaced<IDsInterface, DsInterfaceImpl>(
            detail::UsbMemoryGetAllocator(),
            m_pProtocol,
            this,
            bInterfaceNumber,
            reinterpret_cast<nn::dd::ProcessHandle>(m_ProcessHandle.GetOsHandle())
        );
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::GetStateChangeEvent(Out<NativeHandle> eventHandle) NN_NOEXCEPT
{
    eventHandle.Set(NativeHandle(m_StateChangeEvent.GetReadableHandle(), false));
    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::GetState(Out<UsbState> state) NN_NOEXCEPT
{
    if (m_pProtocol == nullptr)
    {
        return ResultOperationDenied();
    }

    state.Set(m_pProtocol->GetState(&m_StateChangeEvent));

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::ClearDeviceData() NN_NOEXCEPT
{
    return m_pProtocol->ClearDeviceData();
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::AddUsbStringDescriptor(Out<uint8_t>pIndex, nn::sf::InBuffer pUsbStringDescriptor) NN_NOEXCEPT
{
    UsbStringDescriptor usbStringDescriptor;

    if (pUsbStringDescriptor.GetSize() != sizeof(UsbStringDescriptor))
    {
        return ResultBufferSizeError();
    }

    memcpy(&usbStringDescriptor, pUsbStringDescriptor.GetPointerUnsafe(), sizeof(UsbStringDescriptor));
    uint8_t index;

    Result result = m_pProtocol->AddUsbStringDescriptor(&usbStringDescriptor, &index);

    if (result.IsSuccess())
    {
        *pIndex = index;
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::DeleteUsbStringDescriptor(uint8_t index) NN_NOEXCEPT
{
    return m_pProtocol->DeleteUsbStringDescriptor(index);
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::SetUsbDeviceDescriptor(InBuffer pUsbDeviceDescriptor, UsbDeviceSpeed usbDeviceSpeed) NN_NOEXCEPT
{
    UsbDeviceDescriptor usbDeviceDescriptor;

    if (pUsbDeviceDescriptor.GetSize() != sizeof(usbDeviceDescriptor))
    {
        return ResultBufferSizeError();
    }

    memcpy(&usbDeviceDescriptor, pUsbDeviceDescriptor.GetPointerUnsafe(), sizeof(UsbDeviceDescriptor));

    return m_pProtocol->SetUsbDeviceDescriptor(&usbDeviceDescriptor, usbDeviceSpeed);
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::SetBinaryObjectStore(InBuffer pBinaryObjectStore) NN_NOEXCEPT
{
    uint8_t binaryObjectStore[BinaryObjectStoreSize];

    if (pBinaryObjectStore.GetSize() > BinaryObjectStoreSize)
    {
        return ResultBufferSizeError();
    }

    memcpy(&binaryObjectStore, pBinaryObjectStore.GetPointerUnsafe(), pBinaryObjectStore.GetSize());

    return m_pProtocol->SetBinaryObjectStore(binaryObjectStore, pBinaryObjectStore.GetSize());
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::Enable() NN_NOEXCEPT
{
    return m_pProtocol->EnableDevice();
}


//////////////////////////////////////////////////////////////////////////////
Result DsServiceImpl::Disable() NN_NOEXCEPT
{
    return m_pProtocol->DisableDevice();
}


//////////////////////////////////////////////////////////////////////////////
void DsServiceImpl::OnAddInterface(uint8_t ifNum, DsInterfaceImpl *pInterface) NN_NOEXCEPT
{
    m_pInterface[ifNum] = pInterface;
}


//////////////////////////////////////////////////////////////////////////////
void DsServiceImpl::OnDelInterface(uint8_t ifNum) NN_NOEXCEPT
{
    m_pInterface[ifNum] = nullptr;
}

} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
