﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//#define NN_USB_ENABLE_TRACE 1

#include "usb_DsDescriptor.h"
#include "usb_DsInterfaceData.h"
#include "usb_DsProtocol.h"

#undef  NN_USB_TRACE_CLASS_NAME
#define NN_USB_TRACE_CLASS_NAME "DsInterfaceData"

using namespace nn::usb;

namespace nn {
namespace usb {
namespace ds {
//////////////////////////////////////////////////////////////////////////////
// Public Methods
//////////////////////////////////////////////////////////////////////////////
Result DsInterfaceData::Initialize() NN_NOEXCEPT
{
    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsInterfaceData::Finalize() NN_NOEXCEPT
{
    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsInterfaceData::ClearData() NN_NOEXCEPT
{
    memset(m_ConfigurationDataFullSpeed,        0, sizeof(m_ConfigurationDataFullSpeed));
    memset(m_ConfigurationDataHighSpeed,        0, sizeof(m_ConfigurationDataHighSpeed));
    memset(m_ConfigurationDataSuperSpeed,       0, sizeof(m_ConfigurationDataSuperSpeed));
    memset(m_ConfiguratioDataSizeFullSpeed,     0, sizeof(m_ConfiguratioDataSizeFullSpeed));
    memset(m_ConfiguratioDataSizeHighSpeed,     0, sizeof(m_ConfiguratioDataSizeHighSpeed));
    memset(m_ConfiguratioDataSizeSuperSpeed,    0, sizeof(m_ConfiguratioDataSizeSuperSpeed));

    memset(m_ReportDescriptor,                  0, sizeof(m_ReportDescriptor));
    memset(m_ReportDescriptorSize,              0, sizeof(m_ReportDescriptorSize));

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result DsInterfaceData::AppendConfigurationData(uint8_t bInterfaceNumber, UsbDeviceSpeed usbDeviceSpeed, uint8_t *pData, int bytes) NN_NOEXCEPT
{
    Result result = ResultInvalidParameter();

    if (bInterfaceNumber < DsLimitMaxInterfacesPerConfigurationCount)
    {
        uint8_t *pDest  = NULL;
        int *pSize      = NULL;

        switch (usbDeviceSpeed)
        {
        case UsbDeviceSpeed_Low:
        case UsbDeviceSpeed_Full:

            pSize   = &m_ConfiguratioDataSizeFullSpeed[bInterfaceNumber];
            pDest   = &m_ConfigurationDataFullSpeed[bInterfaceNumber][*pSize];

            break;

        case UsbDeviceSpeed_High:

            pSize   = &m_ConfiguratioDataSizeHighSpeed[bInterfaceNumber];
            pDest   = &m_ConfigurationDataHighSpeed[bInterfaceNumber][*pSize];

            break;

        case UsbDeviceSpeed_Super:

            pSize   = &m_ConfiguratioDataSizeSuperSpeed[bInterfaceNumber];
            pDest   = &m_ConfigurationDataSuperSpeed[bInterfaceNumber][*pSize];

            break;

        default:

            NN_USB_LOG_INFO("%s::%s(@%d) unsupported UsbDeviceSpeed\n", NN_USB_TRACE_CLASS_NAME, __FUNCTION__,__LINE__);

            break;
        }

        if (pDest)
        {
            result = ResultBufferSizeError();

            if ((MaxInterfaceDataSize - *pSize) >= bytes)
            {
                memcpy(pDest, pData, bytes);
                *pSize = *pSize + bytes;
                result = ResultSuccess();
            }
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsInterfaceData::GetConfigurationData(uint8_t bInterfaceNumber, UsbDeviceSpeed usbDeviceSpeed, uint8_t *pDest, int *pSize) NN_NOEXCEPT
{
    Result result = ResultInvalidParameter();

    if (bInterfaceNumber < DsLimitMaxInterfacesPerConfigurationCount)
    {
        uint8_t *pSrc   = NULL;
        int size        = 0;

        switch (usbDeviceSpeed)
        {
        case UsbDeviceSpeed_Low:
        case UsbDeviceSpeed_Full:

            size    = m_ConfiguratioDataSizeFullSpeed[bInterfaceNumber];
            pSrc    = m_ConfigurationDataFullSpeed[bInterfaceNumber];

            break;

        case UsbDeviceSpeed_High:

            size    = m_ConfiguratioDataSizeHighSpeed[bInterfaceNumber];
            pSrc    = m_ConfigurationDataHighSpeed[bInterfaceNumber];

            break;

        case UsbDeviceSpeed_Super:

            size    = m_ConfiguratioDataSizeSuperSpeed[bInterfaceNumber];
            pSrc    = m_ConfigurationDataSuperSpeed[bInterfaceNumber];

            break;

        default:

            NN_USB_LOG_INFO("%s::%s(@%d) unsupported UsbDeviceSpeed\n", NN_USB_TRACE_CLASS_NAME, __FUNCTION__,__LINE__);

            break;
        }

        if (pSrc)
        {
            if (size)
            {
                memcpy(pDest, pSrc, size);
            }

            *pSize = size;

            result = ResultSuccess();
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsInterfaceData::SetReportData(uint8_t bInterfaceNumber, uint8_t *pData, int bytes) NN_NOEXCEPT
{
    Result result = ResultInvalidParameter();

    if (bInterfaceNumber < DsLimitMaxInterfacesPerConfigurationCount)
    {
        result = ResultBufferSizeError();

        if (bytes <= MaxInterfaceDataSize)
        {
            memcpy(m_ReportDescriptor[bInterfaceNumber], pData, bytes);
            m_ReportDescriptorSize[bInterfaceNumber] = bytes;

            result = ResultSuccess();
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DsInterfaceData::GetReportData(uint8_t bInterfaceNumber, uint8_t *pDest, int *pSize) NN_NOEXCEPT
{
    Result result = ResultInvalidParameter();

    if (bInterfaceNumber < DsLimitMaxInterfacesPerConfigurationCount)
    {
        memcpy(pDest, m_ReportDescriptor[bInterfaceNumber], m_ReportDescriptorSize[bInterfaceNumber]);
        *pSize = m_ReportDescriptorSize[bInterfaceNumber];

        result = ResultSuccess();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
} // end of namespace ds
} // end of namespace usb
} // end of namespace nn
