﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/os/os_SdkThreadCommon.h>

#include "usb_Platform.h"
#include "usb_Util.h"
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/settings/system/settings_Usb.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace nn {
namespace usb {
namespace detail {

Result UsbPlatform::Initialize() NN_NOEXCEPT
{
    m_pRequestHolder = nullptr;
    m_pFreeHolder    = nullptr;

    LoadPlatformConfig();

    // prepare ISR thread control events
    nn::os::InitializeMultiWait(&m_MultiWait);
    nn::os::InitializeMultiWaitHolder(&m_BreakHolder, m_BreakEvent.GetBase());
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_BreakHolder);
    nn::os::InitializeMultiWaitHolder(&m_ContinueHolder, m_ContinueEvent.GetBase());
    nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_ContinueHolder);

    // create and start ISR thread
    nn::os::CreateThread(&m_IsrThread,
                         IsrThreadEntry,
                         this,
                         m_IsrThreadStack,
                         sizeof(m_IsrThreadStack),
                         NN_SYSTEM_THREAD_PRIORITY(usb,InterruptHandler));
    nn::os::SetThreadNamePointer(&m_IsrThread, NN_SYSTEM_THREAD_NAME(usb, InterruptHandler));
    nn::os::StartThread(&m_IsrThread);

    return ResultSuccess();
}

Result UsbPlatform::Finalize() NN_NOEXCEPT
{
    // terminate ISR thread
    m_BreakEvent.Signal();
    nn::os::WaitThread(&m_IsrThread);
    nn::os::DestroyThread(&m_IsrThread);

    // destroy ISR thread control events
    nn::os::UnlinkAllMultiWaitHolder(&m_MultiWait);
    nn::os::FinalizeMultiWaitHolder(&m_ContinueHolder);
    nn::os::FinalizeMultiWaitHolder(&m_BreakHolder);
    nn::os::FinalizeMultiWait(&m_MultiWait);

    return ResultSuccess();
}

const UsbPlatform::Config& UsbPlatform::GetConfig() const NN_NOEXCEPT
{
    return m_Config;
}

Result UsbPlatform::AddComplex(UsbComplex *pComplex) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ComplexListMutex);

    Result result = pComplex->Initialize(this);

    if (result.IsSuccess())
    {
        m_ComplexList.push_back(pComplex);
    }

    return result;
}

UsbComplex *UsbPlatform::GetComplex(ComplexId complexId) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ComplexListMutex);

    for (UsbComplex *pComplex : m_ComplexList)
    {
        if (pComplex->id == complexId)
        {
            return pComplex;
        }
    }

    return nullptr;
}

Result UsbPlatform::DelComplex(UsbComplex *pComplex) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ComplexListMutex);

    m_ComplexList.remove(pComplex);

    return pComplex->Finalize();
}

Result UsbPlatform::RequestIrq(nn::os::MultiWaitHolderType *holder,
                               IrqHandlerType *handler) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_IsrMutex);

    nn::os::SetMultiWaitHolderUserData(
        holder, reinterpret_cast<intptr_t>(handler)
    );

    if (nn::os::GetCurrentThread() == &m_IsrThread)
    {
        nn::os::LinkMultiWaitHolder(&m_MultiWait, holder);
    }
    else
    {
        m_pRequestHolder = holder;
        m_ContinueEvent.Signal();

        m_IsrSemaphore.Acquire();
    }

    return ResultSuccess();
}

Result UsbPlatform::FreeIrq(nn::os::MultiWaitHolderType *holder) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_IsrMutex);

    if (nn::os::GetCurrentThread() == &m_IsrThread)
    {
        nn::os::UnlinkMultiWaitHolder(holder);
        nn::os::SetMultiWaitHolderUserData(holder, 0);
    }
    else
    {
        m_pFreeHolder = holder;
        m_ContinueEvent.Signal();

        m_IsrSemaphore.Acquire();
    }

    return ResultSuccess();
}

void UsbPlatform::Isr() NN_NOEXCEPT
{
    IrqHandlerType *handler;
    nn::os::MultiWaitHolderType *holder;

    while (true)
    {
        holder = nn::os::WaitAny(&m_MultiWait);

        if (holder == &m_BreakHolder)
        {
            m_BreakEvent.Clear();
            break;
        }

        if (holder == &m_ContinueHolder)
        {
            m_ContinueEvent.Clear();

            if (m_pRequestHolder != nullptr)
            {
                nn::os::LinkMultiWaitHolder(&m_MultiWait, m_pRequestHolder);
                m_pRequestHolder = nullptr;
            }
            else // m_pFreeHolder != nullptr
            {
                nn::os::UnlinkMultiWaitHolder(m_pFreeHolder);
                nn::os::SetMultiWaitHolderUserData(m_pFreeHolder, 0);
                m_pFreeHolder = nullptr;
            }

            m_IsrSemaphore.Release();

            continue;
        }

        handler = reinterpret_cast<IrqHandlerType*>(
            nn::os::GetMultiWaitHolderUserData(holder)
        );

        (*handler)();
    }
}

void UsbPlatform::LoadPlatformConfig() NN_NOEXCEPT
{
    std::memset(&m_Config, 0, sizeof(m_Config));

    m_Config.socName                   = GetSocName();
    m_Config.isRootPortPowerControlled = IsRootPortPowerControlled();
    m_Config.isUsb30Enabled            = IsUsb30Enabled();
    m_Config.hsCurrLevelOffset         = GetDriveStrengthOffset();
    m_Config.portCount                 = GetPortCount();

    // And what does each port look like?
    for (uint32_t i = 0; i < m_Config.portCount; i++)
    {
        auto& port = m_Config.port[i];

        port.capability = GetSpeedCapability(i) | GetRoleCapability(i);
        port.hsLane     = GetHsLane(i);

        if ((port.capability & UsbCapability_SuperSpeed)   ||
            (port.capability & UsbCapability_SuperSpeedPlus))
        {
            port.ssLane = GetSsLane(i);
        }
    }
}

UsbPlatform::SocName
UsbPlatform::GetSocName() const NN_NOEXCEPT
{
    struct {
        const char *string;
        SocName socName;
    } const socNameMap[] = {
        { "T210", SocName_Tx1    },
        { "T214", SocName_Mariko },
    };

    char   key[] = "soc_name";
    char   value[8];
    size_t size;

    size = nn::settings::fwdbg::GetSettingsItemValue(
        &value, sizeof(value), "usb", key
    );

    for (const auto& map : socNameMap)
    {
        if (nn::util::Strncmp(value, map.string, sizeof(value)) == 0)
        {
            return map.socName;
        }
    }

    value[sizeof(value) - 1] = 0;
    NN_USB_ABORT("Unsupported SoC: %s\n", value);
}

bool UsbPlatform::IsRootPortPowerControlled() const NN_NOEXCEPT
{
    char   key[] = "is_root_port_power_controlled";
    bool   value;
    size_t size;

    size = nn::settings::fwdbg::GetSettingsItemValue(
        &value, sizeof(value), "usb", key
    );
    NN_USB_ABORT_UNLESS(size == sizeof(value));

    return value;
}

bool UsbPlatform::IsUsb30Enabled() const NN_NOEXCEPT
{
    char   key[] = "usb30_force_enabled"; // SIGLO-51105
    bool   value;
    size_t size;

    size = nn::settings::fwdbg::GetSettingsItemValue(
        &value, sizeof(value), "usb", key
    );
    NN_USB_ABORT_UNLESS(size == sizeof(value));

    return value || nn::settings::system::IsUsb30Enabled();
}

int32_t UsbPlatform::GetDriveStrengthOffset() const NN_NOEXCEPT
{
    char     key[] = "hs_curr_level_offset"; // NSBG-8267
    int32_t  value;
    size_t   size;

    size = nn::settings::fwdbg::GetSettingsItemValue(
        &value, sizeof(value), "usb", key
    );
    NN_USB_ABORT_UNLESS(size == sizeof(value));

    return value;
}

uint32_t UsbPlatform::GetPortCount() const NN_NOEXCEPT
{
    char     key[] = "port_count";
    uint32_t value;
    size_t   size;

    size = nn::settings::fwdbg::GetSettingsItemValue(
        &value, sizeof(value), "usb", key
    );
    NN_USB_ABORT_UNLESS(size == sizeof(value));
    NN_USB_ABORT_UNLESS(value <= HwLimitMaxPortCount);

    return value;
}

uint32_t UsbPlatform::GetRoleCapability(uint8_t portNumber) const NN_NOEXCEPT
{
    struct {
        const char *string;
        uint32_t    capability;
    } roleMap[] = {
        { "Host",     UsbCapability_Host                        },
        { "Device",   UsbCapability_Device                      },
        { "DualRole", UsbCapability_Host | UsbCapability_Device },
    };

    char   key[] = "port0_role";
    char   value[8];
    size_t size;

    key[4] += portNumber;

    size = nn::settings::fwdbg::GetSettingsItemValue(
        value, sizeof(value), "usb", key
    );

    for (auto& map : roleMap)
    {
        if (strncasecmp(value, map.string, size) == 0)
        {
            return map.capability;
        }
    }

    NN_USB_ABORT("Invalid port role specifier\n");
}

uint32_t UsbPlatform::GetSpeedCapability(uint8_t portNumber) const NN_NOEXCEPT
{
    char   key[]  = "port0_is_ss_capable";
    bool   value;
    size_t size;

    uint32_t capability = UsbCapability_LowSpeed  |
                          UsbCapability_FullSpeed |
                          UsbCapability_HighSpeed;

    if (!m_Config.isUsb30Enabled)
    {
        return capability;
    }

    key[4] += portNumber;

    size = nn::settings::fwdbg::GetSettingsItemValue(
        &value, sizeof(value), "usb", key
    );

    NN_USB_ABORT_UNLESS(size == sizeof(value));

    if (value == true)
    {
        capability |= UsbCapability_SuperSpeed;
    }

    return capability;
}

uint32_t UsbPlatform::GetHsLane(uint8_t portNumber) const NN_NOEXCEPT
{
    char     key[]  = "port0_hs_lane";
    uint32_t value;
    size_t   size;

    key[4] += portNumber;

    size = nn::settings::fwdbg::GetSettingsItemValue(
        &value, sizeof(value), "usb", key
    );

    NN_USB_ABORT_UNLESS(size == sizeof(value));

    return value;
}

uint32_t UsbPlatform::GetSsLane(uint8_t portNumber) const NN_NOEXCEPT
{
    char     key[]  = "port0_ss_lane";
    uint32_t value;
    size_t   size;

    key[4] += portNumber;

    size = nn::settings::fwdbg::GetSettingsItemValue(
        &value, sizeof(value), "usb", key
    );

    NN_USB_ABORT_UNLESS(size == sizeof(value));

    return value;
}

} // end of namespace detail
} // end of namespace usb
} // end of namespace nn
