﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/uart/uart_PortTypes.h>
#include <nn/uart/uart_Result.h>
#include <nn/uart/driver/uart_PortApi.h>
#include "uart_PortAccessor.h"
#include "uart_Interrupt.h"
#include "uart_CircularBuffer.h"
#include "uart_Command.h"
#include "uart_Util.h"

namespace nn {
namespace uart {
namespace driver {
namespace detail {

/**
 * @brief   ポートセッションの実装
 */
class PortSessionImpl
{
    NN_DISALLOW_COPY(PortSessionImpl);
    NN_DISALLOW_MOVE(PortSessionImpl);

public:
    PortSessionImpl(int portIndex, const PortConfigType& portConfig) NN_NOEXCEPT :
        m_IsOpen(false),
        m_IsOpenSuspended(false),
        m_PortIndex(portIndex),
        m_ErrorStatus(),
        m_BaudRate(BaudRate_115200),
        m_FlowControlMode(FlowControlMode_None),
        m_PortAccessor(),
        m_SendBuffer(),
        m_ReceiveBuffer(),
        m_PortEvent()
    {
        m_MutexSend.Initialize();
        m_MutexReceive.Initialize();
        ConfigurePort(portConfig);
    }

    // デストラクタが non-trivial なので placement new した際には明示的なデストラクタコールが必須
    ~PortSessionImpl() NN_NOEXCEPT
    {
        Close(); // Force close
        DeconfigurePort();
    }

    void Open() NN_NOEXCEPT;
    bool IsOpen() const NN_NOEXCEPT
    {
        return m_IsOpen;
    }
    void Close() NN_NOEXCEPT;

    void Suspend() NN_NOEXCEPT;
    void Resume() NN_NOEXCEPT;
    void DisablePowerAndClock() NN_NOEXCEPT;

    int GetPortIndex() const NN_NOEXCEPT
    {
        return m_PortIndex;
    }

    void Send(CommandSend& command) NN_NOEXCEPT;
    void Receive(CommandReceive& command) NN_NOEXCEPT;

    size_t GetWritableLength() const NN_NOEXCEPT
    {
        return m_SendBuffer.GetWritableLength();
    }
    size_t GetReadableLength() const NN_NOEXCEPT
    {
        return m_ReceiveBuffer.GetReadableLength();
    }

    bool BindEvent(nn::os::SystemEventType* pEvent,
                   PortEventType eventType,
                   size_t threshold) NN_NOEXCEPT;
    bool UnbindEvent(nn::os::SystemEventType* pEvent) NN_NOEXCEPT;

private:

    // Flush 操作のリトライ回数
    const int      MaxFlushRetryCount = 10;

    // ポートグローバルなロックの提供。ロック順序を保証してデッドロックを回避する
    class GlobalLock
    {
    public:
        explicit GlobalLock(PortSessionImpl* p) NN_NOEXCEPT :
            m_LockSend(p->m_MutexSend),m_LockReceive(p->m_MutexReceive) {}
    private:
        std::lock_guard<StaticMutex>  m_LockSend;
        std::lock_guard<StaticMutex>  m_LockReceive;
    };

    struct EventHolder
    {
        EventHolder() NN_NOEXCEPT :
            pEvent(nullptr), threshold() {}

        nn::os::SystemEventType* pEvent;
        size_t                   threshold;
    };

    // ポートの状態変化を通知するためのイベントを格納
    struct PortEvent
    {
        PortEvent() NN_NOEXCEPT :
            sendBufferEmpty(), sendBufferReady(), receiveBufferReady() {}

        EventHolder sendBufferEmpty;
        EventHolder sendBufferReady;
        EventHolder receiveBufferReady;
        EventHolder receiveEnd;
    };

private:
    static void HandleInterruptCallback(uintptr_t arg) NN_NOEXCEPT;
    void HandleInterrupt() NN_NOEXCEPT;
    static void HandleDmaInterruptCallback(uintptr_t arg) NN_NOEXCEPT;
    void HandleDmaInterrupt(int channelIndex) NN_NOEXCEPT;
    void HandleSendBufferEvent() NN_NOEXCEPT;
    void HandleReceiveBufferEvent() NN_NOEXCEPT;
    void HandleReceiveEndEvent() NN_NOEXCEPT;

    void ConfigurePort(const PortConfigType& portConfig) NN_NOEXCEPT;
    void DeconfigurePort() NN_NOEXCEPT;
    void ReceiveDataToCircularBuffer() NN_NOEXCEPT;

private:
    // 内部状態管理変数
    bool                m_IsOpen;
    bool                m_IsOpenSuspended;

    NN_PADDING3;

    StaticMutex         m_MutexSend;
    StaticMutex         m_MutexReceive;

    // ポート情報
    int                 m_PortIndex;

    // エラーステート保存
    PortStatus          m_ErrorStatus;

    // 通信パラメータ
    BaudRate            m_BaudRate;
    FlowControlMode     m_FlowControlMode;
    bool                m_IsInvertTx;
    bool                m_IsInvertRx;
    bool                m_IsInvertRts;
    bool                m_IsInvertCts;

    // レジスタアクセサ
    PortAccessor        m_PortAccessor;

    // 送受信の循環バッファ
    CircularBuffer      m_SendBuffer;
    CircularBuffer      m_ReceiveBuffer;

    // ポートの状態変化を通知するイベント
    PortEvent           m_PortEvent;

    bool                m_IsForceRtsHighBeforeSleep;

#ifdef NN_DETAIL_UART_ENABLE_DMA

    // DMA の割り込みハンドラに渡す引数
    struct DmaInterruptArgs
    {
        PortSessionImpl* pSessionImpl;
        int channelIndex;
    };

    DmaInterruptArgs  m_DmaInterruptArgs[DmaChannelCountPerPort];
#endif
};

struct PortSessionImplPadded
{
    PortSessionImpl     impl;

    // PortSession 型の TypedStorage とサイズを合わせるための微調整部分
    char                _padding[PortSessionSize - sizeof(PortSessionImpl)];
};

NN_FORCEINLINE
PortSessionImpl& ToPortSessionImpl(PortSession& session)
{
    return nn::util::Get<PortSessionImplPadded>(session._impl).impl;
}

NN_FORCEINLINE
const PortSessionImpl& ToPortSessionImpl(const PortSession& session)
{
    return nn::util::Get<PortSessionImplPadded>(session._impl).impl;
}

} // detail
} // driver
} // uart
} // nn
