﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "uart_CircularBuffer.h"
#include "uart_PortStatus.h"
#include "uart_TargetSpec.h"

namespace nn {
namespace uart {
namespace driver {
namespace detail {

class PortAccessor
{
    NN_DISALLOW_COPY(PortAccessor);
    NN_DISALLOW_MOVE(PortAccessor);

public:
    PortAccessor() NN_NOEXCEPT : m_Reg(nullptr) {}

    void Initialize(int portIdx) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void EnablePowerAndClock(int baudRate) NN_NOEXCEPT;
    void DisablePowerAndClock() NN_NOEXCEPT;
    void Setup(int baudRate, FlowControlMode flowControlMode) NN_NOEXCEPT;
    void Start() NN_NOEXCEPT;
    void Stop() NN_NOEXCEPT;

    bool IsTransmitterReady() const NN_NOEXCEPT;
    bool IsReceiveReady() const NN_NOEXCEPT;
    bool IsReceiveEnd() const NN_NOEXCEPT;
    bool IsReceiveTimeout() const NN_NOEXCEPT;
    bool WasReceiveEndOccured() NN_NOEXCEPT
    {
        // 非対応
        return false;
    }

    void ResetReceiveEndOccured() NN_NOEXCEPT
    {
        // 何もしない
    }

    bool IsReceiveReadyInterruptEnabled() const NN_NOEXCEPT;
    bool IsReceiveEndInterruptEnabled() const NN_NOEXCEPT;
    void SetReceiveReadyInterruptEnabled(bool enable) NN_NOEXCEPT;
    void SetReceiveEndInterruptEnabled(bool enable) NN_NOEXCEPT;
    void SetReceiveInterruptEnabled(bool enable) NN_NOEXCEPT;
    void ClearReceiveEndInterrupt() NN_NOEXCEPT;

    bool MoveRestDataToCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT;
    void SetRtsToHigh(bool enable) NN_NOEXCEPT;
    bool IsRestDataExist() NN_NOEXCEPT;
    bool IsForceRtsHigh() NN_NOEXCEPT
    {
        return false;
    }

    void SendByte(char data) NN_NOEXCEPT;
    bool IsTransmitterReadyInterruptEnabled() const NN_NOEXCEPT;
    void SetTransmitterReadyInterruptEnabled(bool enable) NN_NOEXCEPT;
    PortStatusType ReceiveByte(char* data) NN_NOEXCEPT;
    void SendDataFromCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& sendBuffer) NN_NOEXCEPT;
    void ReceiveDataToCircularBuffer(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT;
    void ReceiveDataToCircularBufferByPio(PortStatus& outErrorStatus, CircularBuffer& receiveBuffer) NN_NOEXCEPT;

private:
    // ベースレジスタからのオフセット参照のための構造体
    struct RegTable
    {
        volatile uint32_t urxd;             // 0x00
                 uint8_t  padding1[0x3C];
        volatile uint32_t utxd;             // 0x40
                 uint8_t  padding2[0x3C];
        volatile uint32_t ucr1;             // 0x80
        volatile uint32_t ucr2;             // 0x84
        volatile uint32_t ucr3;             // 0x88
        volatile uint32_t ucr4;             // 0x8C
        volatile uint32_t ufcr;             // 0x90
        volatile uint32_t usr1;             // 0x94
        volatile uint32_t usr2;             // 0x98
        volatile uint32_t uesc;             // 0x9C
        volatile uint32_t utim;             // 0xA0
        volatile uint32_t ubir;             // 0xA4
        volatile uint32_t ubmr;             // 0xA8
        volatile uint32_t ubrc;             // 0xAC
        volatile uint32_t onems;            // 0xB0
        volatile uint32_t uts;              // 0xB4
        volatile uint32_t umcr;             // 0xB8
    };

private:
    void SetBaseAddr(int portIdx) NN_NOEXCEPT;

private:
    RegTable*   m_Reg;
};

} // detail
} // driver
} // uart
} // nn
