﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/uart/uart_PortTypes.h>
#include <nn/uart/detail/uart_Log.h>
#include "uart_Interrupt.h"

namespace {

// 割り込みハンドラスレッドに与えるスタックリソース
const size_t InterruptThreadStackSize = nn::os::StackRegionAlignment;
NN_ALIGNAS(nn::os::StackRegionAlignment) char s_Stack[InterruptThreadStackSize];

}

namespace nn {
namespace uart {
namespace driver {
namespace detail {

void InterruptNotifier::Initialize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!m_IsInitialized);
    NN_SDK_ASSERT(!m_IsThreadRunning);

    nn::os::InitializeMultiWait(&m_Waiter);

    for (int i=0; i<InterruptCountMax; i++)
    {
        nn::os::InitializeInterruptEvent(&m_InterruptEvent[i], InterruptNameTable[i], nn::os::EventClearMode_ManualClear);
    }

    for (int i=0; i<PortCountMax; i++)
    {
        m_Callback[i] = nullptr;
        m_CallbackArg[i] = 0;
    }
#ifdef NN_DETAIL_UART_ENABLE_DMA
    for (int i = 0; i < DmaChannelCountMax; i++)
    {
        m_DmaCallback[i] = nullptr;
        m_DmaCallbackArg[i] = 0;
    }
#endif

    // 終了処理用内部イベントの設定
    nn::os::InitializeEvent(&m_FinalizeRequestEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeMultiWaitHolder(&m_FinalizeRequestEventHolder, &m_FinalizeRequestEvent);
    nn::os::LinkMultiWaitHolder(&m_Waiter, &m_FinalizeRequestEventHolder);

    // イベント登録用イベントやセマフォの初期化
    nn::os::InitializeEvent(&m_ConfigStartEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_ConfigStartEventHolder, &m_ConfigStartEvent);
    nn::os::LinkMultiWaitHolder(&m_Waiter, &m_ConfigStartEventHolder);

    nn::os::InitializeEvent(&m_DmaConfigStartEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMultiWaitHolder(&m_DmaConfigStartEventHolder, &m_DmaConfigStartEvent);
    nn::os::LinkMultiWaitHolder(&m_Waiter, &m_DmaConfigStartEventHolder);

    m_ConfigMutex.Initialize();
    m_ConfigCond.Initialize();

    // イベント通知スレッドを作成
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&m_WorkThread, InterruptThread, this,
                             s_Stack, InterruptThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(uart, InterruptHandler))
        );
    nn::os::SetThreadNamePointer(&m_WorkThread, NN_SYSTEM_THREAD_NAME(uart, InterruptHandler));

    m_IsInitialized = true;
}

void InterruptNotifier::Finalize() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);
    NN_SDK_ASSERT(!m_IsThreadRunning);

    m_IsInitialized = false;

    nn::os::DestroyThread(&m_WorkThread);

    nn::os::UnlinkMultiWaitHolder(&m_ConfigStartEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_ConfigStartEventHolder);
    nn::os::FinalizeEvent(&m_ConfigStartEvent);

    nn::os::UnlinkMultiWaitHolder(&m_ConfigStartEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_ConfigStartEventHolder);
    nn::os::FinalizeEvent(&m_ConfigStartEvent);

    nn::os::UnlinkMultiWaitHolder(&m_FinalizeRequestEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_FinalizeRequestEventHolder);
    nn::os::FinalizeEvent(&m_FinalizeRequestEvent);

#ifdef NN_DETAIL_UART_ENABLE_DMA
    for (int i = 0; i < DmaChannelCountMax; i++)
    {
        m_DmaCallback[i] = nullptr;
        m_DmaCallbackArg[i] = 0;
    }
#endif

    for (int i=0; i<PortCountMax; i++)
    {
        m_Callback[i] = nullptr;
        m_CallbackArg[i] = 0;
    }

    for (int i=0; i<InterruptCountMax; i++)
    {
        nn::os::UnlinkMultiWaitHolder(&m_InterruptEventHolder[i]);
        nn::os::FinalizeMultiWaitHolder(&m_InterruptEventHolder[i]);
        nn::os::FinalizeInterruptEvent(&m_InterruptEvent[i]);
    }

    nn::os::FinalizeMultiWait(&m_Waiter);
}

void InterruptNotifier::LinkHandler(int portIndex, InterruptCallbackFuncType callback, uintptr_t arg) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    NN_SDK_ASSERT(0 <= portIndex && portIndex < PortCountMax);


    if (m_Callback[portIndex] != nullptr)
    {
        nn::os::UnlinkMultiWaitHolder(&m_InterruptEventHolder[portIndex]);
        nn::os::FinalizeMultiWaitHolder(&m_InterruptEventHolder[portIndex]);
    }

    m_CallbackArg[portIndex] = arg;
    m_Callback[portIndex] = callback;

    if (callback != nullptr)
    {
        // MultiWait に入れる前に発生している割り込みはクリアしておく
        nn::os::ClearInterruptEvent(&m_InterruptEvent[portIndex]);
        NN_DETAIL_UART_INFO("Clear InterruptEvent before adding to MultiWait\n");

        nn::os::InitializeMultiWaitHolder(&m_InterruptEventHolder[portIndex],
            &m_InterruptEvent[portIndex]);
        nn::os::LinkMultiWaitHolder(&m_Waiter,
            &m_InterruptEventHolder[portIndex]);
        nn::os::SetMultiWaitHolderUserData(&m_InterruptEventHolder[portIndex],
            static_cast<uintptr_t>(portIndex));
    }
}

#ifdef NN_DETAIL_UART_ENABLE_DMA
void InterruptNotifier::LinkDmaHandler(int portIndex, int channelIndex, InterruptCallbackFuncType callback, uintptr_t arg) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    int eventIndex = portIndex * DmaChannelCountPerPort + channelIndex;
    NN_SDK_ASSERT(0 <= eventIndex && eventIndex < DmaChannelCountMax);

    if (m_DmaCallback[eventIndex] != nullptr)
    {
        nn::os::UnlinkMultiWaitHolder(&m_DmaInterruptEventHolder[eventIndex]);
        nn::os::FinalizeMultiWaitHolder(&m_DmaInterruptEventHolder[eventIndex]);
    }

    m_DmaCallbackArg[eventIndex] = arg;
    m_DmaCallback[eventIndex] = callback;

    if (callback != nullptr)
    {
        nn::os::InitializeMultiWaitHolder(&m_DmaInterruptEventHolder[eventIndex],
                                          &m_DmaInterruptEvent[eventIndex]);
        nn::os::LinkMultiWaitHolder(&m_Waiter,
                                    &m_DmaInterruptEventHolder[eventIndex]);
        nn::os::SetMultiWaitHolderUserData(&m_DmaInterruptEventHolder[eventIndex],
                                           static_cast<uintptr_t>(InterruptCountMax + eventIndex));
    }
}
#endif

void InterruptNotifier::SetHandler(int portIndex, InterruptCallbackFuncType callback, uintptr_t arg) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    NN_SDK_ASSERT(0 <= portIndex && portIndex < PortCountMax);

    std::lock_guard<nn::os::SdkMutexType>  lock(m_ConfigMutex);

    // 他のスレッドで設定中になってないことの確認
    NN_SDK_ASSERT(m_PortIndexForConfig == InvalidIndex);

    m_PortIndexForConfig = portIndex;
    m_CallbackArgForConfig[portIndex] = arg;
    m_CallbackForConfig[portIndex] = callback;

    nn::os::SignalEvent(&m_ConfigStartEvent);

    while (m_ConfigFinishedIndex != portIndex)
    {
        m_ConfigCond.Wait(m_ConfigMutex);
    }

    m_ConfigFinishedIndex = InvalidIndex;
    m_PortIndexForConfig = InvalidIndex;
}

#ifdef NN_DETAIL_UART_ENABLE_DMA
void InterruptNotifier::SetDmaHandler(int portIndex, int channelIndex, InterruptCallbackFuncType callback, uintptr_t arg) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    int eventIndex = portIndex * DmaChannelCountPerPort + channelIndex;

    NN_SDK_ASSERT(0 <= eventIndex && eventIndex < DmaChannelCountMax);

    std::lock_guard<nn::os::SdkMutexType>  lock(m_ConfigMutex);

    // 他のスレッドで設定中になってないことの確認
    NN_SDK_ASSERT(m_PortIndexForConfig == InvalidIndex);

    m_PortIndexForConfig    = portIndex;
    m_ChannelIndexForConfig = channelIndex;
    m_DmaCallbackArgForConfig[eventIndex] = arg;
    m_DmaCallbackForConfig[eventIndex] = callback;

    nn::os::SignalEvent(&m_DmaConfigStartEvent);

    while (m_ConfigFinishedIndex != InterruptCountMax + eventIndex)
    {
        m_ConfigCond.Wait(m_ConfigMutex);
    }

    m_ConfigFinishedIndex = InvalidIndex;
    m_PortIndexForConfig = InvalidIndex;
}
#endif

void InterruptNotifier::StartThread() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);
    NN_SDK_ASSERT(!m_IsThreadRunning);
    nn::os::StartThread(&m_WorkThread);

    m_IsThreadRunning = true;
}

void InterruptNotifier::StopThread() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);
    NN_SDK_ASSERT(m_IsThreadRunning);

    // 終了処理用内部イベントにシグナルを送りスレッドを終了させる
    m_IsFinalizeRequested = true;
    nn::os::SignalEvent(&m_FinalizeRequestEvent);
    nn::os::WaitThread(&m_WorkThread);
    m_IsFinalizeRequested = false;

    m_IsThreadRunning = false;
}

void InterruptNotifier::StartThreadForResuming() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);
    NN_SDK_ASSERT(!m_IsThreadRunning);

    // イベント通知スレッドを作成
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&m_WorkThread, InterruptThread, this,
            s_Stack, InterruptThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(uart, InterruptHandler))
    );
    nn::os::SetThreadNamePointer(&m_WorkThread, NN_SYSTEM_THREAD_NAME(uart, InterruptHandler));
    nn::os::StartThread(&m_WorkThread);

    m_IsThreadRunning = true;
}

void InterruptNotifier::StopThreadForSuspending() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);
    NN_SDK_ASSERT(m_IsThreadRunning);

    // 終了処理用内部イベントにシグナルを送りスレッドを終了させる
    m_IsFinalizeRequested = true;
    nn::os::SignalEvent(&m_FinalizeRequestEvent);
    nn::os::WaitThread(&m_WorkThread);
    nn::os::DestroyThread(&m_WorkThread);
    nn::os::ClearEvent(&m_FinalizeRequestEvent);
    m_IsFinalizeRequested = false;

    m_IsThreadRunning = false;
}

// CreateThread に渡すメイン関数（要 static）
// InterruptNotifier のメンバに触りたいので中でメンバ関数を呼び出す
void InterruptNotifier::InterruptThread(void* arg) NN_NOEXCEPT
{
    // メンバ変数を触るために処理本体はメンバ関数で
    InterruptNotifier* notifier = reinterpret_cast<InterruptNotifier*>(arg);
    notifier->InterruptThreadBody();
}

void InterruptNotifier::InterruptThreadBody() NN_NOEXCEPT
{
    while (1)
    {
        // 登録されたイベントの発生またはスレッド終了シグナル待ち
        nn::os::MultiWaitHolderType* pEventHolder = nn::os::WaitAny(&m_Waiter);

        // Finalize 中だったら while を抜けて終了
        if (m_IsFinalizeRequested)
        {
            break;
        }

        auto interruptIdx = static_cast<int>(nn::os::GetMultiWaitHolderUserData(pEventHolder));

        // イベントの設定要求
        if (pEventHolder == &m_DmaConfigStartEventHolder)
        {
            nn::os::ClearEvent(&m_DmaConfigStartEvent);

            int eventIndex = m_PortIndexForConfig * DmaChannelCountPerPort + m_ChannelIndexForConfig;

            LinkDmaHandler(m_PortIndexForConfig, m_ChannelIndexForConfig, m_DmaCallbackForConfig[eventIndex], m_DmaCallbackArgForConfig[eventIndex]);

            std::lock_guard<nn::os::SdkMutexType>  lock(m_ConfigMutex);
            m_ConfigFinishedIndex = eventIndex + InterruptCountMax;
            m_ConfigCond.Signal();
            continue;
        }
        else if (pEventHolder == &m_ConfigStartEventHolder)
        {
            nn::os::ClearEvent(&m_ConfigStartEvent);

            auto portIndex = m_PortIndexForConfig;

            LinkHandler(portIndex, m_CallbackForConfig[portIndex], m_CallbackArgForConfig[portIndex]);

            std::lock_guard<nn::os::SdkMutexType>  lock(m_ConfigMutex);
            m_ConfigFinishedIndex = portIndex;
            m_ConfigCond.Signal();
            continue;
        }

        // 他はすべて割り込みイベントとして対応

        //NN_DETAIL_UART_INFO("Interrupt Come index = %d\n", interruptIdx);
        if (interruptIdx < InterruptCountMax)
        {
            auto interruptName = InterruptNameTable[interruptIdx];
            auto& interruptEvent = m_InterruptEvent[interruptIdx];

            // 発生した割り込み番号に対応するポートの割り込みハンドラをすべて（登録されていれば）呼び出す
            for (int portIdx = 0; portIdx < PortCountMax; portIdx++)
            {
                auto callback = m_Callback[portIdx];
                auto arg = m_CallbackArg[portIdx];
                if (AssignedInterrupt[portIdx] == interruptName)
                {
                    if (callback)
                    {
                        (*callback)(arg);
                    }
                    else
                    {
                        NN_DETAIL_UART_ERROR("Unexpected Interrupt Occurred. callback is null\n");
                    }
                }
            }

            nn::os::ClearInterruptEvent(&interruptEvent);
        }
        else
        {
#ifdef NN_DETAIL_UART_ENABLE_DMA
            auto channelIndex = interruptIdx - InterruptCountMax;
            auto& interruptEvent = m_DmaInterruptEvent[channelIndex];

            auto callback = m_DmaCallback[channelIndex];
            auto arg = m_DmaCallbackArg[channelIndex];
            if (callback)
            {
                (*callback)(arg);
                nn::os::ClearInterruptEvent(&interruptEvent);
            }

            // DMA の InterruptEvent の Finalize 後にここに来るパスが存在するため、
            // ここではクリアしない。(SIGLO-37487)
            // nn::os::ClearInterruptEvent(&interruptEvent);
#endif
        }
    }
}

} // detail
} // driver
} // uart
} // nn
