﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Macro.h>

#include "uart_CircularBuffer.h"

namespace nn {
namespace uart {
namespace driver {
namespace detail {

size_t CircularBuffer::Write(const char* data, size_t count) NN_NOEXCEPT
{
    size_t currentWritePos = m_WritePos;
    const size_t tailSpace = m_BufferLength - currentWritePos;
    const size_t writableLength = GetWritableLength();
    size_t lengthToWrite = count;

    if (lengthToWrite == 0)
    {
        return 0;
    }

    if (lengthToWrite > writableLength)
    {
        lengthToWrite = writableLength;
    }

    if (tailSpace <= lengthToWrite)
    {
        /* Wrap */
        std::memcpy(m_Buffer + currentWritePos, data, tailSpace);
        currentWritePos = lengthToWrite - tailSpace;
        std::memcpy(m_Buffer, data + tailSpace, currentWritePos);
    }
    else
    {
        /* No Wrap */
        std::memcpy(m_Buffer + currentWritePos, data, lengthToWrite);
        currentWritePos += lengthToWrite;
    }

    m_WritePos = currentWritePos;
    if (lengthToWrite > 0)
    {
        m_IsEmpty = false;
    }
    return lengthToWrite;
}

size_t CircularBuffer::ReadAt(size_t *position, char* data, size_t count) NN_NOEXCEPT
{
    size_t  currentReadPos = (*position >= m_BufferLength) ? m_ReadPos : *position;
    const size_t tailSpace = m_BufferLength - currentReadPos;
    const size_t readableLength = IsFull() ? m_BufferLength
                                           : wrapPosition(m_WritePos + m_BufferLength - currentReadPos);
    size_t  lengthToRead = count;

    if (lengthToRead == 0)
    {
        return 0;
    }

    if (lengthToRead > readableLength)
    {
        lengthToRead = readableLength;
    }

    if (tailSpace <= lengthToRead)
    {
        /* Wrap */
        std::memcpy(data, m_Buffer + currentReadPos, tailSpace);
        currentReadPos = lengthToRead - tailSpace;
        std::memcpy(data + tailSpace, m_Buffer, currentReadPos);
    }
    else
    {
        /* No Wrap */
        std::memcpy(data, m_Buffer + currentReadPos, lengthToRead);
        currentReadPos += lengthToRead;
    }

    *position = currentReadPos;
    return lengthToRead;
}

size_t CircularBuffer::Overwrite(bool *pWasOverwritten, const char* data, size_t count) NN_NOEXCEPT
{
    const size_t writableLength = GetWritableLength();
    size_t lengthToWrite = count;

    if (lengthToWrite == 0)
    {
        return 0;
    }

    bool isOverwriting = (lengthToWrite > writableLength);

    // If specified size is bigger than the whole buffer, write last part only
    if (lengthToWrite >= m_BufferLength)
    {
        data  = data + lengthToWrite - m_BufferLength;
        lengthToWrite = m_BufferLength;

        // この時はバッファ内のデータがすべて上書きされるので、書き込む前には IsEmpty を true にする。
        // これを行わないと、ぴったりバッファ分書き込むときに WritableLength が 0 になってしまう。
        m_IsEmpty = true;
    }

    if (isOverwriting)
    {
        // Force-push the read counter to fit in
        m_ReadPos = wrapPosition(m_ReadPos + lengthToWrite - writableLength);
    }
    if (pWasOverwritten)
    {
        *pWasOverwritten = isOverwriting;
    }

    size_t written = Write(data, lengthToWrite);
    NN_SDK_ASSERT(written == lengthToWrite, "[uart] written = %d, lengthToWrite = %d, writableLength = %d, m_BufferLength = %d\n", written, lengthToWrite, writableLength, m_BufferLength);
    NN_UNUSED(written); // Escape warning for Release build

    return count;
}

} // detail
} // driver
} // uart
} // nn
