﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tmipc_thread.h"
#include "tmipc_result.h"

#include "..\tma\tma_MemMgr.h"
#include "..\tmagent.h"

//==============================================================================
namespace tmipc {
//==============================================================================

// The maximum number of objects the MultiWait will allow.  This is an arbitrary
// number that *should be* quite a bit larger than needed...
// Windows defines this value as 64.
static const s32 MAXIMUM_WAIT_OBJECTS = 64;

Mutex::Mutex()
:   m_Created( false )
{
}

//==============================================================================

Mutex::~Mutex()
{
    // Free resources.
    if( m_Created )
    {
        Destroy();
    }
}

//==============================================================================

void Mutex::Create()
{
    ASSERT( !m_Created );
    nn::os::InitializeMutex( &m_Mutex, true, 0 );
    m_Created = true;
}

//==============================================================================

void Mutex::Destroy()
{
    ASSERT( m_Created );
    nn::os::FinalizeMutex( &m_Mutex );
    m_Created = false;
}

//==============================================================================

void Mutex::Lock()
{
    ASSERT( m_Created );
    nn::os::LockMutex( &m_Mutex );
}

//==============================================================================

void Mutex::Unlock()
{
    ASSERT( m_Created );
    nn::os::UnlockMutex( &m_Mutex );
}

//==============================================================================
//==============================================================================

Event::Event()
:   m_Created( false )
{
}

//==============================================================================

Event::~Event()
{
    if( m_Created )
    {
        Destroy();
    }
}

//==============================================================================

void Event::Create()
{
    ASSERT( !m_Created );
    nn::os::InitializeEvent( &m_Event, false, nn::os::EventClearMode_ManualClear );
    m_Created = true;
}

//==============================================================================

void Event::Destroy()
{
    ASSERT( m_Created );
    nn::os::FinalizeEvent( &m_Event );
    m_Created = false;
}

//==============================================================================

void Event::Set()
{
    ASSERT( m_Created );
    nn::os::SignalEvent( &m_Event );
}

//==============================================================================

void Event::Reset()
{
    ASSERT( m_Created );
    nn::os::ClearEvent( &m_Event );
}

//==============================================================================

s32 Event::Wait( s32 Timeout )
{
    ASSERT( m_Created );
    nn::os::WaitEvent( &m_Event );
    return( TMIPC_RESULT_OK );
}

//==============================================================================
//==============================================================================

MultiEventWait::MultiEventWait()
{
    nn::os::InitializeMultiWait( &m_MultiWait );
    m_pWaitHolders   = NULL;
    m_pEvents        = NULL;
    m_NumberOfEvents = 0;
}

//==============================================================================

MultiEventWait::~MultiEventWait()
{
    Deallocate();
    nn::os::FinalizeMultiWait(&m_MultiWait);
}

//==============================================================================
void MultiEventWait::Deallocate()
{
    // Unlink the MultiWaitHolders from the MultiWait struct
    for( s32 Index = 0; Index < m_NumberOfEvents; Index++ )
    {
        nn::os::UnlinkMultiWaitHolder( m_pWaitHolders + Index );
    }
    // Finalize the MultiWaitHolders
    for( s32 Index = 0; Index < m_NumberOfEvents; Index++ )
    {
        nn::os::FinalizeMultiWaitHolder( m_pWaitHolders + Index );
    }

    tma::s_Deallocate( m_pWaitHolders, sizeof( nn::os::MultiWaitHolderType ) * m_NumberOfEvents );
    m_pWaitHolders = NULL;
//TODO: Determine if the m_MultiWait object needs to be destroyed here and
//      re-initialized again, or if the FinalizeMultiWaitHolder(...) is
//      enough to clean up m_MultiWait...
//    nn::os::FinalizeMultiWait(&m_MultiWait);

    tma::s_Deallocate( m_pEvents, sizeof(tmipc::Event) * m_NumberOfEvents );
    m_pEvents = NULL;

    m_NumberOfEvents = 0;
}

//==============================================================================

// A [very] common case: exactly two Events to wait for.
s32 MultiEventWait::SetTwoEvents( Event* pEvent1, Event* pEvent2)
{
    s32 Result = TMIPC_RESULT_OK;

    // Validate the parameters.
    if( Result == TMIPC_RESULT_OK )
    {
        if( (pEvent1 == NULL) || (pEvent2 == NULL) )
        {
//TODO: this should be: tmapi::result::RESULT_INVALID_PARAMETER.
            Result = TMIPC_RESULT_FAILED;
        }
    }

    // Set up the array(s) to be exactly two elements for each parameter.
    if( Result == TMIPC_RESULT_OK )
    {
        // Deallocate the current MultiWaitHolder(s) and Event(s).
        Deallocate();

        m_NumberOfEvents = 2;

        void* pMem = tma::s_Allocate( sizeof( Event* ) * m_NumberOfEvents );
        m_pEvents = new (pMem) Event*[m_NumberOfEvents];
        m_pEvents[0] = pEvent1;
        m_pEvents[1] = pEvent2;

        // Allocate the new MultiWaitEvents array
        pMem = tma::s_Allocate( sizeof( nn::os::MultiWaitHolderType ) * m_NumberOfEvents );
        m_pWaitHolders = new (pMem) nn::os::MultiWaitHolderType[m_NumberOfEvents];
        for (s32 Index = 0; Index < m_NumberOfEvents; Index++)
        {
            nn::os::InitializeMultiWaitHolder( m_pWaitHolders + Index, &m_pEvents[Index]->m_Event );
            nn::os::LinkMultiWaitHolder( &m_MultiWait, m_pWaitHolders + Index );

            // Setting the user data to "Index" will help simulate the results
            // of a ::WaitForMultipleObjects(...) Win32 function call.
            nn::os::SetMultiWaitHolderUserData( m_pWaitHolders + Index, Index );
        }
    }

//TODO: This needs to be implemented.  Do so after the Win32 version is working.
    Result = TMIPC_RESULT_NOT_IMPLEMENTED;

    return Result;
}

//==============================================================================

// Allow the user to add an arbitrary number of Events to wait for.
s32 MultiEventWait::AddEvent( Event* pEvent )
{
    s32 Result = TMIPC_RESULT_OK;

    // Make sure there are not too many Events.
    if( Result == TMIPC_RESULT_OK )
    {
        if( m_NumberOfEvents >= MAXIMUM_WAIT_OBJECTS )
        {
//TODO: Determine if this is a correct response.  It should be in the tmapi::result enum.
            Result = TMIPC_RESULT_MAX;
        }
    }

    // Add the Event.
    if( Result == TMIPC_RESULT_OK )
    {
        // Allocate and copy the existing Events to the new array
        void* pMem = tma::s_Allocate( sizeof( Event ) * ( m_NumberOfEvents + 1 ) );
        Event** pNewEvents = new (pMem) Event*[m_NumberOfEvents + 1];
        for( s32 Index = 0; Index < m_NumberOfEvents; Index++ )
        {
            pNewEvents[Index] = m_pEvents[Index];
        }

        // Add the new Event to the array.
        pNewEvents[m_NumberOfEvents] = pEvent;
        s32 NewNumberOfEvents = m_NumberOfEvents + 1;

        // Deallocate the current event(s) and MultiWaitHolder(s)
        Deallocate();

        // Allocate the new MultiWaitEvents array
        pMem = tma::s_Allocate( sizeof( nn::os::MultiWaitHolderType ) * ( NewNumberOfEvents ) );
        nn::os::MultiWaitHolderType *pWaitHolders = new (pMem) nn::os::MultiWaitHolderType[NewNumberOfEvents];
        for( s32 Index = 0; Index < NewNumberOfEvents; Index++ )
        {
            nn::os::InitializeMultiWaitHolder( pWaitHolders + Index, &pNewEvents[Index]->m_Event );
            nn::os::LinkMultiWaitHolder( &m_MultiWait, pWaitHolders + Index );
            // Setting the user data to "Index" will help simulate the results
            // of a ::WaitForMultipleObjects(...) Win32 function call.
            nn::os::SetMultiWaitHolderUserData( pWaitHolders + Index, Index );
        }

        // Replace the existing Event array with the new array.
        m_pWaitHolders   = pWaitHolders;
        m_pEvents        = pNewEvents;
        m_NumberOfEvents = NewNumberOfEvents;
    }

//TODO: This needs to be implemented.  Do so after the Win32 version is working.
    Result = TMIPC_RESULT_NOT_IMPLEMENTED;

    return Result;
}

//==============================================================================

s32 MultiEventWait::Wait( s32& SignalledIndex )
{
    s32 Result = TMIPC_RESULT_OK;

    // Default the out parameter.
    SignalledIndex = -1;

    // Ensure there is at least one Event to wait for...
    if( Result == TMIPC_RESULT_OK )
    {
        if( m_NumberOfEvents < 1 )
        {
            Result = TMIPC_RESULT_FAILED;
        }
    }

    // Do the wait.
    nn::os::MultiWaitHolderType *pSignalledHolder = { nullptr };
    if( Result == TMIPC_RESULT_OK )
    {
        // The WaitAny returns the "MultiWaitHolderType" that was signalled.
        // Use the signalled return value from WaitAny to retrieve the user
        // data (in this case, the 'event index').
        pSignalledHolder = nn::os::WaitAny( &m_MultiWait );
    }

    // Process the returned value.
    if( Result == TMIPC_RESULT_OK )
    {
        // A nullptr means the wait timed out.
        if( pSignalledHolder != nullptr )
        {
            SignalledIndex = nn::os::GetMultiWaitHolderUserData( pSignalledHolder );
        }
        else
        {
            Result = TMIPC_RESULT_TIMEOUT;
        }
    }

    return Result;
}

//==============================================================================

s32 MultiEventWait::Wait( s32 TimeoutMS, s32& SignalledIndex )
{
    s32 Result = TMIPC_RESULT_OK;

    // Default the out parameter.
    SignalledIndex = -1;

    // Ensure there is at least one Event to wait for...
    if( Result == TMIPC_RESULT_OK )
    {
        if( m_NumberOfEvents < 1 )
        {
            Result = TMIPC_RESULT_FAILED;
        }
    }

    // Do the wait.
    nn::os::MultiWaitHolderType *pSignalledHolder = { nullptr };
    if( Result == TMIPC_RESULT_OK )
    {
        // The WaitAny returns the "MultiWaitHolderType" that was signalled.
        // Use the signalled return value from WaitAny to retrieve the user
        // data (in this case, the 'event index').
        nn::TimeSpan TimeoutSpan = { nn::TimeSpan::FromMilliSeconds( TimeoutMS ) };
        pSignalledHolder = nn::os::TimedWaitAny( &m_MultiWait, TimeoutSpan );
    }

    // Process the returned value.
    if( Result == TMIPC_RESULT_OK )
    {
        // A nullptr means the wait timed out.
        if( pSignalledHolder != nullptr )
        {
            SignalledIndex = nn::os::GetMultiWaitHolderUserData( pSignalledHolder );
        }
        else
        {
            Result = TMIPC_RESULT_TIMEOUT;
        }
    }

    return Result;
}

//==============================================================================
//==============================================================================

Thread::Thread()
//:   m_hThread   ( INVALID_HANDLE_VALUE )
//,   m_ThreadId  ( 0 )
{
    memset(&m_Thread, 0, sizeof(m_Thread));
}

//==============================================================================

Thread::~Thread()
{
}

//==============================================================================

tmipc::Result Thread::Start( ThreadFunction* pFn, void* pArg, void* pStack, size_t StackSize, s32 Priority, const char* pName )
{
    TMA_TRACE( "Thread", "Start Pri:%d", Priority );

    ASSERT( ((u64)pStack & (nn::os::StackRegionAlignment - 1)) == 0 );

    // Create a new system thread.
    nn::Result Result = nn::os::CreateThread( &m_Thread, (nn::os::ThreadFunction)pFn, (void*)pArg, pStack, StackSize, Priority );
    ASSERT( Result.IsSuccess() );

    // Set the thread's name.
    nn::os::SetThreadName( &m_Thread, pName );

    nn::os::StartThread( &m_Thread );

    return( TMIPC_RESULT_OK );
}

//==============================================================================

tmipc::Result Thread::Join( s32 Timeout )
{
    // Wait for the thread to exit.
    u8 State = m_Thread._state;
    if( ( State == nn::os::ThreadType::State_Initialized ) ||
        ( State == nn::os::ThreadType::State_Started ) ||
        ( State == nn::os::ThreadType::State_Exited) )
    {
        nn::os::WaitThread( &m_Thread );
        nn::os::DestroyThread( &m_Thread );
    }

    return( TMIPC_RESULT_OK );
}

//==============================================================================

void Thread::SleepMs( s32 Ms )
{
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( Ms ) );
}

//==============================================================================
//==============================================================================

ScopedLock::ScopedLock( Mutex& Mutex )
:   m_pMutex( &Mutex )
{
    m_pMutex->Lock();
}

//==============================================================================

ScopedLock::~ScopedLock()
{
    m_pMutex->Unlock();
}

//==============================================================================
//==============================================================================

MsgQueue::MsgQueue( )
:   m_Created   ( false ),
    m_pQueue    ( NULL  ),
    m_Allocated ( false ),
    m_Capacity  ( 0     ),
    m_Count     ( 0     ),
    m_iRead     ( 0     ),
    m_iWrite    ( 0     )
{
}

//==============================================================================

MsgQueue::~MsgQueue( )
{
    if( m_Created )
    {
        Destroy();
    }

    if( m_Allocated )
    {
        tma::s_Deallocate( m_pQueue, sizeof( void* ) * m_Capacity );
    }
}

//==============================================================================

void MsgQueue::Create( s32 Capacity )
{
    ASSERT( !m_Created );
    ASSERT( Capacity > 0 );

    // Save the capacity
    m_Capacity = Capacity;

    // Allocate storage
    void* pMem = tma::s_Allocate( sizeof( void* ) * ( m_Capacity ) );
    m_pQueue = new (pMem) void*[m_Capacity];
    ASSERT( m_pQueue );
    m_Allocated = true;

    // Create the message queue
    nn::os::InitializeMessageQueue( &m_q, (uintptr_t*)m_pQueue, Capacity );

    m_Mutex.Create();

    // Created
    m_Created = true;
}

//==============================================================================

void MsgQueue::Create( s32 Capacity, void** pStorage )
{
    ASSERT( !m_Created );
    ASSERT( Capacity > 0 );

    // Save the capacity
    m_Capacity = Capacity;

    // Set the queue pointer to the provided storage
    ASSERT( pStorage );
    m_pQueue = pStorage;
    m_Allocated = false;

    // Create the message queue
    nn::os::InitializeMessageQueue( &m_q, (uintptr_t*)m_pQueue, Capacity );

    m_Mutex.Create();

    // Created
    m_Created = true;
}

//==============================================================================

void MsgQueue::Destroy()
{
    ASSERT( m_Created );

    nn::os::FinalizeMessageQueue( &m_q );

    m_Mutex.Destroy();

    m_Created = false;
}

//==============================================================================

bool MsgQueue::IsValid()
{
    return m_Created;
}

//==============================================================================

bool MsgQueue::Send( void* pMessage, bool Block )
{
    (void)Block;

    ASSERT( m_Created );

    if( Block )
    {
        nn::os::SendMessageQueue( &m_q, (uintptr_t)pMessage );
        m_Mutex.Lock();
        ++m_Count;
        m_Mutex.Unlock();
        return( true );
    }
    else
    {
        bool Success = nn::os::TrySendMessageQueue( &m_q, (uintptr_t)pMessage );
        m_Mutex.Lock();
        ++m_Count;
        m_Mutex.Unlock();
        return( Success );
    }
}

//==============================================================================

bool MsgQueue::Receive( void** ppMessage, bool Block )
{
    (void)Block;

    ASSERT( m_Created );
    ASSERT( ppMessage );

    if( Block )
    {
        uintptr_t outData;
        nn::os::ReceiveMessageQueue( &outData, &m_q );
        *ppMessage = (void*)outData;
        m_Mutex.Lock();
        --m_Count;
        m_Mutex.Unlock();
        return( true );
    }
    else
    {
        uintptr_t outData;
        bool Success = nn::os::TryReceiveMessageQueue( &outData, &m_q );
        if( Success )
        {
            m_Mutex.Lock();
            --m_Count;
            m_Mutex.Unlock();
            *ppMessage = (void*)outData;
        }
        return( Success );
    }
}

//==============================================================================

void* MsgQueue::Receive()
{
    void* pMessage;
    bool bReceived = Receive( &pMessage );
    (void)bReceived;
    ASSERT( bReceived );
    return( pMessage );
}

//==============================================================================

s32 MsgQueue::GetCount() const
{
    s32 Count = m_Count;
    return Count;
}

//==============================================================================
} // namespace tmipc
//==============================================================================
