﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "tmipc_defines.h"
#include "tmipc_thread.h"
#include "tmipc_tasklist.h"
#include "tmipc_result.h"
#include <string>

#ifdef TMIPC_TARGET_HORIZON
#include "..\tma\tma_MemMgr.h"
#endif

#include "..\tmagent.h"

//==============================================================================
namespace tmipc {
//==============================================================================

class Node;
class Packet;
class Service;
class Task;
struct WorkItem;

//==============================================================================

class ServicesManager
{
private:
        bool                m_Initialized;
        Mutex               m_Lock;
        Service*            m_pFirstService;
        MsgQueue            m_FreeSendPacketQ;
        MsgQueue            m_FreeRecvPacketQ;

        s32                 m_ThreadPriority;
        TaskList            m_TaskList;
        Thread              m_WorkThread;
        MsgQueue            m_WorkFreeQ;
        MsgQueue            m_WorkQ;
        void*               m_WorkFreeQueueStorage[TMIPC_WORK_QUEUE_DEPTH];
        void*               m_WorkQueueStorage[TMIPC_WORK_QUEUE_DEPTH];

        u32                 m_NodeId;                   // Server / Client identifier
        u32                 m_NextTaskId;               // Next Task ID.

        size_t              m_WorkThreadStackSize;
        void*               m_pWorkThreadStack;

        Event               m_DisconnectEvent;

        Event               m_WakeUpEvent;
        Event               m_SleepEvent;
        bool                m_IsSleeping;
        bool                m_bPreserveStateThroughSleepWake;
        Node*               m_pNode;                    // The Node "protocol" to transmit/receive Packets with TargetManager.

public:
virtual                    ~ServicesManager     ();
                            ServicesManager     ();

        void                Init                ();
        void                Kill                ();
        void                DiagnosticReport    ();

        // This should only be called *after* GoToSleep() has been called.
        void                WakeUp              ( Node*             pNode   );
        // This is a blocking call, until the WorkThread processes the "Sleep request".
        void                GoToSleep           ();
        bool                IsSleeping() const;
        bool                IsConnected() const;
        // This function lets PowerManagementCallback (tmagent.cpp) know to wait for a connection before responding to a Minimum Awake Power Event..
        // This is a bit hacky, but it helps prevent tasks from entering the WorkThread and getting cancelled because a connection has not been made
        // to Target Manager.  (See Siglo-55743 for an example.)
        bool                PreserveStateThroughSleepWake() const;
        // This function should *only* be called by the PowerManagementCallback (tmagent.cpp) function, to indicate that "the flag" is no longer needed.
        void                ResetPreserveStateThroughSleepWake();

        Result              AddService          ( Service*          pService );
        void                RemoveServices      ();

        Service*            GetService          ( u32               ServiceId );
        u32                 AllocTaskId         ();
        Packet*             AllocSendPacket     ();
        Packet*             AllocRecvPacket     ();
        void                FreePacket          ( Packet*           pPacket );
        Result              SubmitTask          ( Task*             pTask,
                                                  Packet*           pPacket );
        Result              FreeTask            ( Task*             pTask );
        // This is a dangerous function.  Do not use this unless you are aware of the consquences of "externally owned" task flag...
        void                CancelTask          ( u32               TaskId);
        void                CancelAllTasks      ();
        void                Tick                ();
        void                Disconnected        ();

        void                SetThreadPriority   ( s32               Priority );
        s32                 GetThreadPriority   ();

        void                SetNode             ( Node* pNode );
        Result              Send                ( Packet* pPacket );
        void                ProcessReceived     ( Packet* pPacket );

        // Returns the number of Packets that have been allocated for: Sending and Receiving.
        void                GetAllocatedPacketCounts( s32& Sending, s32& Receiving ) const;
        // Returns the number of outstanding Tasks (not completed and not cancelled).
        void                GetCurrentTaskCount     ( s32& Count ) const;
        // Returns the number of outstanding m_WorkQ items.
        void                GetCurrentWorkQueueCount( s32& Count ) const;

private:
static  s32                 WorkThread          ( void*             pArg );
        void                ProcessNewTask      ( WorkItem*         pItem );
        void                ProcessFreeTask     ( WorkItem*         pItem );
        void                ProcessRecvPacket   ( WorkItem*         pItem );
        void                ProcessTick         ();
        void                ProcessDisconnected ();
        void                ProcessDeleteServices();
        void                ProcessSleep        ();
};

//==============================================================================
} // namespace tmipc
//==============================================================================
