﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>

#include <nn/nn_SystemThreadDefinition.h>

#include <nn/socket.h>

#include <nn/result/result_HandlingUtility.h>

#include "tmipc_node_htclow.h"
#include "tmipc_result.h"
#include "tmipc_packet.h"
#include "tmipc_service.h"
#include "tmipc_task.h"
#include "../DejaInsight.h"
#include "../Version.h"

namespace tmipc {

Thread NodeHtclow::m_ListenThread;
Thread NodeHtclow::m_SendThread;
Thread NodeHtclow::m_RecvThread;

NN_OS_ALIGNAS_THREAD_STACK u8 NodeHtclow::m_ListenThreadStack[TMIPC_STACK_SIZE];
NN_OS_ALIGNAS_THREAD_STACK u8 NodeHtclow::m_SendThreadStack[TMIPC_STACK_SIZE];
NN_OS_ALIGNAS_THREAD_STACK u8 NodeHtclow::m_RecvThreadStack[TMIPC_STACK_SIZE];

Result NodeHtclow::Initialize() NN_NOEXCEPT
{
    nn::htclow::Initialize();
    return TMIPC_RESULT_OK;
}

Result NodeHtclow::Finalize() NN_NOEXCEPT
{
    return TMIPC_RESULT_OK;
}

NodeHtclow::NodeHtclow() NN_NOEXCEPT
    : m_IsConnected(false)
    , m_IsListening(false)
    , m_StopListening(false)
{
    m_DisconnectLock.Create();
    m_SendQueue.Create(SendQueueSize);
    m_ListenEvent.Create();
    m_ListenEvent.Reset();

    Init();
}

NodeHtclow::~NodeHtclow() NN_NOEXCEPT
{
    Disconnect();
    StopListening();

    m_HtclowChannel.Close();

    m_ListenEvent.Destroy();
    m_SendQueue.Destroy();
    m_DisconnectLock.Destroy();
}

Result NodeHtclow::Listen() NN_NOEXCEPT
{
    Result result = (TMIPC_RESULT_OK);

    m_StopListening = false;

    if (!m_IsListening)
    {
        // Start listening.
        result = m_ListenThread.Start(ListenThreadEntry, this, &m_ListenThreadStack, sizeof(m_ListenThreadStack),
            NN_SYSTEM_THREAD_PRIORITY(tma, TmaHtclowListen), NN_SYSTEM_THREAD_NAME(tma, TmaHtclowListen) );
        m_IsListening = true;
    }

    return result;
}

void NodeHtclow::StopListening() NN_NOEXCEPT
{
    if (!m_IsListening)
    {
        return;
    }

    m_StopListening = true;

    m_HtclowChannel.Shutdown();
    m_HtclowChannel.Close();

    m_ListenEvent.Set();

    VERIFY(m_ListenThread.Join() == Result::TMIPC_RESULT_OK);
    m_IsListening = false;
}

Result NodeHtclow::Disconnect() NN_NOEXCEPT
{
    {
        // Disconnected() を2回以上呼び出さないためのロック
        ScopedLock lock(m_DisconnectLock);

        if (!m_IsConnected)
        {
            return TMIPC_RESULT_OK;
        }

        m_IsConnected = false;
    }

    // Trigger the Listen to "iterate to the next session".
    m_ListenEvent.Set();

    Disconnected();

    return TMIPC_RESULT_OK;
}

bool NodeHtclow::IsConnected() NN_NOEXCEPT
{
    return m_IsConnected;
}

Result NodeHtclow::Send(Packet* pPacket) NN_NOEXCEPT
{
    {
        ScopedLock lock(m_Lock);

        if (IsConnected())
        {
            // Add the packet to the Message Queue.
            m_SendQueue.Send(pPacket);

            // All good.
            return (TMIPC_RESULT_OK);
        }
        else
        {
            // No connection, just free the packet.
            FreePacket(pPacket);
        }
    }

    // Need to Tick the WorkThread if we get here to process the FreePacket.
    Tick();

    // We are diconnected.
    return (TMIPC_RESULT_DISCONNECTED);
}

s32 NodeHtclow::ListenThreadEntry(void* pArg) NN_NOEXCEPT
{
    reinterpret_cast<NodeHtclow*>(pArg)->ListenThread();

    // Thread exit.
    return 0;
}

s32 NodeHtclow::RecvThreadEntry(void* pArg) NN_NOEXCEPT
{
    reinterpret_cast<NodeHtclow*>(pArg)->RecvThread();

    // Thread exit.
    return 0;
}

s32 NodeHtclow::SendThreadEntry(void* pArg) NN_NOEXCEPT
{
    reinterpret_cast<NodeHtclow*>(pArg)->SendThread();

    // Thread exit.
    return 0;
}

nn::Result NodeHtclow::ListenThreadBody() NN_NOEXCEPT
{
    // Fill in the Beacon Response - this should not change for the duration
    // of this thread.
    const int bufferSize = 256;
    char beaconResponse[bufferSize];

    // Target Manager に対しては TCP 接続だと見せかける
    NN_ABORT_UNLESS(tma::GenerateBeaconResponse(beaconResponse, sizeof(beaconResponse), "TCP", true) == true);

    const s32 beaconResponseLength = static_cast<s32>(strlen(beaconResponse) + 1);

    // connect htclow
    NN_RESULT_DO(m_HtclowChannel.Connect());

    // Read the remote version number.
    u32 remoteVersionBuffer;

    NN_RESULT_DO(HtclowRecv(&remoteVersionBuffer, sizeof(remoteVersionBuffer)));

    const u32 remoteVersion = nn::socket::InetNtohl(remoteVersionBuffer);

    // Send the local version number.
    const u32 localVersion = TMIPC_VERSION;
    u32 localVersionBuffer = nn::socket::InetHtonl(localVersion);

    NN_RESULT_DO(HtclowSend(&localVersionBuffer, sizeof(localVersionBuffer)));

    if (remoteVersion != localVersion)
    {
        TMIPC_ERROR_LOG( "Version number mismatch detected.\n" );

        return nn::htclow::ResultTmModuleError();
    }

    // Send the beacon response length first.
    s32 beaconResponseLengthBuffer = nn::socket::InetHtonl(beaconResponseLength);

    NN_RESULT_DO(HtclowSend(&beaconResponseLengthBuffer, sizeof(beaconResponseLengthBuffer)));

    // Send the beacon response string.
    NN_RESULT_DO(HtclowSend(&beaconResponse, beaconResponseLength));

    // Handshake done.
    m_IsConnected = true;

    // Start the send/receive threads.
    StartThreads();

    // Wait for the Listen event to trigger - either through a SendThread
    // shutdown, or a StopListening().
    m_ListenEvent.Wait(TMIPC_INFINITE);

    // Disconnect.
    Disconnect();

    // Stop threads.
    StopThreads();

    // Reset the Listen event.
    m_ListenEvent.Reset();

    NN_RESULT_SUCCESS;
}

void NodeHtclow::ListenThread() NN_NOEXCEPT
{
    while (!m_StopListening)
    {
        auto module = nn::htclow::Module(HtclowModuleId);
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_HtclowChannel.Open(&module, HtclowChannelId));

        ListenThreadBody();

        m_HtclowChannel.Close();
    }
}

void NodeHtclow::SendThread() NN_NOEXCEPT
{
    // Loop until the htclow channel closes.
    while (IsConnected())
    {
        // Read a packet from the Message Queue.
        auto pPacket = reinterpret_cast<Packet*>(m_SendQueue.Receive());

        // Terminate thread if a nullptr packet.
        if (!pPacket)
        {
            break;
        }

        // Send the packet over the htclow.
        if (IsConnected())
        {
            u8* pData = pPacket->GetBuffer();
            s32 length = pPacket->GetSendLen();

            auto result = HtclowSend(pData, length);
            if (result.IsFailure())
            {
                break;
            }
        }

        // Free the packet.
        FreePacket(pPacket);
        Tick();
    }

    // Disconnect the node.
    Disconnect();

    // Delete any pending packets.
    while (m_SendQueue.GetCount() > 0 )
    {
        auto pPacket = reinterpret_cast<Packet*>(m_SendQueue.Receive());
        if (pPacket)
        {
            FreePacket(pPacket);
        }
    }

    // ASSERT no more packets.
    ASSERT(m_SendQueue.GetCount() == 0);

    // Trigger the Listen to "iterate to the next session".
    m_ListenEvent.Set();
}

void NodeHtclow::RecvThread() NN_NOEXCEPT
{
    // Loop until the htclow closes.
    while (IsConnected())
    {
        nn::Result result;

        // Allocate a packet.
        Packet* pPacket = AllocRecvPacket();
        ASSERT(pPacket);

        // Read packet from htclow.
        u8* pBuffer = pPacket->GetBuffer();

        // Read the packet header.
        result = HtclowRecv(pBuffer, sizeof(Packet::Header));
        if (result.IsFailure())
        {
            // If we didn't get a full header then we lost connection.
            FreePacket(pPacket);
            break;
        }

        s32 dataLen = pPacket->GetDataLen();
        if (dataLen > 0)
        {
            // Read the packet contents.
            result = HtclowRecv(&pBuffer[sizeof(Packet::Header)], dataLen);
            if (result.IsFailure())
            {
                // If we didn't get all the data then we lost connection.
                FreePacket(pPacket);
                break;
            }
        }

        // Dispatch the packet.
        ProcessReceived(pPacket);
        pPacket = nullptr;
    }

    // No longer connected.
    OnEvent(NodeEvent_Disconnected);
    Disconnect();

    // Send a nullptr to terminate the SendThread.
    m_SendQueue.Send(nullptr);
}

void NodeHtclow::StartThreads() NN_NOEXCEPT
{
    // Cancel any existing blocking calls.
    VERIFY(m_RecvThread.Join() == Result::TMIPC_RESULT_OK);
    VERIFY(m_SendThread.Join() == Result::TMIPC_RESULT_OK);

    // Start Send/Receive threads.
    m_RecvThread.Start(RecvThreadEntry, this, &m_RecvThreadStack, sizeof(m_RecvThreadStack),
        NN_SYSTEM_THREAD_PRIORITY(tma, TmaHtclowRecv), NN_SYSTEM_THREAD_NAME(tma, TmaHtclowRecv));
    m_SendThread.Start(SendThreadEntry, this, &m_SendThreadStack, sizeof(m_SendThreadStack),
        NN_SYSTEM_THREAD_PRIORITY(tma, TmaHtclowSend), NN_SYSTEM_THREAD_NAME(tma, TmaHtclowSend));
}

void NodeHtclow::StopThreads() NN_NOEXCEPT
{
    // Cancel any existing blocking calls.
    VERIFY(m_RecvThread.Join() == Result::TMIPC_RESULT_OK);
    VERIFY(m_SendThread.Join() == Result::TMIPC_RESULT_OK);
}

nn::Result NodeHtclow::HtclowSend(const void* pBuffer, size_t bufferSize) NN_NOEXCEPT
{
    nn::Result result;
    size_t sendSize;

    // Send length
    NN_ABORT_UNLESS(bufferSize < std::numeric_limits<int32_t>::max(), "Numeric overflow happend.");
    int32_t bufferSizeInt32 = static_cast<int32_t>(bufferSize);

    result = m_HtclowChannel.Send(&sendSize, &bufferSizeInt32, sizeof(int32_t));
    if (result.IsFailure() || sendSize != sizeof(int32_t))
    {
        return nn::htclow::ResultTmModuleError();
    }

    // Send data
    result = m_HtclowChannel.Send(&sendSize, pBuffer, bufferSize);
    if (result.IsFailure() || sendSize != bufferSize)
    {
        return nn::htclow::ResultTmModuleError();
    }

    NN_RESULT_SUCCESS;
}

nn::Result NodeHtclow::HtclowRecv(void* pOutBuffer, size_t bufferSize) NN_NOEXCEPT
{
    size_t size;

    nn::Result result = m_HtclowChannel.Receive(&size, pOutBuffer, bufferSize);
    if (result.IsFailure() || size != bufferSize)
    {
        return nn::htclow::ResultTmModuleError();
    }

    NN_RESULT_SUCCESS;
}

}
