﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\tmagent.h"
#include "tma_Thread.h"

#if defined(NN_TMA_SIM_INSTANCE)
#include <nn/nn_Windows.h>
#endif

//==============================================================================
namespace tma {
//==============================================================================

//==============================================================================

Thread::Thread()
:   m_pStack        ( NULL )
,   m_AllocatedSize ( 0 )
{
}

//==============================================================================

Thread::~Thread()
{
    Destroy();
}

//==============================================================================

void Thread::Start( THREAD_FUNCTION* pfnThread, void* pArg, size_t StackSize, int Priority, const char* pName )
{
    (void)pName;

    ASSERT( !m_AllocatedSize );

    TMA_TRACE( "Thread", "Start Pri:%d", Priority );

    TMA_TRACE( "TMAThread::Start", "Create Thread '%s'", pName );
    m_AllocatedSize = (((StackSize + (nn::os::StackRegionAlignment - 1)) & ~(nn::os::StackRegionAlignment - 1)) + (nn::os::StackRegionAlignment - 1));
    m_pUnalignedStackBegin = tma::s_Allocate( (size_t)m_AllocatedSize );
    m_pStack = (void*)(((u64)m_pUnalignedStackBegin - 1 + nn::os::StackRegionAlignment) & ~(nn::os::StackRegionAlignment - 1) );
    ASSERT( ((u64)m_pStack & (nn::os::StackRegionAlignment - 1)) == 0 );
    ASSERT( StackSize == ((StackSize / nn::os::StackRegionAlignment) * nn::os::StackRegionAlignment) );
    nn::os::CreateThread( &m_Thread, (nn::os::ThreadFunction)pfnThread, (void*)pArg, m_pStack, StackSize, Priority );
    nn::os::SetThreadName( &m_Thread, pName );
    nn::os::StartThread( &m_Thread );
}

//==============================================================================

void Thread::CancelSync()
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::Result Result = nn::svc::CancelSynchronization( static_cast<nn::svc::Handle>(m_Thread._handle) );
    ASSERT( Result.IsSuccess() );
#endif
}

//==============================================================================

void Thread::Join()
{
    nn::os::WaitThread( &m_Thread );
}

//==============================================================================

void Thread::Destroy()
{
    if( m_AllocatedSize )
    {
        nn::os::DestroyThread( &m_Thread );
        tma::s_Deallocate( m_pUnalignedStackBegin, (size_t)m_AllocatedSize  );
        m_pUnalignedStackBegin = NULL;
        m_pStack = NULL;
        m_AllocatedSize = 0;
    }
}

//==============================================================================
#if defined(NN_TMA_SIM_INSTANCE)
bool Thread::IsRunning() const
{
    bool IsRunning = { false };

    if( m_AllocatedSize )
    {
        const HANDLE hWin32Thread = static_cast<HANDLE>(m_Thread._handle._win32Handle);
        if( hWin32Thread != nullptr )
        {
            // Check to see if the Thread has exited...
            // https://msdn.microsoft.com/en-us/library/windows/desktop/ms683190(v=vs.85).aspx
            DWORD ExitCode = { 0 };
            const BOOL  Result = { ::GetExitCodeThread( hWin32Thread, &ExitCode ) };
            const bool  FunctionCallSucceeded   = { Result != 0 };
            // If the function succeeds *and* the Result was STILL_ALIVE, then
            // the thread is running.
            if( FunctionCallSucceeded && (ExitCode == STILL_ACTIVE) )
            {
                IsRunning = true;
            }
        }
    }

    return IsRunning;
}

//==============================================================================

bool Thread::HasExited() const
{
    bool HasExited = { false };

    HasExited = !IsRunning();

    return HasExited;
}
#endif
//==============================================================================

void Thread::Sleep( s32 MS )
{
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( MS ) );
}

//==============================================================================
} // namespace tma
//==============================================================================
