﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "..\tmipc\tmipc_result.h"
#include "diagnostics_service.h"
#include "diagnostics_Task.h"
#include <nn/dmnt/dmnt_Api.h>
#include <nn/nn_SystemThreadDefinition.h>

#define DIAGNOSTICS_MEMORY_REPORT(...) NN_SDK_LOG( "[tma][diagnostics] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )

#if DEBUG
#define DIAGNOSTICS_TRACE( channel, ... ) NN_SDK_LOG( "[tma][diagnostics][" ); \
    NN_SDK_LOG( channel ); \
    NN_SDK_LOG( "] - " ); \
    NN_SDK_LOG( __VA_ARGS__ ); \
    NN_SDK_LOG( "\n" )
#define DIAGNOSTICS_LOG(...) NN_SDK_LOG( "[tma][diagnostics] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )
#else
#define DIAGNOSTICS_TRACE(channel, ... )
#define DIAGNOSTICS_LOG(...)
#endif

//==============================================================================
namespace tma { namespace diagnostics {
//==============================================================================

//==============================================================================

namespace memory_tracking
{

//==============================================================================

#if TRACK_MEMORY

struct tracked_allocation
{
    tracked_allocation(size_t Size, void* pMem, const char* pFileName, int LineNumber)
    {
        m_Size = Size;
        m_pMem = pMem;
        memset(m_pFileName, 0, sizeof(m_pFileName));
        int NameLength = strlen(pFileName);
        int StartIndex = 0;
        int Length = sizeof(m_pFileName) - 1;
        Length = (NameLength < Length) ? NameLength : Length;
        StartIndex = NameLength - Length;
        memcpy(m_pFileName, pFileName + StartIndex, Length);
        m_LineNumber = LineNumber;
        m_pNext = NULL;
    }

    //=============================================================================

    static void add( tracked_allocation* pNew, tracked_allocation** pAddToList )
    {
        if (pNew != NULL)
        {
            //=============================================================================
            // For tracking allocations, it's more likely we'll be adding and removing
            // entries from the end, so a FIFO linked list should be performant enough.
            //=============================================================================
            pNew->m_pNext = *pAddToList;
            *pAddToList = pNew;
        }
    }

    //=============================================================================

    static tracked_allocation* remove( void* pMem, tracked_allocation** pRemoveFromList )
    {
        tracked_allocation* pRemove = *pRemoveFromList;
        tracked_allocation* pPrev = NULL;

        // First, find this guy.
        while (pRemove != NULL)
        {
            if (pRemove->m_pMem == pMem)
            {
                break;
            }
            pPrev = pRemove;
            pRemove = pRemove->m_pNext;
        }

        // Find him?
        if (pRemove != NULL)
        {
            // If he was the root, special treatment
            if (pPrev == NULL)
            {
                *pRemoveFromList = pRemove->m_pNext;
            }
            else
            {
                pPrev->m_pNext = pRemove->m_pNext;
            }
        }
        return pRemove;
    }

    //=============================================================================

    static size_t report( tracked_allocation* pList )
    {
        size_t Ret = 0;

        while ( pList != NULL )
        {
            DIAGNOSTICS_MEMORY_REPORT( "%d - %s (line %d) (%p)", pList->m_Size, pList->m_pFileName, pList->m_LineNumber, pList->m_pMem );
            Ret += pList->m_Size;
            pList = pList->m_pNext;
        }

        return Ret;
    }

    //=============================================================================

    static void clear( nn::lmem::HeapHandle HeapHandle, tracked_allocation** pClearList )
    {
        tracked_allocation* pList = *pClearList;
        while (pList != NULL)
        {
            tracked_allocation* pNext = pList->m_pNext;

            pList->~tracked_allocation();
            nn::lmem::FreeToExpHeap( HeapHandle, pList );

            pList = pNext;
        }
        *pClearList = NULL;
    }

    //=============================================================================

    size_t  m_Size;
    void*   m_pMem;
    char    m_pFileName[32];
    int     m_LineNumber;
    tracked_allocation* m_pNext;
};

#endif //#if TRACK_MEMORY

//==============================================================================
// Static variables for this namespace
//==============================================================================
static nn::lmem::HeapHandle s_HeapHandle = nullptr;
#if TRACK_MEMORY
static tracked_allocation* s_pAllocations = NULL;
static tma::Mutex s_AllocationTrackingMutex;
#endif

//==============================================================================

void Init( nn::lmem::HeapHandle HeapHandle )
{
#if TRACK_MEMORY
    s_AllocationTrackingMutex.Create();
#endif
    s_HeapHandle = HeapHandle;
}

//==============================================================================

void Kill()
{
#if TRACK_MEMORY
    tracked_allocation::clear( s_HeapHandle, &s_pAllocations );
    s_AllocationTrackingMutex.Destroy();
#endif
    s_HeapHandle = nullptr;
}

//==============================================================================

void Report()
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::lmem::DumpExpHeap( s_HeapHandle );
    size_t totalFree = nn::lmem::GetExpHeapTotalFreeSize( s_HeapHandle );
    size_t allocatable = nn::lmem::GetExpHeapAllocatableSize( s_HeapHandle, nn::lmem::DefaultAlignment );
    DIAGNOSTICS_MEMORY_REPORT( "Memory Report:  %d bytes available, %d bytes total available",  allocatable, totalFree );
#if TRACK_MEMORY
    {
        ScopedLock Lock( s_AllocationTrackingMutex );
        size_t Size = tracked_allocation::report( s_pAllocations );
        DIAGNOSTICS_MEMORY_REPORT( "Total tracked:  %d", Size );
    }
#else
    DIAGNOSTICS_MEMORY_REPORT( "Set the TRACK_MEMORY macro (tmagent.h, line 49) to non-zero and rebuild tmagent to turn on memory tracking." );
#endif //TRACK_MEMORY
#else //#if defined(NN_BUILD_CONFIG_OS_HORIZON)

#endif
}

//==============================================================================

void Remember( size_t Size, void* pMem, const char* pFileName, int LineNumber )
{
    DIAGNOSTICS_LOG( "Remember alloc %d (%s, %d) -> %p",  Size, pFileName, LineNumber, pMem );
#if TRACK_MEMORY
    if( s_HeapHandle != nullptr && pMem != NULL )
    {
        ScopedLock Lock( s_AllocationTrackingMutex );
        void* p = nn::lmem::AllocateFromExpHeap( s_HeapHandle, sizeof(tracked_allocation) );
        if( p != NULL )
        {
            tracked_allocation* pNew = new (p) tracked_allocation( Size, pMem, pFileName, LineNumber );
            tracked_allocation::add( pNew, &s_pAllocations );
        }
    }
#endif
}

//==============================================================================

void Forget( void* pMem )
{
    DIAGNOSTICS_LOG( "Forget alloc %p", pMem );
    if( s_HeapHandle != nullptr )
    {
#if TRACK_MEMORY
        ScopedLock Lock( s_AllocationTrackingMutex );
        tracked_allocation* pMemNode = tracked_allocation::remove( pMem, &s_pAllocations );
        if( pMemNode != NULL )
        {
            pMemNode->~tracked_allocation();
            nn::lmem::FreeToExpHeap( s_HeapHandle, pMemNode );
        }
#endif
    }
}

//==============================================================================

} // namespace memory_tracking


//==============================================================================

DiagnosticsTask::DiagnosticsTask()
{
    DIAGNOSTICS_LOG( "DiagnosticsTask::DiagnosticsTask" );

    //====================================================================================
    // Because this is a longer-running task, we'll mark ourselves as un-owned - this way
    // we will be properly cleaned by the system regardless of when we're finished.
    //====================================================================================
    SetExternallyOwned( false );
}

//==============================================================================

DiagnosticsTask::~DiagnosticsTask()
{
    DIAGNOSTICS_LOG("DiagnosticsTask::~DiagnosticsTask");
}

//==============================================================================

void DiagnosticsTask::OnRecvPacket( tmipc::Packet* pPacket )
{
    DIAGNOSTICS_LOG("DiagnosticsTask::OnRecvPacket", "OnRecvPacket" );
    Complete();
}

//==============================================================================

void DiagnosticsTask::OnSendPacket( tmipc::Packet* pPacket )
{
    DIAGNOSTICS_LOG("DiagnosticsTask::OnSendPacket", "OnSendPacket" );
    (void)pPacket;
    Complete();
}

//==============================================================================

void DiagnosticsTask::OnInitiate( tmipc::Packet* pPacket )
{
    s32 Opcode = 0;
    s32 Result = tmapi::result::RESULT_OK;
    Service* pOriginatingService = (Service*)GetService();

    // Read our opcode
    pPacket->ReadS32( Opcode );
    DIAGNOSTICS_TRACE("DiagnosticsTask::OnInitiate", "OnInitiate - Opcode %d", Opcode );
    switch( Opcode )
    {
    case tma::diagnostics::DIAGNOSTICS_TMIPC:
        m_pServicesManager->DiagnosticReport();
        break;

    case tma::diagnostics::DIAGNOSTICS_MEMORY:
        memory_tracking::Report();
        break;

    case tma::diagnostics::DIAGNOSTICS_PROFILING:
        break;

    default:
        NN_SDK_LOG( "Unrecognized diagnostics opcode:  %d\n", Opcode );
        break;
    }

    tmipc::Packet* p = AllocSendPacket();
    p->WriteS32( Result );
    m_pServicesManager->Send( p );
    Complete();
}

//==============================================================================

//==============================================================================
}} // namespace
//==============================================================================
