﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include "dbg_SimService.h"
#include "../tmipc/tmipc_packet.h"
#include "dbg_Enums.h"

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

//==============================================================================

SimDebugService::SimDebugService()
{
    TMA_TRACE( "SimDebugService", "SimDebugService" );

    // Let SimDebugTask initialize.
    SimDebugTask::Initialize();

    // Use "DebugService" instead of "SimDebugService", because TargetManager
    // only knows of "DebugService" (or 0xddde6636).
    m_ServiceId = HashString( "DebugService" );
    m_DispatchTaskQ.Create( 4 );
    m_DispatchTaskThread.Start( DispatchTaskThread, this, (16 * 1024), NN_SYSTEM_THREAD_PRIORITY(tma, SimDbgSrvDispatch), NN_SYSTEM_THREAD_NAME(tma, SimDbgSrvDispatch) );
}

//==============================================================================

SimDebugService::~SimDebugService()
{
    // Oasis-832: SDK Assertion Failure on tmipc_node_tcp.cpp:250
    // If the user presses Ctrl-c or closes the Console Windows (red X) then
    // there is sending the DispatchDebugTask message might hang.
    // If the thread is still active, then go ahead and send the message.
//TODO: NOTE: this may introduce a race condition, if the thread exits *after*
//      this check is made.
    if( m_DispatchTaskThread.IsRunning() )
    {
        void* pMem = tma::s_Allocate( sizeof( DispatchDebugTask ) );
        m_DispatchTaskQ.Send( new (pMem) DispatchDebugTask() );
    }
    m_DispatchTaskThread.Join();

    m_DispatchTaskThread.Destroy();
    while( m_DispatchTaskQ.GetCount() )
    {
        DispatchDebugTask* Task = (DispatchDebugTask*)m_DispatchTaskQ.Receive();
        Task->~DispatchDebugTask();
        tma::s_Deallocate( Task, sizeof( DispatchDebugTask ) );
    }
    m_DispatchTaskQ.Destroy();

    // Let DebugTask shut down.
    SimDebugTask::Finalize();
}

//==============================================================================

void* SimDebugService::DispatchTaskThread( void* pArg )
{
    bool Continue = true;
    SimDebugService* pThis = reinterpret_cast<SimDebugService*>(pArg);
    while( Continue )
    {
        TMA_TRACE( "SimDebugService", "DispatchTaskThread - Wait Task" );
        DispatchDebugTask* Recv = (DispatchDebugTask*)pThis->m_DispatchTaskQ.Receive();
        TMA_TRACE( "SimDebugService", "DispatchTaskThread - Got Task" );
        if( Recv->m_ShouldDispatch )
        {
            tmipc::Task* pTask = NULL;
            switch( Recv->m_CommandOp )
            {
            case dbg::TMAgent_message_type::TMAGENT_MESSAGE_TYPE_INFO:
                {
                    DispatchDebugMessageTask* pMsgData = (DispatchDebugMessageTask*)Recv;
                    pTask = pThis->StartDebugTask( dbg::TMAgent_message_type::TMAGENT_MESSAGE_TYPE_INFO, pMsgData->m_ProcessId, pMsgData->m_Message, static_cast<tma::s32>(strlen(pMsgData->m_Message) + 1) );
                }
                break;
            case dbg::TMAgent_message_type::TMAGENT_MESSAGE_TYPE_EXCEPTION:
                {
                    DispatchDebugExceptionTask* pExceptionData = (DispatchDebugExceptionTask*)Recv;
                    pTask = pThis->StartDebugTask( dbg::TMAgent_message_type::TMAGENT_MESSAGE_TYPE_EXCEPTION, pExceptionData->m_ProcessId, &pExceptionData->m_exceptionData, sizeof(pExceptionData->m_exceptionData) );
                    if( pExceptionData->m_exceptionData.m_ExceptionId == tma::DEBUG_EXCEPTION_PROCESS_EXIT )
                    {
//                        pThis->m_pProcessMgr->ProcessHasExited( pExceptionData->m_ProcessId );
                    }
                }
                break;
            default:
                break;
            }
            if( pTask != NULL )
            {
                pTask->WaitComplete( TMIPC_INFINITE );
                pThis->m_pServicesManager->FreeTask(pTask);
                pTask = nullptr;
            }
        }
        else
        {
            Continue = false;
        }
        Recv->~DispatchDebugTask();
        tma::s_Deallocate( Recv, sizeof( DispatchDebugTask ) );
    }

    return NULL;
}

//==============================================================================

void SimDebugService::Init()
{
    Create();
}

//==============================================================================

void SimDebugService::Kill()
{
    Destroy();
}

//==============================================================================

tmipc::Task* SimDebugService::StartDebugTask( TMAgent_message_type MessageType, tma::dbg::process_id ProcessId, void* pData, s32 SizeOfData )
{
    TMA_TRACE( "SimDebugService", "StartDebugTask" );

    void* pMem = tma::s_Allocate( sizeof( SimDebugTask ) );
    SimDebugTask* pTask = new (pMem) SimDebugTask();
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( m_pServicesManager->AllocTaskId() );
    pTask->Start( MessageType, ProcessId, pData, SizeOfData );
    return pTask;
}

//==============================================================================

tmipc::Task* SimDebugService::OnNewTask( tmipc::Packet* pPacket )
{
    void* pMem = tma::s_Allocate( sizeof( SimDebugTask ) );
    SimDebugTask* pTask = new (pMem) SimDebugTask();
    pTask->SetServicesManager( m_pServicesManager );
    pTask->SetServiceId( m_ServiceId );
    pTask->SetTaskId( pPacket->GetTaskId() );
    pTask->OnInitiate( pPacket );
    return( pTask );
}

//==============================================================================

void SimDebugService::NotifyException( tma::dbg::process_id ProcessId, tma::dbg::Exception* pExceptionData )
{
    TMA_TRACE( "SimDebugService", "NotifyException" );
    TMA_TRACE( "SimDebugService", "---------------" );

    void* pMem = tma::s_Allocate( sizeof( DispatchDebugExceptionTask ) );
    DispatchDebugExceptionTask* pToSend = new (pMem) DispatchDebugExceptionTask( ProcessId, pExceptionData );
    m_DispatchTaskQ.Send( pToSend );
}

//==============================================================================

void SimDebugService::NotifyMessage( tma::dbg::process_id ProcessId, const char* pMessage )
{
    void* pMem = tma::s_Allocate( sizeof( DispatchDebugMessageTask ) );
    DispatchDebugMessageTask* pToSend = new (pMem) DispatchDebugMessageTask( ProcessId, pMessage );
    m_DispatchTaskQ.Send( pToSend );
}

//==============================================================================
}}
//==============================================================================
