﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/dbg/dbg_Api.h>
#include <nn/dmnt/dmnt_Api.h>
#include <cinttypes>
#include <nn/nn_SystemThreadDefinition.h>

#include "..\tmagent.h"

#include "dbg_opcodes.h"
#include "dbg_CommandArgs.h"
#include "dbg_Exception.h"
#include "dbg_Process.h"
#include "dbg_ProcessMgr.h"
#include "dbg_RegisterDefs.h"
#include "dbg_CoreDump.h"
#include "..\tm_result.h"
#include "../dbghlp/dbghlp.h"
#include "..\threadfrozen\thread_frozen_api.h"
#include <nn/ro/detail/ro_NroHeader.h>

#define PRINT_EXCEPTIONS TMA_MACRO_VALUE(0)

#if PRINT_EXCEPTIONS

#define EXCEPT_PRINTF( ... ) NN_SDK_LOG( __VA_ARGS__ )

#else

#define EXCEPT_PRINTF( ... )

#endif

//==============================================================================
namespace tma { namespace dbg {
//==============================================================================

enum
{
    TMA_SIZE_OF_THREAD_ID_BUFFER = 512,
    TMA_SIZE_OF_MEMORY_BLOCK_BUFFER = 512
};

// This is implemented in main.cpp.
extern s32 CheckProcessHandleForError( nn::svc::Handle ProcessHandle );

//==============================================================================

void Process::ClearBreadcrumbs()
{
#if NUM_BREADCRUMBS

    NN_SDK_LOG( "\nProcess::ClearBreadcrumbs\n" );

    m_iBreadcrumb = 0;

    for( s32 i = 0; i < NUM_BREADCRUMBS; i++ )
    {
        m_Breadcrumbs[i] = 0;
    }

#endif
}

//==============================================================================

void Process::DropBreadcrumb( u64 Value )
{
    (void)Value;

#if NUM_BREADCRUMBS

    m_Breadcrumbs[ m_iBreadcrumb++ ] = Value;

    if( m_iBreadcrumb >= NUM_BREADCRUMBS )
    {
        m_iBreadcrumb = 0;
    }

#endif
}

//==============================================================================

void Process::ShowBreadcrumbs()
{
#if NUM_BREADCRUMBS

    NN_SDK_LOG( "\nProcess::Breadcrumbs:\n" );

    s32 iBreadcrumb = m_iBreadcrumb;

    for( s32 i = 0; i < NUM_BREADCRUMBS; i++ )
    {
        NN_SDK_LOG( " %016llx", m_Breadcrumbs[ iBreadcrumb++ ] );

        if( iBreadcrumb >= NUM_BREADCRUMBS )
        {
            iBreadcrumb = 0;
        }

        if( ( i & 7 ) == 7 )
        {
            NN_SDK_LOG( "\n" );
        }
    }

#endif
}

//==============================================================================

Process::Process( ProcessMgr* pOwner, const char* pSource, const char* pArgs, nn::svc::Handle processHandle )
{
    m_pOwner = pOwner;
    m_Handle                    = processHandle;

    // Save our source name for later.
    strncpy( m_Name, pSource, sizeof(m_Name) - 1 );
    m_Name[sizeof(m_Name) - 1]  = 0;

    // Save our args in case we need them later.
    strncpy( m_Args, pArgs, sizeof(m_Args) - 1 );
    m_Args[sizeof(m_Args) - 1]  = 0;

//NN_SDK_LOG("Process::Process:  Name %s\n", m_Name );

    nn::Bit64 PID;
    nn::Result result = dbghlp::GetProcessId( &PID, m_Handle );
    if( result.IsFailure() )
    {
        TMA_RESULT_TRACE( result );
    }
    ASSERT( (PID >> 32) == 0 );
    m_PID = PID;

    Init();

    m_HasLoaded                 = true;
}

//==============================================================================================

Process::Process( ProcessMgr* pOwner, process_id PID, nn::svc::Handle processHandle )
{
    m_pOwner    = pOwner;
    m_PID       = PID;
    m_Handle    = processHandle;    // m_Handle must be set before calling Init()

    // Generate a name
    sprintf( m_Name, "Process %" PRIu64, PID );

    // We've got no args
    memset( m_Args, 0, sizeof(m_Args) );

    //Init everyone else.
    Init();

    m_HasLoaded     = true;
    m_ProcessState  = PS_RUNNING;
}

//==============================================================================

Process::~Process()
{
    TMA_TRACE("Process::~Process", "Process::~Process" );

    ShowBreadcrumbs();
    Kill();
}

//==============================================================================

void Process::Init()
{
    TMA_TRACE( "Process::Init", "Started" );
    ClearBreadcrumbs();

    m_BreakManager.Init( this );
    m_StepManager.Init( this );

    m_Modules.Init( m_Name, m_Handle, m_PID );

    m_ProcessState              = PS_NOT_STARTED;
    m_HasAttached               = false;
    m_HasLoaded                 = false;
    m_ModulesChanged            = false;
    m_DebugEventThreadStarted   = false;
    m_DebugEventThreadKill      = false;
    m_64Bit                     = false;
    m_64BitAddressSpace         = false;
    m_NeedUpdate                = true;
    m_LastExceptionId           = NO_EXCEPTION_ID;
    m_LastExceptionAddress      = 0;
    m_NumActiveThreads          = 0;
    m_pRegisterDefinitions      = 0;
    m_SizeRegisterDefs          = 0;
    m_CurrentThreadId           = TMA_NO_THREAD_ID;
    m_Threads.Clear();
    m_CoreDumpSessionId         = 0;
}

//==============================================================================

tmapi::result Process::WaitForAttach()
{
    bool isAttached = false;

    while ( isAttached == false || m_NumActiveThreads <= 0 )
    {
        int32_t handleIndex = -1;
        nn::Result result = dbghlp::WaitSynchronization( &handleIndex, &m_Handle, 1, 10 * 1000 * 1000 );
        if( result.IsFailure() )
        {
            return tmapi::result::RESULT_EXECUTABLE_NOT_COMPATIBLE;
        }
        nn::svc::DebugEventInfo eventInfo;
        result = dbghlp::GetDebugEvent(&eventInfo, m_Handle);
        if( result.IsFailure() )
        {
            return tmapi::result::RESULT_EXECUTABLE_NOT_COMPATIBLE;
        }

//NN_SDK_LOG( "[Process::WaitForAttach]:  Event %d\n", eventInfo.event );
        switch (eventInfo.event)
        {

        case nn::svc::DebugEvent_CreateProcess:
            OnProcessCreated( &eventInfo );
            break;

        case nn::svc::DebugEvent_Exception:
            if ( eventInfo.info.exception.exceptionCode == nn::svc::DebugException_AttachBreak )
            {
                isAttached = true;
            }
            else if ( ( eventInfo.info.exception.exceptionCode != nn::svc::DebugException_BreakPoint ) &&
                      ( eventInfo.info.exception.exceptionCode != nn::svc::DebugException_DebuggerBreak ) )
            {
                m_LastExceptionId = eventInfo.info.exception.exceptionCode;
                m_LastExceptionAddress = eventInfo.info.exception.exceptionAddress;
            }
            break;

        case nn::svc::DebugEvent_CreateThread:
            OnThreadCreated( &eventInfo, false );
            break;

        case nn::svc::DebugEvent_ExitThread:
            OnThreadExited( eventInfo.threadId, false );
            break;

        default:
            break;
        }
    }

    return tmapi::result::RESULT_OK;
}

//==============================================================================

bool Process::Attach( nn::svc::Handle processHandle )
{
    //===============================================
    // If we're trying to attach, mark our
    // process state to not started so that
    // the initialization code works as it ought.
    //===============================================
    m_ProcessState              = PS_NOT_STARTED;

    if( m_Handle == nn::svc::INVALID_HANDLE_VALUE )
    {
        m_Handle = processHandle;
    }

    if( m_Handle == nn::svc::INVALID_HANDLE_VALUE )
    {
        nn::Result result = dbghlp::GetProcessHandle( &m_Handle, m_PID );
    }

    m_Modules.Attach( m_Handle );

    if( m_Handle != nn::svc::INVALID_HANDLE_VALUE )
    {
//NN_SDK_LOG("Attaching to %lld\n", m_PID );
        WaitForAttach();
        StartDebugProcess( 0 );

        //===================================================================================================
        // Check our exceptions - if we have one, then we should not continue.  This fixes SigloNTD-10579.
        //===================================================================================================
        if( m_LastExceptionId == NO_EXCEPTION_ID )
        {
            ContinueDebugProcess( 0 );
        }
        else
        {
            //=================================
            // Mark our state appropriately.
            //=================================
            SetProcessState( PS_HALT );
        }
    }
//NN_SDK_LOG("Attached to %lld\n", m_PID );
    return m_Handle != nn::svc::INVALID_HANDLE_VALUE;
}

//==============================================================================

s32 Process::Close( bool KillProcess )
{
    TMA_TRACE("Process::Close", "Process::Close");
    s32 Status = -1;
    if (m_Handle != nn::svc::INVALID_HANDLE_VALUE)
    {
        tma::process_state ProcessState = GetProcessState();
        SetProcessState(PS_EXITING);

        m_BreakManager.Reset();

        s32 Count = m_Threads.Count();
        while (Count--)
        {
            tma::threadfrozen::ClearThreadFrozen(m_Threads.Get(Count)->GetThreadId());
        }

        //==================================================================
        // Close any coredump tasks going on for this process.  This fixes
        // "killing a process while doing a coredump locks up the system".
        // This is safe to call if there is no coredump going on.
        //==================================================================
        CoreDump::Close(this);

        if (KillProcess)
        {
            //NN_SDK_LOG("[tma] Process::Close:  Preparing to call dbghlp::TerminateDebugProcess\n");
            TMA_TRACE("Process::Close", "TerminateProcess");
            nn::Result Result = dbghlp::TerminateDebugProcess(m_Handle);

            //ASSERT( Result.IsSuccess() );
            if (ProcessState == PS_NOT_STARTED)
            {
                //NN_SDK_LOG("[tma] Process::Close:  Waiting for termination (ProcessState == PS_NOT_STARTED) \n");
                WaitForTermination();
                //NN_SDK_LOG("[tma] Process::Close:  ProcessState == PS_NOT_STARTED, WaitForTermination complete \n" );
            }
        }

        //NN_SDK_LOG("[tma] Process::Close:  dbghlp::CloseHandle( m_Handle )\n");
        TMA_TRACE("Process::Close", "CloseHandle");
        nn::Result result = dbghlp::CloseHandle(m_Handle);
        ASSERT(result.IsSuccess());

        ResetState();

        Status = tmapi::RESULT_OK;
    }

    m_pOwner->OnProcessKilled(this);

    return(Status);
}

//==============================================================================

s32 Process::Detach()
{
    return Close( false );
}

//==============================================================================

s32 Process::GetJITException()
{
    //===================================================================
    // If we don't know what our exception is, we'll have to go get it.
    if( m_LastExceptionId == NO_EXCEPTION_ID )
    {
//NN_SDK_LOG("[tma] Process::GetJITException:  m_LastExceptionId == NO_EXCEPTION_ID\n" );
        bool CreatedDebugHandle = false;
        if( m_Handle == nn::svc::INVALID_HANDLE_VALUE )
        {
            nn::Result result = dbghlp::GetProcessHandle( &m_Handle, m_PID );
            if( result.IsFailure() )
            {
                NN_SDK_LOG("[tma] Process::GetJITException:  GetProcessHandle returned FAILURE.\n" );
                return 0;
            }
            CreatedDebugHandle = true;
        }

        //Now go until we find the killer exception.
        while ( m_LastExceptionId == NO_EXCEPTION_ID )
        {
            int32_t handleIndex = -1;
            nn::Result result = dbghlp::WaitSynchronization( &handleIndex, &m_Handle, 1, 10 * 1000 * 1000 );
            if( nn::svc::ResultTimeout::Includes( result ) )
            {
                continue;
            }

            if( result.IsFailure() )
            {
///                NN_SDK_LOG("[tma] Process::GetJITException:  dbghlp::WaitSynchronization returned FAILURE (0x%x).\n", result.GetInnerValueForDebug() );
                break;
            }
            nn::svc::DebugEventInfo eventInfo;
            result = dbghlp::GetDebugEvent(&eventInfo, m_Handle);
            if( result.IsFailure() )
            {
//                NN_SDK_LOG("[tma] Process::GetJITException:  dbghlp::GetDebugEvent returned FAILURE (0x%x).\n", result.GetInnerValueForDebug() );
                break;
            }

//NN_SDK_LOG("[tma] Process::GetJITException:  Got event %d.\n", eventInfo.event );
            switch (eventInfo.event)
            {
            case nn::svc::DebugEvent_Exception:
                switch( eventInfo.info.exception.exceptionCode )
                {
                case nn::svc::DebugException_UndefinedInstruction:
                case nn::svc::DebugException_AccessViolationInstruction:
                case nn::svc::DebugException_AccessViolationData:
                case nn::svc::DebugException_DataTypeMissaligned:
                case nn::svc::DebugException_UserBreak:
                case nn::svc::DebugException_UndefinedSystemCall:
                case nn::svc::DebugException_MemorySystemError:
                default:
                    m_LastExceptionId       = eventInfo.info.exception.exceptionCode;
                    m_LastExceptionAddress  = eventInfo.info.exception.exceptionAddress;
                    break;

                case nn::svc::DebugException_AttachBreak:
                case nn::svc::DebugException_BreakPoint:
                case nn::svc::DebugException_DebuggerBreak:
                    ContinueDebugProcess( 0 );
                    break;
                }
                break;

            case nn::svc::DebugEvent_CreateProcess:
                OnProcessCreated( &eventInfo );
                // Fall through to the default:

            case nn::svc::DebugEvent_CreateThread:
            case nn::svc::DebugEvent_ExitThread:
                // Fall through to the default:

            default:
                ContinueDebugProcess( 0 );
                break;
            }
        }

        if( CreatedDebugHandle == true )
        {
            dbghlp::CloseHandle( m_Handle );
            m_Handle = nn::svc::INVALID_HANDLE_VALUE;
        }
    }

    return m_LastExceptionId;
}

//==============================================================================

void Process::Kill()
{
    // Terminate the Debug Event Thread?
    if( m_DebugEventThreadStarted )
    {
        m_DebugEventThreadKill = true;
        m_DebugEventThread.Join();
        m_DebugEventThread.Destroy();
    }

    m_BreakManager.Kill();
    m_StepManager.Kill();
    m_Modules.Kill();
    m_CurrentThreadId = TMA_NO_THREAD_ID;
    m_Threads.Clear();

    if( m_pRegisterDefinitions != NULL )
    {
        s_Deallocate( m_pRegisterDefinitions, m_SizeRegisterDefs);
        m_pRegisterDefinitions = NULL;
        m_SizeRegisterDefs = 0;
    }
}

//==============================================================================

void Process::Update()
{
    if( m_NeedUpdate == true )
    {
        s32 Result = UpdateThreadList();
        if( Result == tmapi::RESULT_OK )
        {
            UpdateProcessThread( this );
        }
        m_NeedUpdate = false;
    }
}

//==============================================================================
// Updating threads takes considerable time
// A good time to grab other data as well

void* Process::UpdateProcessThread( void* pArg )
{
    Process* pThis = (Process*)pArg;
    s32 NumberOfThreads = pThis->m_Threads.Count();
    ThreadDefinition::UpdateAll( pThis->m_Handle, &pThis->m_Threads, NumberOfThreads );

    return NULL;
}

//==============================================================================

s32 Process::UpdateThreadList()
{
    s32 Ret = tmapi::RESULT_OK;
    nn::Bit64 ThreadIds[TMA_SIZE_OF_THREAD_ID_BUFFER];
    int32_t NumThreads = 0;
    nn::Result Result = dbghlp::GetThreadList( &NumThreads, ThreadIds, TMA_SIZE_OF_THREAD_ID_BUFFER, m_Handle );
    if( Result.IsSuccess() )
    {
        if( NumThreads > 0 )
        {
            m_Threads.Update( NumThreads, ThreadIds );

            //Make sure what we think is "current" is still alive
            int NumberThreadsChecked = 0;
            for( NumberThreadsChecked = 0; NumberThreadsChecked < NumThreads; NumberThreadsChecked++ )
            {
                if( ThreadIds[NumberThreadsChecked] == m_CurrentThreadId )
                {
                    break;
                }
            }

            //Did we find it in our "live" list?
            if( NumberThreadsChecked == NumThreads )
            {
                //Default to the first one.
                m_CurrentThreadId = ThreadIds[0];
            }
        }
        else
        {
            m_Threads.Clear();
        }
    }
    else
    {
        TMA_RESULT_TRACE( Result );
        Ret = CheckProcessHandleForError( m_Handle );
    }

    TMA_TRACE("Process", "UpdateThreadList:  Found %d threads, CurrentId = %d", m_Threads.Count(), m_CurrentThreadId );
    return Ret;
}

//==============================================================================

nn::Result Process::WaitForCreateProcess()
{
//  NN_SDK_LOG( "Starting WaitForCreateProcess for process %lld\n", m_PID );
    nn::Result result = nn::ResultSuccess();
    bool isAttached = false;

    while ( isAttached == false && m_NumActiveThreads <= 0 )
    {
        int32_t handleIndex = -1;
        result = dbghlp::WaitSynchronization( &handleIndex, &m_Handle, 1, 10 * 1000 * 1000 );
        if( result.IsFailure() )
        {
            NN_SDK_LOG( "WaitForCreateProcess FAIL on WaitSynchronization for process %lld\n", m_PID );
            return result;
        }
        nn::svc::DebugEventInfo eventInfo;
        result = dbghlp::GetDebugEvent(&eventInfo, m_Handle);
        if( result.IsFailure() )
        {
            NN_SDK_LOG( "WaitForCreateProcess FAIL on GetDebugEvent for process %lld\n", m_PID );
            return result;
        }
//        NN_SDK_LOG( "WaitForCreateProcess received debug event %d\n", eventInfo.event );

        switch (eventInfo.event)
        {
        case nn::svc::DebugEvent_CreateProcess:
            {
                OnProcessCreated( &eventInfo );
                isAttached = true;
            }
            break;

        case nn::svc::DebugEvent_Exception:
            if (eventInfo.info.exception.exceptionCode == nn::svc::DebugException_AttachBreak)
            {
                isAttached = true;
            }
            break;

        case nn::svc::DebugEvent_CreateThread:
            OnThreadCreated( &eventInfo );
            m_NumActiveThreads += 1;
            break;

        case nn::svc::DebugEvent_ExitThread:
            m_NumActiveThreads -= 1;
            break;

        default:
            break;
        }
    }

//    NN_SDK_LOG( "WaitForCreateProcess finished for process %lld\n", m_PID);
    return result;
}

//==============================================================================

process_id Process::GetProcessId()
{
    return( m_PID );
}

//==============================================================================

u64 Process::GetProgramId()
{
    return m_DebugInfoCreateProcess.programId;
}

//==============================================================================

process_state Process::GetProcessState()
{
    return( m_ProcessState );
}

//==============================================================================

void Process::SetProcessState( process_state State )
{
    DEJA_TRACE( "Process::SetProcessState", "ProcessState = %(process_state)", State );
    TMA_TRACE(  "Process::SetProcessState", "ProcessState = %d", State );
    //NN_SDK_LOG( "Setting process %lld state from %2d to %d\n", m_PID, m_ProcessState, State );

    m_ProcessState = State;
}

//==============================================================================

bool Process::Is64Bit()
{
    return( m_64Bit );
}

//==============================================================================

bool Process::Is64BitAddressSpace()
{
    return( m_64BitAddressSpace );
}

//==============================================================================

dbg::BreakpointMgr* Process::GetBreakpointManager()
{
    return( &m_BreakManager );
}

//==============================================================================

dbg::Modules* Process::GetModules()
{
    return( &m_Modules );
}

//==============================================================================

dbg::StepMgr* Process::GetStepManager()
{
    return( &m_StepManager );
}

//==============================================================================

bool Process::GetPerThreadStepInfo( u64 ThreadId, per_thread_step_info& PTSI )
{
    return m_Threads.GetPerThreadStepInfo( ThreadId, PTSI );
}

//==============================================================================

bool Process::SetPerThreadStepInfo( u64 ThreadId, per_thread_step_info& PTSI )
{
    return m_Threads.SetPerThreadStepInfo( ThreadId, PTSI );
}

//==============================================================================

void Process::SetCoredumpSessionId( u64 SessionId )
{
    m_CoreDumpSessionId = SessionId;
}

//==============================================================================

u64  Process::GetCoredumpSessionId()
{
    return m_CoreDumpSessionId;
}

//==============================================================================

nn::svc::Handle Process::GetHandle()
{
    return( m_Handle );
}

//==============================================================================

void Process::GetName( char* pBuffer, s32 BufferSize )
{
    if( BufferSize > 0 )
    {
        strncpy( pBuffer, m_Name, BufferSize - 1 );
        pBuffer[BufferSize - 1] = 0;
    }
}

//==============================================================================

void Process::GetArgs( char* pBuffer, s32 BufferSize )
{
    if( BufferSize > 0 )
    {
        strncpy( pBuffer, m_Args, BufferSize - 1 );
        pBuffer[BufferSize - 1] = 0;
    }
}

//==============================================================================

uint32_t Process::GetLastExceptionId()
{
    return m_LastExceptionId;
}

//==============================================================================

uint64_t Process::GetLastExceptionAddress()
{
    return m_LastExceptionAddress;
}

//==============================================================================

void Process::GetDefinition( ProcessDefinition& ProcessDefinition )
{
    ProcessDefinition.m_NumberOfModules = m_Modules.GetCount();
    ProcessDefinition.m_PID             = m_PID;
    ProcessDefinition.m_NumThreads      = m_Threads.Count();
    ProcessDefinition.m_LastExceptionId = m_LastExceptionId;

    // For the 64-bit flags, Bit 0 = 64 bit process, bit 1 = 64 bit addresses
    ProcessDefinition.m_64BitFlags      =  ( ( Is64Bit() ) | ( Is64BitAddressSpace() << 1 ) );

    // Set the state of this process.  0 = running, 1 = stopped
    if( GetProcessState() == PS_RUNNING )
    {
        ProcessDefinition.m_State       = 0;
    }
    else
    {
        ProcessDefinition.m_State       = 1;
    }

    GetName( ProcessDefinition.m_Name, sizeof(ProcessDefinition.m_Name) );
}

//==============================================================================

void Process::OnThreadCreated( nn::svc::DebugEventInfo* pEventInfo, bool Continue )
{
    //tma::dbg::MemToLog( pEventInfo, sizeof(nn::svc::DebugEventInfo), "DebugEventInfo:\n" ); /////////////// TEMPORARY

    nn::osdbg::ThreadInfo threadInfo;
    nn::Result result = dbghlp::InitializeThreadInfo( &threadInfo, m_Handle, &m_DebugInfoCreateProcess, &pEventInfo->info.createThread );

    if( ! result.IsSuccess() )
    {
        // This can happen if the process hasn't started yet, so no need to report a failure.
        //NN_SDK_LOG("tma::dbg::Process::OnThreadCreated: tma::dbghelp::InitializeThreadInfo FAILED for thread %d:  Error = 0x%x\n",
        //    pEventInfo->threadId, result.GetInnerValueForDebug() );
        threadInfo._stack                   = 0;    // Save enough info to try this again later
        threadInfo._stackSize               = 0;
        threadInfo._debugHandle             = m_Handle;
        threadInfo._debugInfoCreateProcess  = m_DebugInfoCreateProcess;
        threadInfo._debugInfoCreateThread   = pEventInfo->info.createThread;
    }
    //tma::dbg::MemToLog( &threadInfo, sizeof(threadInfo), "ThreadInfo:\n" ); /////////////// TEMPORARY

    //TMA_TRACE("Process::OnThreadCreated", "OnThreadCreated" );
    m_Threads.Add( pEventInfo->threadId, &threadInfo );

    //m_Threads.ToLog(); /////////////// TEMPORARY

    //if( m_HasAttached == false )
    //{
    //    UpdateThreadList();
    //    m_HasAttached = true;
    //}
    m_NumActiveThreads += 1;
    if( Continue )
    {
        ContinueDebugProcess( 0 );
    }
}

//==============================================================================

void Process::OnProcessCreated( nn::svc::DebugEventInfo* pEventInfo )
{
    // We'll need this process data for resolving thread details later.
    m_DebugInfoCreateProcess = pEventInfo->info.createProcess;

    // Now we have our name.
    if( strlen( m_Name ) == 0 )
    {
        strcpy( m_Name, m_DebugInfoCreateProcess.programName );
    }

    m_64Bit              = (m_DebugInfoCreateProcess.flag & nn::svc::CreateProcessParameterFlag_64Bit) != 0;
    uint32_t addressSpace = m_DebugInfoCreateProcess.flag & nn::svc::CreateProcessParameterFlag_AddressSpaceMask;
    m_64BitAddressSpace   = (addressSpace == nn::svc::CreateProcessParameterFlag_AddressSpace64Bit)
        || (m_DebugInfoCreateProcess.flag == nn::svc::CreateProcessParameterFlag_AddressSpace64BitOld);
}

//==============================================================================

void Process::OnProcessExited( u64 ThreadId )
{
    //TMA_TRACE("Process::OnProcessExited", "OnProcessExited:  NumActiveThreads = %d", m_NumActiveThreads );
    if( m_NumActiveThreads != 0 )
    {
        NN_SDK_LOG("ERROR:  TMA received ExitProcess notification before receiving all ExitThread notifications:  Still have %d threads active.\n", m_NumActiveThreads );
    }

    m_DebugEventThreadKill = true;

    //Are we exiting on purpose?
    if( GetProcessState() != PS_EXITING )
    {
        // TODO: HACK: Sleep to allow the LogManager to flush out logs for 1/2 second.
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 500 ) );

        // The process has ended naturally.
        NotifyHalt( tma::dbg::TMAGENT_MESSAGE_TYPE_EXCEPTION, tma::DEBUG_EXCEPTION_PROCESS_EXIT, ThreadId, (u64)0 );
    }
    else
    {
        // We've been killed
    }

    //=======================================================================
    // Even if we have no more threads, we must call Continue here so that
    // KProcess::Finalize() will be called, which cleans up any allocations.
    //=======================================================================
    ContinueDebugProcess( 0 );
    tma::threadfrozen::ClearAllThreadFrozen();
}

//==============================================================================
// Returns TRUE if this is the last thread in our process, FALSE if not.
bool Process::OnThreadExited( u64 ThreadId, bool Continue )
{
    TMA_TRACE("Process::OnThreadExited", "OnThreadExited:  %d", ThreadId );
    bool IsLastThread = false;

    m_Threads.Remove( ThreadId );

    m_NumActiveThreads -= 1;

    //Are we out of threads?
    if( m_NumActiveThreads <= 0 )
    {
        IsLastThread = true;
        OnProcessExited( ThreadId );
    }
    else if( Continue )
    {
        //=======================================================================
        // Even if we have no more threads, we must call Continue here so that
        // KProcess::Finalize() will be called, which cleans up any allocations.
        //=======================================================================
        ContinueDebugProcess( 0 );
    }

    return IsLastThread;
}

//==============================================================================

s32 Process::StartDebugProcess( u64 StopAtAddress )
{
//NN_SDK_LOG( "[tma] Process::StartDebugProcess Process %lld, processState = %d \n", GetProcessId(), GetProcessState() );
    if( GetProcessState() != PS_NOT_STARTED )
    {
        NotifyUser( "Process was previously Started\n" );
        return( tmapi::RESULT_PROCESS_ALREADY_STARTED );
    }
    if( !m_HasLoaded )
    {
        NotifyUser( "Unable to Start, no Loaded Process!\n" );
        return( tmapi::RESULT_PROCESS_NOT_LOADED );
    }

    if( StopAtAddress != 0 )
    {
        TMA_TRACE( "Process::StartDebugProcess", "Setting initial breakpoint at %p", StopAtAddress );
        dbg::BreakpointMgr* pBreakpointMgr = GetBreakpointManager();

        //-------------------------------------------------------------
        // At this point, we have no valid thread IDs.  However,
        // we don't really need a thread ID, just an address to break,
        // so the thread ID is irrelevant (and ignored in the
        // SetTempBreakpoint call).
        //-------------------------------------------------------------
        break_point BP( StopAtAddress, BP_STEP );
        pBreakpointMgr->SetBreakpoint( BP );
    }

    m_DebugEventThread.Start( DebugEventThread, this, 16 * 1024, NN_SYSTEM_THREAD_PRIORITY(tma, DbgEvents), NN_SYSTEM_THREAD_NAME(tma, DbgEvents) );
    m_DebugEventThreadStarted = true;

    SetProcessState( PS_RUNNING );

    return tmapi::RESULT_OK;
}

//==============================================================================
// Module load and unload functionality.
//==============================================================================

void Process::OnModuleLoadStart( nn::svc::DebugEventInfo* pEventInfo )
{
//NN_SDK_LOG( "[tma] Process::OnModuleLoadStart Process %lld\n", GetProcessId() );

#if TESTING
    NN_STATIC_ASSERT( sizeof(nn::ro::detail::NroHeader) == 0x80 );
    NN_STATIC_ASSERT( sizeof(nn::ro::detail::ModuleId) == 0x20 );

    NN_SDK_ASSERT( sizeof(nn::ro::detail::NroHeader) == pEventInfo->info.exception.detail.userBreak.size );
    nn::ro::detail::NroHeader header;
    nn::Result res = dbghlp::ReadDebugProcessMemory( (uintptr_t)&header, m_Handle, pEventInfo->info.exception.detail.userBreak.data, sizeof(header) );
    if( res.IsSuccess() )
    {
        const nn::ro::detail::ModuleId* pModuleId = header.GetModuleId();
        char Print[256];
        memset( Print, 0, sizeof(Print) );
        int Index;
        for( Index = 0; Index < sizeof(pModuleId->m_Data) / sizeof(pModuleId->m_Data[0]); Index += 1)
        {
            char Temp[32];
            sprintf(Temp, "%02x", pModuleId->m_Data[Index]);
            strcat( Print, Temp );
        }
        Print[Index] = '\0';
        NN_SDK_LOG( "[tma] Process::OnModuleLoadStart loading module %s\n", Print );
    }
    else
    {
        NN_SDK_LOG( "[tma] Process::OnModuleLoadComplete dbghlp::ReadDebugProcessMemory ERROR:  0x%x\n", res.GetInnerValueForDebug() );
    }
    NotifyHalt( tma::dbg::TMAGENT_MESSAGE_TYPE_EXCEPTION, tma::DEBUG_EXCEPTION_MODULE_LOADING, GetCurrentThreadId(), pEventInfo->info.exception.exceptionAddress );
#endif
    //==================================================================================================
    // Calling ContinueDebugProcess with just this event's thread Id might not be the proper response -
    // That releases just this thread, but we might need all threads.  In that case, we need to
    // set it to 0.
    //==================================================================================================
    ContinueDebugProcess( pEventInfo->threadId );
}

//==============================================================================

void Process::OnModuleLoadComplete( nn::svc::DebugEventInfo* pEventInfo )
{
//    NN_SDK_LOG( "[tma] Process::OnModuleLoadComplete Process %lld\n", GetProcessId() );
    SetProcessState( PS_HALT );
    NotifyHalt( tma::dbg::TMAGENT_MESSAGE_TYPE_EXCEPTION, tma::DEBUG_EXCEPTION_MODULE_LIST_CHANGE, GetCurrentThreadId(), pEventInfo->info.exception.exceptionAddress );
}

//==============================================================================

void Process::OnModuleUnloadStart( nn::svc::DebugEventInfo* pEventInfo )
{
//    NN_SDK_LOG( "[tma] Process::OnModuleUnloadStart Process %lld\n", GetProcessId() );
    //==================================================================================================
    // Calling ContinueDebugProcess with just this event's thread Id might not be the proper response -
    // That releases just this thread, but we might need all threads.  In that case, we need to
    // set it to 0.
    //==================================================================================================
    ContinueDebugProcess( pEventInfo->threadId );
}

//==============================================================================

void Process::OnModuleUnloadComplete( nn::svc::DebugEventInfo* pEventInfo )
{
//    NN_SDK_LOG( "[tma] Process::OnModuleUnloadComplete Process %lld\n", GetProcessId() );
    SetProcessState( PS_HALT );
    NotifyHalt( tma::dbg::TMAGENT_MESSAGE_TYPE_EXCEPTION, tma::DEBUG_EXCEPTION_MODULE_LIST_CHANGE, GetCurrentThreadId(), pEventInfo->info.exception.exceptionAddress );
}

//==============================================================================
// Exception-handling code.

void Process::HandleException( nn::svc::DebugEventInfo& ExceptionInfo )
{
    process_state ProcessState = GetProcessState();

    if( ProcessState != PS_EXITING )  //Don't bother with this if we're trying to kill the program
    {
        ASSERT( ExceptionInfo.event == nn::svc::DebugEvent_Exception );
        m_LastExceptionId           = (uint32_t)ExceptionInfo.info.exception.exceptionCode;

        TMA_TRACE("Process::HandleException", "ExceptionCode: %d, ProcessState: %d", (int)ExceptionInfo.info.exception.exceptionCode, (int)ProcessState );

#if PRINT_EXCEPTIONS

        if( true ) // ExceptionInfo.info.exception.exceptionCode != nn::svc::DebugException_UndefinedInstruction )
        {
            if( ExceptionInfo.info.exception.exceptionCode == nn::svc::DebugException_AttachBreak )
            {
                ClearBreadcrumbs();     // Reset Breadcrumbs when attaching to new process
            }
            else
            {
                EXCEPT_PRINTF( "\n==> ExceptionCode: %d, ThreadId: %d, ProcessState: %d\n", (u32)ExceptionInfo.info.exception.exceptionCode, (u32)ExceptionInfo.threadId, (u32)ProcessState );
                EXCEPT_PRINTF( "ExceptionInfo: Size = %d\n", sizeof(ExceptionInfo) );

                tma::dbg::MemToLog( &ExceptionInfo, sizeof(ExceptionInfo) );
            }
        }
#endif

        //==========================================================
        // Whoever threw this exception is now the "current" thread -
        // This syncs us up with the debug engine, which will get
        // this ID back when we send the exception info.
        // Sometimes, the thread Id the system sends us is 0.  In
        // that case, SetCurrentThreadId will ignore it and get
        // a default instead.
        //==========================================================
        SetCurrentThreadId( ExceptionInfo.threadId );

        //==========================================================
        // GetCurrentThreadId will resolve a valid thread Id in
        // the cases where the system has sent us a threadId of 0.
        //==========================================================
        uint32_t CurrentThreadId = GetCurrentThreadId();

//NN_SDK_LOG("[tma] Exception %d for thread %lld, CurrentThreadId = %d\n", ExceptionInfo.info.exception.exceptionCode, ExceptionInfo.threadId, CurrentThreadId );

        //=============================================================================================
        // Module loading exceptions come through this break.  Because this case is a fall-through
        // in the switch statement below, let's reduce risk by checking for this case outside of that
        // switch statement - That way, the code and logic remain untouched.
        //=============================================================================================
        if( ExceptionInfo.info.exception.exceptionCode == nn::svc::DebugException_UserBreak )
        {
            nn::svc::BreakReason passedReason = ExceptionInfo.info.exception.detail.userBreak.reason;
            uint32_t reason = (passedReason & (~nn::svc::BreakReason_NotificationOnlyFlag) );
//NN_SDK_LOG("[tma] Testing exception for DLL load service event: passedReason = 0x%x, reason = %d\n", passedReason, reason );

            switch( reason )
            {
            case nn::svc::BreakReason_PreLoadDll:
                OnModuleLoadStart( &ExceptionInfo );
                return;

            case nn::svc::BreakReason_PostLoadDll:
                OnModuleLoadComplete( &ExceptionInfo );
                return;

            case nn::svc::BreakReason_PreUnloadDll:
                OnModuleUnloadStart( &ExceptionInfo );
                return;

            case nn::svc::BreakReason_PostUnloadDll:
                OnModuleUnloadComplete( &ExceptionInfo );
                return;

            default:
            case nn::svc::BreakReason_Panic:
            case nn::svc::BreakReason_Assert:
            case nn::svc::BreakReason_User:
                break;
            }
        }

        ssm_exception   SSME;
        SSME.exceptionAddress   = ExceptionInfo.info.exception.exceptionAddress;
        SSME.exceptionCode      = ExceptionInfo.info.exception.exceptionCode;

        DropBreadcrumb( ( ( 0xE0000000 | (u64)ExceptionInfo.info.exception.exceptionAddress ) << 32 ) | (u64)ExceptionInfo.info.exception.detail.undefinedInstruction.code );

        switch( SSME.exceptionCode )
        {
        case nn::svc::DebugException_AttachBreak:
            TMA_TRACE( "Process::HandleException", "DebugException_AttachBreak\n" );
            break; // Ignore this exception

        case nn::svc::DebugException_UndefinedInstruction:  // Breakpoints are implemented as invalid instructions on ARM
        {
            TMA_TRACE( "Process::HandleException", "Handling UndefinedInstruction" );

            GetStepManager()->SteppingStateMachine( CurrentThreadId, SS_NONE, &SSME );

            if( GetProcessState() == PS_BKPT ) // We need to return control to TargetManager
            {
                if( GetBreakpointManager()->IsBreakpoint( ExceptionInfo.info.exception.detail.undefinedInstruction.code ) )
                {
                    EXCEPT_PRINTF( "Breakpoint Instruction %08x on thread %d at %llx\n", ExceptionInfo.info.exception.detail.undefinedInstruction.code, (u32)CurrentThreadId, SSME.exceptionAddress );
                    NotifyHalt( TMAGENT_MESSAGE_TYPE_BREAKPOINT, nn::svc::DebugException_BreakPoint, CurrentThreadId, SSME.exceptionAddress );
                }
                else
                {
                    EXCEPT_PRINTF( "Undefined Instruction %08x on thread %d at %llx\n", ExceptionInfo.info.exception.detail.undefinedInstruction.code, CurrentThreadId, SSME.exceptionAddress );
                    TMA_TRACE( "Process::HandleException", "Undefined Instruction exception: %08x", ExceptionInfo.info.exception.detail.undefinedInstruction.code );

                    NotifyHalt( TMAGENT_MESSAGE_TYPE_EXCEPTION, SSME.exceptionCode, CurrentThreadId, SSME.exceptionAddress );
                }
            }
            break;
        }

        case nn::svc::DebugException_BreakPoint:
        {
            EXCEPT_PRINTF( "Hardware Breakpoint at data address %llx\n", SSME.exceptionAddress );

            GetStepManager()->SteppingStateMachine( CurrentThreadId, SS_NONE, &SSME );

            if( GetProcessState() == PS_BKPT ) // We need to return control to TargetManager
            {
                EXCEPT_PRINTF( "Data Breakpoint on thread %d at %llx\n", (u32)CurrentThreadId, SSME.exceptionAddress );
                NotifyHalt( TMAGENT_MESSAGE_TYPE_BREAKPOINT, nn::svc::DebugException_BreakPoint, CurrentThreadId, SSME.exceptionAddress );
            }
            break;
        }

        case nn::svc::DebugException_DebuggerBreak:
            if( ProcessState == PS_HALT ) // Expected debugger break
            {
                break; // Ignore this exception
            }
            // Fall into default exception handling

        case nn::svc::DebugException_UserBreak:
        case nn::svc::DebugException_AccessViolationInstruction:
        case nn::svc::DebugException_AccessViolationData:
        case nn::svc::DebugException_DataTypeMissaligned:
        default:

            EXCEPT_PRINTF( "Default exception handling on thread %d at %llx\n", (u32)CurrentThreadId, SSME.exceptionAddress );

            GetStepManager()->SteppingStateMachine( CurrentThreadId, SS_BREAK, &SSME );

            if( SSME.exceptionCode == nn::svc::DebugException_AccessViolationData )
            {
                ThreadDefinition*       pThreadDef  = m_Threads.Find( CurrentThreadId );
                u64                     StackAddr   = pThreadDef->GetStackAddr();
                u64                     DataAddr    = SSME.exceptionAddress;
                u64                     GuardPages  = (StackAddr) - ( 16 * 1024 );

#if PRINT_EXCEPTIONS
                EXCEPT_PRINTF( "\n==> ExceptionCode: %d, ThreadId: %d (0x%X), ProcessState: %d\n",
                    (u32)ExceptionInfo.info.exception.exceptionCode, (u32)ExceptionInfo.threadId, (u32)ExceptionInfo.threadId, (u32)ProcessState );

                EXCEPT_PRINTF( "Guard Pages: %08llx - %08llx\n", GuardPages, StackAddr );

                EXCEPT_PRINTF( "tma::dbg::ThreadDefinition:     %d\n", sizeof(tma::dbg::ThreadDefinition) );
                EXCEPT_PRINTF( "nn::osdbg::ThreadInfo:          %d\n", sizeof(nn::osdbg::ThreadInfo) );
                EXCEPT_PRINTF( "tma::dbg::per_thread_step_info: %d\n", sizeof(tma::dbg::per_thread_step_info) );

                m_Threads.ToLog();
#endif

                if( ( GuardPages <= DataAddr ) && ( DataAddr < StackAddr ) )
                {
                    NN_SDK_LOG( "Process::HandleException: Data access detected in stack guard pages: %llx\n", DataAddr );
                    SSME.exceptionCode = (nn::svc::DebugException)DEBUG_EXCEPTION_STACK_OVERFLOW;
                }
            }
            NotifyHalt( TMAGENT_MESSAGE_TYPE_EXCEPTION, SSME.exceptionCode, CurrentThreadId, SSME.exceptionAddress );
            break;
        }

#if PRINT_EXCEPTIONS

        if(    ( ExceptionInfo.info.exception.exceptionCode != nn::svc::DebugException_UndefinedInstruction )
            && ( ExceptionInfo.info.exception.exceptionCode != nn::svc::DebugException_AttachBreak ) )
        {
            EXCEPT_PRINTF( "<== ExceptionCode: %d, ThreadId: %d, StepState: %d\n", (u32)ExceptionInfo.info.exception.exceptionCode, (u32)ExceptionInfo.threadId, (u32)GetProcessState() );
        }
    }
    else
    {
        EXCEPT_PRINTF( "PS_EXITING: ExceptionCode: %d, ThreadId: %d, StepState: %d\n", (u32)ExceptionInfo.info.exception.exceptionCode, (u32)ExceptionInfo.threadId, (u32)ProcessState );

#endif
    }
} // NOLINT(readability/fn_size)

//==============================================================================
// Debug event-handling thread

void* Process::DebugEventThread( void* pArg )
{
    Process*            pThis           = (Process*)pArg;
    nn::svc::Handle     TargetHandle    = pThis->GetHandle();
    bool                ExitDebugThread = false;

    // TODO: Need to be able to force this thread to exit, will need WaitSynchronization
    // below to respond to a secondary event.
    while( !ExitDebugThread && !pThis->m_DebugEventThreadKill )
    {
        nn::svc::DebugEventInfo Info;

        // Wait for an event.
        // FIXME: This WaitSynchronization needs to be a wait for multiple events.
        int32_t handleIndex = -1;
        nn::Result result;
        result = dbghlp::WaitSynchronization( &handleIndex, &TargetHandle, 1, 1 * 1000 * 1000 * 20 );

        // Exit if we are trying to kill this thread.
        if( pThis->m_DebugEventThreadKill )
        {
            break;
        }

        // No need to get the debug event if this was a wait timeout.
        if( nn::svc::ResultTimeout::Includes( result ) )
            continue;

        //TMA_TRACE( "Process::Process", "GetDebugEvent" );
        result = dbghlp::GetDebugEvent( &Info, TargetHandle );
        if( result.IsSuccess() )
        {
            ThreadDefinition::ClearThreadCache();

            //TMA_TRACE("DebugEventThread", "GetDebugEvent %d", (int)Info.event );
            switch( Info.event )
            {
            case nn::svc::DebugEvent_Exception:
                pThis->HandleException( Info );
                break;

            case nn::svc::DebugEvent_CreateThread:
                pThis->OnThreadCreated( &Info );
                break;

            case nn::svc::DebugEvent_ExitThread:
                TMA_TRACE("Process::DebugEventThread", "DebugEventThread: DebugEvent_ExitThread" );
                ExitDebugThread = pThis->OnThreadExited( Info.threadId );
//                NN_SDK_LOG("Process::DebugEventThread:  nn::svc::DebugEvent_ExitThread, Info.threadId == %lld\n", Info.threadId );
                break;

            case nn::svc::DebugEvent_ExitProcess:
                TMA_TRACE("Process::DebugEventThread", "DebugEventThread: DebugEvent_ExitProcess" );
                pThis->OnProcessExited( Info.threadId );
                ExitDebugThread = true;
//                NN_SDK_LOG("Process::DebugEventThread:  nn::svc::DebugEvent_ExitProcess, ExitDebugThread == %d\n", ExitDebugThread );
                break;

            case nn::svc::DebugEvent_CreateProcess:
                pThis->OnProcessCreated( &Info );
                TMA_TRACE( "DebugEventThread", "DebugEvent_CreateProcess: 64Bit: %s, 64BitAddr: %s",
                           ( pThis->m_64Bit ? "true" : "false" ), ( pThis->m_64BitAddressSpace ? "true" : "false" ) );
                break;

            default:
                TMA_TRACE( "DebugEventThread", "Ignored Event Code %d\n", Info.event );
                NN_SDK_LOG("[tma] Process::DebugEventThread:  Ignored Event Code %d\n", Info.event );
                //====================================================================
                // For events we don't recognize, we should let the process continue.
                //====================================================================
                pThis->ContinueDebugProcess( 0 );
            }
        }
    }

    TMA_TRACE( "DebugEventThread", "DebugEventThread ending" );
//    NN_SDK_LOG("Process::DebugEventThread:  DebugEventThread ending\n" );
    return( NULL );
}

//=========================================================================================================

bool Process::WaitForTermination()
{
//    NN_SDK_LOG( "[tma] Starting WaitForTermination for process %lld\n", m_PID );
    bool isTerminated = false;
    nn::Result result = ContinueDebugProcess( 0 );
    if( result.IsFailure() )
    {
//        NN_SDK_LOG( "[tma] WaitForTermination FAIL on ContinueDebugProcess for process %lld:  %d\n", m_PID, result.GetInnerValueForDebug() );
        return false;
    }

    while ( isTerminated == false ) //&& m_NumActiveThreads > 0 )
    {
        int32_t handleIndex = -1;
        result = dbghlp::WaitSynchronization( &handleIndex, &m_Handle, 1, 10 * 1000 * 1000 );
        if( result.IsFailure() )
        {
//            NN_SDK_LOG( "[tma] WaitForTermination FAIL on WaitSynchronization for process %lld\n", m_PID );
            return false;
        }
        nn::svc::DebugEventInfo eventInfo;
        result = dbghlp::GetDebugEvent(&eventInfo, m_Handle);
        if( result.IsFailure() )
        {
//            NN_SDK_LOG( "[tma] WaitForTermination FAIL on GetDebugEvent for process %lld:  %d\n", m_PID, result.GetInnerValueForDebug() );
            return false;
        }

        NN_SDK_LOG( "[tma] WaitForTermination received debug event %d\n", eventInfo.event );

        switch (eventInfo.event)
        {
        case nn::svc::DebugEvent_ExitProcess:
            {
                isTerminated = true;
            }
            break;

        case nn::svc::DebugEvent_CreateProcess:
            break;

        case nn::svc::DebugEvent_Exception:
            if ( eventInfo.info.exception.exceptionCode == nn::svc::DebugException_AttachBreak )
            {
                isTerminated = true;
            }
            //Ignore any others
            break;

        case nn::svc::DebugEvent_CreateThread:
            m_NumActiveThreads += 1;
            break;

        case nn::svc::DebugEvent_ExitThread:
            m_NumActiveThreads -= 1;
            if( m_NumActiveThreads <= 0 )
            {
                isTerminated = true;
            }
            break;

        default:
            break;
        }

        //=======================================================================
        // Even if we have no more threads, we must call Continue here so that
        // KProcess::Finalize() will be called, which cleans up any allocations.
        //=======================================================================
        ContinueDebugProcess( 0 );
    }

//    NN_SDK_LOG( "[tma] WaitForTermination finished for process %lld\n", m_PID);
    return true;
}

//==============================================================================

s32 Process::HaltDebugProcess()
{
    TMA_TRACE( "Process", "HaltDebugProcess()" );

    //=================================================================================================
    // If this process hasn't been started, then doing our normal stop procedure will change
    // it's state (in the SteppingStateMachine call, below).  If that happens, then StartDebugProcess
    // will not complete properly (which, if we're in the Not Started state, will presumably be
    // called soon).  This fixes SigloNTD-5584, "Attach on launch doesn't work".
    //=================================================================================================
    if( GetProcessState() == PS_NOT_STARTED )
    {
        return tmapi::RESULT_PROCESS_NOT_STARTED;
    }

    if( !m_Handle.IsValid() )
    {
        return tmapi::RESULT_PROCESS_NOT_RUNNING;
    }

    if( GetProcessState() == PS_BKPT )
    {
        return( tmapi::RESULT_PROCESS_HALTED );
    }

    dbghlp::BreakDebugProcess( m_Handle );
    m_StepManager.SteppingStateMachine( GetCurrentThreadId(), SS_BREAK ); // Clean up SSM

    m_ModulesChanged = true; //TODO - Is this overkill?

    return( tmapi::RESULT_OK );
}

//==============================================================================

s32 Process::Continue( s32 ThreadId )
{
    ASSERT(0); // Replaced by Step
    process_state ProcessState = GetProcessState();

    if( ProcessState == PS_RUNNING )
    {
        return tmapi::RESULT_PROCESS_IS_RUNNING;
    }
    if( ProcessState == PS_NOT_STARTED )
    {
        return tmapi::RESULT_PROCESS_NOT_RUNNING;
    }

    m_StepManager.SteppingStateMachine( ThreadId, SS_CONT );

    return( tmapi::RESULT_OK );
}

//==============================================================================

s32 Process::Step( Command* pCommand )
{
    StepDebugCommandArgs* pArgs = (StepDebugCommandArgs*)pCommand->m_pData;
    if( pArgs == NULL )
    {
        return tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
    }
    TMA_TRACE( "Process::Step", "StepArgs - ThreadId:%d, StepKind:%d, IP:%llx, Instr:%08x, LO=%llx, HI=%llx",
                pArgs->m_ThreadId, pArgs->m_StepKind, pArgs->m_IP, pArgs->m_Instruction,
                pArgs->m_InstrRange[0].m_AddrLo, pArgs->m_InstrRange[0].m_AddrHi );

    if( pArgs->m_StepKind == STEP_CONT ) // Step is Continue
    {
        process_state ProcessState = GetProcessState();

        if( ProcessState == PS_RUNNING )
        {
            return tmapi::RESULT_PROCESS_IS_RUNNING;
        }
        if( ProcessState == PS_NOT_STARTED )
        {
            return tmapi::RESULT_PROCESS_NOT_RUNNING;
        }
    }

    step_state State = m_StepManager.SetSteppingInfo( pArgs );

    m_StepManager.SteppingStateMachine( pArgs->m_ThreadId, State );

    return( tmapi::RESULT_OK );
}

//==============================================================================

bool Process::HaltResume( process_state State )
{
    nn::Result result;

    bool Success = false;

    // Special handling of Halt/Resume to avoid disrupting state machine

    if( State == PS_HALT )
    {
        if( m_ProcessState != PS_RUNNING )
        {
            return( false ); // No reason to continue again
        }
        TMA_PRINTF( "HaltResume( PS_HALT ): State: %d, ThreadId: %d\n", m_ProcessState, (u32)m_CurrentThreadId );

        SetProcessState( State );

        result = dbghlp::BreakDebugProcess( m_Handle );

        Success = result.IsSuccess();
        if( !Success )
        {
            TMA_PRINTF( "HaltResumme( PS_HALT ) failed\n" );
            SetProcessState( PS_RUNNING );
        }
        return( Success );
    }

    if( State == PS_RESUME ) // PS_RESUME never retained as the state
    {
        if( m_ProcessState == PS_HALT)
        {
            result = ContinueDebugProcess( 0 );

            Success = result.IsSuccess();
            if( !Success )
            {
                PurgeEventsAndContinue();
            }
            TMA_PRINTF( "HaltResume( PS_RESUME ): ThreadId: %d\n", (u32)m_CurrentThreadId );
            SetProcessState( PS_RUNNING );
        }
    }
    return( Success );
}

//==============================================================================

s32 Process::SetBreakpoint( break_point& BP )
{
    if( !m_HasLoaded )
    {
        return tmapi::RESULT_PROCESS_NOT_LOADED;
    }

    bool bShouldResume = HaltResume( PS_HALT );

    bp_state State = BP.m_State;
    m_BreakManager.SetBreakpoint( BP );

    if( bShouldResume )
    {
        HaltResume( PS_RESUME );
    }
    return( tmapi::RESULT_OK );
}

//==============================================================================

void Process::PurgeEventsAndContinue()
{
    TMA_TRACE( "Process::PurgeEventsAndContinue", "PurgeEventsAndContinue" );

    //TODO - Review this code.  This could ignore serveral Debug Events!

    nn::Result Result = ContinueDebugProcess( 0 );
    while( nn::svc::ResultBusy().Includes( Result ) )
    {
        nn::svc::DebugEventInfo Info;
        dbghlp::GetDebugEvent( &Info, m_Handle );
        Result = ContinueDebugProcess( 0 );
    }
}

//==============================================================================

s32 Process::ReadMemory( u64 Address, void* pBuffer, u32 nBytes )
{
    TMA_TRACE( "Process", "ReadMemory( %llx, %p, %d )", Address, pBuffer, nBytes );
    bool bShouldResume = HaltResume( PS_HALT );
    s32 Ret = tmapi::RESULT_OK;

    nn::Result Result = dbghlp::ReadDebugProcessMemory( (uintptr_t)pBuffer, m_Handle, (uintptr_t)Address, nBytes );
    if( Result.IsFailure() ) //nn::svc::ResultInvalidAddressState().Includes( Result ) )
    {
        memset( pBuffer, 0xFF, nBytes );
        Ret = tmapi::RESULT_INVALID_PARAMETER;
        TMA_TRACE( "Process", "Read FAILED at %llx, Result = %d", Address, Result.GetInnerValueForDebug() );
    }
    else
    {
        m_BreakManager.PatchReadMemory( pBuffer, Address, nBytes );
    }

    if( bShouldResume )
    {
        HaltResume( PS_RESUME );
    }

    return Ret;
}

//==============================================================================

s32 Process::WriteMemory( u64 Address, void* pBuffer, u32 nBytes )
{
    //NN_SDK_LOG( "Process::WriteMemory( %llx, %p, %d ): %x\n", Address, pBuffer, nBytes, (nBytes >= 4) ? *((u32*)pBuffer) : 0 );

    s32 Ret             = tmapi::RESULT_OK;
    bool bShouldResume  = HaltResume( PS_HALT );
    nn::Result Result   = dbghlp::WriteDebugProcessMemory( m_Handle, (uintptr_t)pBuffer, (uintptr_t)Address, nBytes );
    if( Result.IsFailure() )
    {
        //Are they wanting an invalid location?
        if( (nn::svc::ResultInvalidAddressState().Includes( Result ) == false) )
        {
            //See if this failed because our handle has gone bad.
            Ret = CheckProcessHandleForError( m_Handle );
        }
        else
        {
            Ret = tmapi::RESULT_INVALID_ADDRESS;
        }
    }

    if( bShouldResume )
    {
        HaltResume( PS_RESUME );
    }

    return Ret;
}

//==============================================================================

s32 Process::WriteMemory( Command* pCommand )
{
    s32 Result = tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
    WriteMemoryCommandArgs* pArgs = (WriteMemoryCommandArgs*)pCommand->m_pData;
    if( pArgs != NULL )
    {
        //Are these arguments validly formed?
        if( pArgs->HasValidData() == true )
        {
            Result = WriteMemory( pArgs->m_WriteToAddress, pArgs->m_Data, pArgs->m_SizeOfData );
        }
        else
        {
           TMA_TRACE( "s32 Process::WriteMemory", "Bad argument data for write memory command:  SizeOfData = %d, WriteTo = %x\n", pArgs->m_SizeOfData, (pArgs->m_WriteToAddress >> 32) );
        }
    }

    return Result;
}

//==============================================================================

s32 Process::ReadRegisterDefinitions( u32& BufferSize, void* pReturnBuffer )
{
    TMA_TRACE( "Process::ReadRegisterDefinitions", "ReadRegisterDefinitions( %d )", BufferSize );

    if( m_pRegisterDefinitions == NULL )
    {
        TMA_TRACE( "Process::ReadRegisterDefinitions", "64Bit: %s, 64BitAddr: %s",
                   ( Is64Bit() ? "true" : "false" ), ( Is64BitAddressSpace() ? "true" : "false" ) );

        m_SizeRegisterDefs = 0;
        tmapi::result Res = GetRegisterDefinitions ( Is64Bit(), Is64BitAddressSpace(), m_SizeRegisterDefs, (void*)&m_SizeRegisterDefs );

        void* pMem = s_Allocate( sizeof( u8 ) * m_SizeRegisterDefs );
        m_pRegisterDefinitions  = new (pMem) u8[m_SizeRegisterDefs];

        GetRegisterDefinitions( Is64Bit(), Is64BitAddressSpace(), m_SizeRegisterDefs, m_pRegisterDefinitions );
    }

    s32 Result = tmapi::result::RESULT_OK;

    if( pReturnBuffer == NULL )
    {
        Result = tmapi::result::RESULT_INVALID_PARAMETER;
    }
    else if( BufferSize < m_SizeRegisterDefs )
    {
        Result = tmapi::result::RESULT_BUFFER_TOO_SMALL;
        *((u32*)pReturnBuffer) = m_SizeRegisterDefs;
    }
    else
    {
        memcpy( pReturnBuffer, m_pRegisterDefinitions, m_SizeRegisterDefs );
    }
    BufferSize = m_SizeRegisterDefs;

    return( Result );
} // NOLINT(readability/fn_size)

//==============================================================================

s32 Process::ReadRegisterData( Command* pCommand, void* pReturnBuffer )
{
    if( !m_HasLoaded )
    {
        return( tmapi::RESULT_PROCESS_NOT_LOADED );
    }
    if( GetProcessState() == PS_RUNNING )
    {
        return( tmapi::RESULT_PROCESS_NOT_HALTED );
    }
    ReadRegDataCommandArgs* pArgs = (ReadRegDataCommandArgs*)pCommand->m_pData;

    if( pArgs == NULL )
    {
        return( tmapi::RESULT_PROCESS_INVALID_ARGUMENTS );
    }

    TMA_TRACE( "Process", "ReadRegisterData( %d, %d )", pArgs->m_ThreadId, pArgs->m_DataSize );

    if( pArgs->m_DataSize < sizeof(tma::dbg::thread_context) )
    {
        return( tmapi::RESULT_BUFFER_TOO_SMALL );
    }

    tma::dbg::thread_context ThreadContext;
    nn::Result Res = ThreadDefinition::GetThreadContext( &ThreadContext, m_Handle, pArgs->m_ThreadId, ALL_REGISTERS );

    if( Res.IsSuccess() )
    {
        memcpy( pReturnBuffer, (void*)&ThreadContext, sizeof(tma::dbg::thread_context) );
        return( tmapi::RESULT_OK );
    }

    if( nn::svc::ResultInvalidThreadId().Includes( Res ) )
    {
        return tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
    }

    //If we couldn't read the registers, see if our handle has gone bad
    return CheckProcessHandleForError( m_Handle );
}

//==============================================================================

s32 Process::WriteRegister( s32 ThreadId, s32 RegisterId, u64 Value )
{
    //Are these arguments validly formed?
    if( (ThreadId >= 0) && ( (RegisterId >= 0) || (RegisterId == CF_REG_IP) ) )
    {
        nn::svc::ThreadContext ThreadContext;
        nn::Result Res = dbghlp::GetDebugThreadContext( &ThreadContext, m_Handle, ThreadId, ALL_REGISTERS );

        if( Res.IsSuccess() )
        {
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
            //s32 nCpuRegisters = ( ((char*)&ThreadContext.v[0]) - ((char*)&ThreadContext.r[0]) ) / sizeof(ThreadContext.r[0]); // Include FP, SP, LR, PC, and PSTATE
            s32 nFpuRegisters = sizeof(ThreadContext.v) / sizeof(ThreadContext.v[0]);
#else
            NN_ABORT("This function does not support ARMv7 now.");
            s32 nCpuRegisters = 0; // FIXME
            s32 nFpuRegisters = 0; //FIXME
#endif
            // Change the value of the register in the thread context

            if( RegisterId == CF_REG_IP ) // ARM32/64 independent register id for IP
            {
                ThreadContext.pc = Value;
            }

            else if( RegisterId == ARM64_REGISTER_FP )
            {
                ThreadContext.fp = Value;
            }

            else if( RegisterId == ARM64_REGISTER_LR )
            {
                ThreadContext.lr = Value;
            }

            else if( RegisterId == ARM64_REGISTER_SP )
            {
                ThreadContext.sp = Value;
            }

            else if( RegisterId == ARM64_REGISTER_PC )
            {
                ThreadContext.pc = Value;
            }

            else if( RegisterId == ARM64_REGISTER_CONTROL )
            {
                ThreadContext.pstate = (nn::Bit32)Value;
            }

            else if( (RegisterId == ARM32_REGISTER_SP) && !m_64Bit )
            {
                ThreadContext.sp = Value;
            }

            else if( (RegisterId == ARM32_REGISTER_LR) && !m_64Bit )
            {
                ThreadContext.lr = Value;
            }

            else if( (RegisterId == ARM32_REGISTER_PC) && !m_64Bit )
            {
                ThreadContext.pc = Value;
            }

            else if( (RegisterId == ARM32_REGISTER_CONTROL) && !m_64Bit )
            {
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
                ThreadContext.pstate = (nn::Bit32)Value;
#endif
            }

            //If not one of our control registers, is it one of the other CPU regs?
            else if( ( 0 <= RegisterId ) && ( RegisterId < ARM64_REGISTER_FP ) )
            {
                ThreadContext.r[ RegisterId ] = Value;
            }

            else if( ( 256 <= RegisterId ) && ( RegisterId < ( 256 + nFpuRegisters ) ) ) // 64-/128-bit float registers
            {
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
                if( m_64Bit )
                {
                    *((u64*)&ThreadContext.v[ RegisterId - 256 ]) = Value; // Value is low 64-bits of 128-bit register
                }
                else
                {
                    ((u64*)&ThreadContext.v[0])[ RegisterId - 256 ] = Value; // 64-bit registers are packed in 32-bit mode
                }
#endif
            }

            else if( m_64Bit && ( 512 <= RegisterId ) && ( RegisterId < ( 512 + nFpuRegisters ) ) ) // 128-bit float regs (High 64-bits only)
            {
#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
                s32 RegId = RegisterId - 512;
                ((u64*)&ThreadContext.v[0])[ RegId + RegId + 1 ] = Value; // Overlay only the high 64-bits
#endif
            }

            else
            {
                //Don't do anything - we don't know this register.
                return( tmapi::RESULT_PROCESS_UNKNOWN_REGISTER );
            }
            //And commit it.

            //u32* p = (u32*)&ThreadContext;                          // Keep these lines for context debugging
            //s32  n = sizeof(ThreadContext) / sizeof(u32);
            //NN_SDK_LOG( "WRITE nn::svc::ThreadContext(%d): Size = %d\n", ThreadId, n );
            //for( s32 i = 0; i < n; i += 8 )
            //{
            //    for( s32 j = 0; (j < 8) && ((i + j) < n); j++ )
            //    {
            //        NN_SDK_LOG( " %08x", *p++ );
            //    }
            //    NN_SDK_LOG( "\n" );
            //}

            Res = dbghlp::SetDebugThreadContext( m_Handle, ThreadId, ThreadContext, ALL_REGISTERS );
            ThreadDefinition::ClearThreadCache();

#if TEST_REGISTER_WRITE
            {
                nn::svc::ThreadContext TestThreadContext;
                nn::Result Res = ThreadDefinition::GetThreadContext( &TestThreadContext, m_Handle, ThreadId, ALL_REGISTERS );

                if( (RegisterId >= 0) && (RegisterId < nCpuRegisters) )
                {
                    TMA_TRACE("Process::WriteRegister", "WriteRegister thread %d:  After write, register %d = %p", ThreadId, RegisterId, TestThreadContext.cpuRegisters[RegisterId]);
                }
                else if( RegisterId == ARM32_REGISTER_SP )
                {
                    TMA_TRACE("Process::WriteRegister", "WriteRegister thread %d:  After write, SP = %p", ThreadId, TestThreadContext.sp);
                }
                else if( RegisterId == ARM32_REGISTER_PC )
                {
                    TMA_TRACE("Process::WriteRegister", "WriteRegister thread %d:  After write, PC = %p", ThreadId, TestThreadContext.pc);
                }
                else if( RegisterId < (nCpuRegisters + nFpuRegisters) )
                {
                    TMA_TRACE("Process::WriteRegister", "WriteRegister thread %d:  After write, register %d = %p", ThreadId, RegisterId, TestThreadContext.fpuRegisters[RegisterId - nCpuRegisters] );
                }
            }
#endif
            if( Res.IsSuccess() )
            {
                return tmapi::RESULT_OK;
            }
        }

        else if( nn::svc::ResultInvalidThreadId().Includes( Res ) )
        {
            return tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
        }

        //Made it here so the read failed.  Check our handle.
        return CheckProcessHandleForError( m_Handle );
    }

    return tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
} // NOLINT(readability/fn_size)

//==============================================================================

s32 Process::WriteRegister( Command* pCommand )
{
    if( !m_HasLoaded )
    {
        return( tmapi::RESULT_PROCESS_NOT_LOADED );
    }
    if( GetProcessState() == PS_RUNNING )
    {
        return( tmapi::RESULT_PROCESS_NOT_HALTED );
    }

    WriteRegisterCommandArgs* pArgs = (WriteRegisterCommandArgs*)pCommand->m_pData;

    //Are these arguments validly formed?
    if( (pArgs != NULL) && pArgs->HasValidData() )
    {
        return( WriteRegister( pArgs->m_ThreadId, pArgs->m_RegisterId, pArgs->m_Value ) );
    }

    return tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
}

//==============================================================================

s32 Process::StopDebugProcess()
{
    TMA_TRACE("Process::StopDebugProcess", "Process::StopDebugProcess" );
    return KillProcess();
}

//==============================================================================

s32 Process::KillProcess()
{
    return Close(true);
}

//==============================================================================

s32 Process::GetNumberOfModules( s32& NumberOfModules )
{
    NumberOfModules = 0;
    if( !m_HasLoaded )
    {
        return( tmapi::RESULT_PROCESS_NOT_LOADED );
    }
    //if( GetProcessState() == PS_RUNNING )
    //{
    //    return( tmapi::RESULT_PROCESS_NOT_HALTED );
    //}

    NumberOfModules = m_Modules.GetCount();
    return tmapi::RESULT_OK;
}

//==============================================================================

s32 Process::GetNumberOfThreads()
{
    s32 NumberOfThreads = m_Threads.Count();
    return NumberOfThreads;
}

//==============================================================================

s32 Process::GetThreadDetails( s32 AtIndex, ThreadDefinition** ppThreadDetails )
{
    if( ( AtIndex >= 0 ) && ( AtIndex < m_Threads.Count() ) )
    {
        *ppThreadDetails = m_Threads.Get( AtIndex );
        return tmapi::RESULT_OK;
    }

    return tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
}

//==============================================================================

s32 Process::GetModuleInfo( s32 Index, void* Buffer )
{
    if( !m_HasLoaded )
    {
        return( tmapi::RESULT_PROCESS_NOT_LOADED );
    }
    //if( GetProcessState() == PS_RUNNING )
    //{
    //    return( tmapi::RESULT_PROCESS_NOT_HALTED );
    //}
    if( Index < m_Modules.GetCount() )
    {
        m_Modules.GetModuleAtIndex( Index, Buffer );
    }
    else
    {
        return tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
    }

    return tmapi::RESULT_OK;
}

//==============================================================================

void Process::ResetState()
{
    m_HasAttached   = false;
    m_HasLoaded     = false;
    SetProcessState( PS_NOT_STARTED );
    // Needed for reference after cleanup in ProcessMgr
    // m_PID           = 0;
    m_Handle  = nn::svc::INVALID_HANDLE_VALUE;
}

//==============================================================================

TMAgent_message_type Process::HandleCommand( Command* pCommand, s32& ResultCode, void* pReplyBuffer )
{
    s32 ReplyValue = 0;

    //Init these to known values.
    TMAgent_message_type RetValue = TMAGENT_MESSAGE_TYPE_UNKNOWN;
    ResultCode = tmapi::RESULT_NOT_IMPLEMENTED;

    u64 ParameterValue = pCommand->m_Parameter;
    TMA_TRACE("Process::HandleCommand", "Command( %d, %llx, %llx, %d, %p )", pCommand->m_Command, pCommand->m_ProcessId, pCommand->m_Parameter, pCommand->m_DataSize, pCommand->m_pData );
//NN_SDK_LOG( "[tma] Process::HandleCommand %d, State = %d\n", pCommand->m_Command, m_ProcessState );

    //=========================================================
    // Update now, if we can.  Notice that this returns
    // immediately if we've already updated all of our fields.
    //=========================================================
    if( pCommand->m_Command == TMA_OP_HALT_DEBUG )
    {
        TMA_TRACE( "Process", "OP_HALT_DEBUG" );
        ResultCode = HaltDebugProcess();
        RetValue   = TMAGENT_MESSAGE_TYPE_RESULT;
        Update(); // Make sure we are Halted before updating
    }
    else
    {
        Update(); // Make sure we are updated before sending back state information
        switch( pCommand->m_Command )
        {
            case TMA_OP_START_DEBUG:
                TMA_TRACE( "Process", "OP_START_DEBUG" );
                ResultCode = StartDebugProcess( ParameterValue );
                RetValue   = TMAGENT_MESSAGE_TYPE_RESULT;
                break;

            case TMA_OP_CONT_DEBUG:
                //ASSERT(0); // Replaced by Step
                //ResultCode = Continue( ParameterValue );
                //====================================================
                // Now when we get this call, it's to continue after
                // a servicing event.
                //====================================================
                ContinueDebugProcess( 0 );
                ResultCode = tmapi::RESULT_OK;
                RetValue   = TMAGENT_MESSAGE_TYPE_RESULT;
//NN_SDK_LOG( "[tma] Process::HandleCommand ContinueDebugProcess complete.\n" );
                break;

            case TMA_OP_STEP_DEBUG:
                ResultCode = Step( pCommand );
                RetValue   = TMAGENT_MESSAGE_TYPE_RESULT;
                break;
            case TMA_OP_SET_BKPT:
                {
                    break_point* pBP = (break_point*)pCommand->m_pData;
                    ResultCode = tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
                    if( (pBP != 0) && (pReplyBuffer != 0) )
                    {
                        ResultCode = SetBreakpoint( *pBP );
                        memcpy( pReplyBuffer, pBP, sizeof(break_point) );
                    }
                    RetValue   = TMAGENT_MESSAGE_TYPE_RAW;
                    break;
                }
            case TMA_OP_READ_MEM:
                {
                    ReadMemCommandArgs* pArgs = (ReadMemCommandArgs*)pCommand->m_pData;
                    ResultCode = tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
                    if( pArgs != NULL )
                    {
                        TMA_TRACE( "Process::HandleCommand", "ReadMemory( %llx, %d )", pArgs->m_Address, pArgs->m_DataSize );
                        ResultCode = ReadMemory( pArgs->m_Address, pReplyBuffer, pArgs->m_DataSize );
                    }
                }
                RetValue = TMAGENT_MESSAGE_TYPE_RAW;
                break;
            case TMA_OP_WRITE_MEM:
                ResultCode = WriteMemory( pCommand );
                RetValue   = TMAGENT_MESSAGE_TYPE_RESULT;
                break;
            case TMA_OP_READ_REG_DEFS:
                {
                    ReadRegDefsCommandArgs* pArgs = (ReadRegDefsCommandArgs*)pCommand->m_pData;
                    ResultCode = tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
                    if( pArgs != NULL )
                    {
                        TMA_TRACE( "Process::HandleCommand", "ReadRegisterDefinitions( %d, %p )", pArgs->m_DataSize, pReplyBuffer );
                        ResultCode = ReadRegisterDefinitions( pArgs->m_DataSize, pReplyBuffer );
                    }
                }
                RetValue   = TMAGENT_MESSAGE_TYPE_RAW;
                break;
            case TMA_OP_READ_REG_DATA:
                ResultCode = ReadRegisterData( pCommand, pReplyBuffer );
                RetValue   = TMAGENT_MESSAGE_TYPE_RAW;
                break;
            case TMA_OP_WRITE_REG:
                ResultCode = WriteRegister( pCommand );
                RetValue   = TMAGENT_MESSAGE_TYPE_RESULT;
                break;
            case TMA_OP_NUM_MODULES:
                ResultCode = GetNumberOfModules( ReplyValue );
                RetValue = TMAGENT_MESSAGE_TYPE_NUMERIC_S32;
                break;
            case TMA_OP_MODULE_INFO:
                ResultCode = GetModuleInfo( ParameterValue, pReplyBuffer );
                RetValue = TMAGENT_MESSAGE_TYPE_MODULE;
                break;
            case TMA_OP_NUM_THREADS:
                ResultCode = tmapi::RESULT_OK;
                ReplyValue = m_Threads.Count();
                RetValue = TMAGENT_MESSAGE_TYPE_NUMERIC_S32;
                break;
            case TMA_OP_THREAD_INFO:
                ResultCode = GetThreadInfo( ParameterValue, pReplyBuffer );
                RetValue = TMAGENT_MESSAGE_TYPE_THREAD;
                break;
            case TMA_OP_MAIN_THREAD:
                ResultCode = tmapi::RESULT_OK;
                ReplyValue = GetCurrentThreadId();
                RetValue = TMAGENT_MESSAGE_TYPE_NUMERIC_S32;
                break;

            // Deprecated:
            case TMA_OP_NUM_MEMORY_SEGMENTS:
                ResultCode = tmapi::RESULT_NOT_IMPLEMENTED;
                ReplyValue = 0;
                RetValue = TMAGENT_MESSAGE_TYPE_NUMERIC_S32;
                break;
            case TMA_OP_MEMORY_SEGMENT_INFO:
                ResultCode = tmapi::RESULT_NOT_IMPLEMENTED;
                ReplyValue = tmapi::RESULT_NOT_IMPLEMENTED;
                RetValue = TMAGENT_MESSAGE_TYPE_NUMERIC_S32;
                break;

            default:
                ResultCode = tmapi::RESULT_NOT_IMPLEMENTED;
                ReplyValue = tmapi::RESULT_NOT_IMPLEMENTED;
                break;
        }
    }

    switch( RetValue )
    {
        case TMAGENT_MESSAGE_TYPE_NUMERIC_S32:
            *(s32*)pReplyBuffer = ReplyValue;
            break;

        case TMAGENT_MESSAGE_TYPE_STRING:
            if( strlen( (char*)pReplyBuffer ) == 0)
            {
                ((char*)pReplyBuffer)[0] = ' ';
            }
            break;
        default:
            break;
    }

    return RetValue;
} // NOLINT(readability/fn_size)

//==============================================================================

void Process::SetCurrentThreadId( uint32_t ThreadId )
{
    if( ThreadId > 0 )
    {
        m_CurrentThreadId = ThreadId;
    }
    else
    {
        //UpdateThreadList();
        m_CurrentThreadId = m_Threads.GetDefaultId();
    }
}

//==============================================================================

uint32_t Process::GetCurrentThreadId()
{
    // Do we need a new thread Id?
    if( m_CurrentThreadId == TMA_NO_THREAD_ID || m_CurrentThreadId == 0 )
    {
        m_CurrentThreadId = m_Threads.GetDefaultId();
        if( m_CurrentThreadId == 0 )
        {
            UpdateThreadList();
            m_CurrentThreadId = m_Threads.GetDefaultId();
        }
    }

    return m_CurrentThreadId;
}

//==============================================================================

s32 Process::GetThreadInfo( s32 Index, void* pBuffer )
{
    s32 Result = tmapi::RESULT_OK;
    if ( Index >= 0 && Index < m_Threads.Count() )
    {
        ThreadDefinition* pThreadDef = m_Threads.Get(Index);
        if (pThreadDef != NULL)
        {
            ThreadData data( pThreadDef );
            memcpy( pBuffer, &data, sizeof(data) );
            //NN_SDK_LOG( "GetThreadInfo: Thread %lld priority = %d\n", pThreadDef->GetThreadId(), pThreadDef->GetPriority()  );
        }
        else
        {
            TMA_TRACE( "Process::GetThreadInfo", "GetThreadInfo:  Unable to find thread data for index %d", Index );
        }
    }
    else
    {
        TMA_TRACE("Process::GetThreadInfo", "GetThreadInfo:  Invalid thread index:  Requesting %d, but only have %d", Index, m_Threads.Count());
        return tmapi::RESULT_PROCESS_INVALID_ARGUMENTS;
    }

    return Result;
}

//==============================================================================

void Process::NotifyHalt( TMAgent_message_type HaltType, uint32_t HaltId, s32 ThreadId, u64 ExceptionAddress )
{
    EXCEPT_PRINTF( "Process::NotifyHalt: Exception %d at %llx on thread %d\n", HaltId, ExceptionAddress, ThreadId );
    TMA_TRACE( "Process", "NotifyHalt" );

    tma::dbg::Exception exceptionData;
    exceptionData.m_ThreadId            = ThreadId;
    exceptionData.m_ExceptionId         = HaltId;
    exceptionData.m_ExceptionAddress    = ExceptionAddress;

    ThreadDefinition::GetState( m_Handle, ThreadId, &exceptionData.m_SP, &exceptionData.m_IP );

    //==============================================================================
    // FOR NOW:  Update the modules here and return the "update the modules" flag.
    // TODO:  Respond to kernel event.
    //==============================================================================
    exceptionData.m_ModulesChanged = m_Modules.Update();
    if ( exceptionData.m_ModulesChanged )
    {
        //NN_SDK_LOG("[tma] Process::NotifyHalt marking modules changed.\n");
        exceptionData.m_ModulesChanged = 1;
        m_ModulesChanged = false;
    }

    //==============================================================================
    // Save our last reported exception.  Moved here because the actual exception
    // may have been caused by a breakpoint, and when it's reached this function,
    // it's been cleaned for public consumption.
    //==============================================================================
    m_LastExceptionId = HaltId;

    DebugService& Svc = GetDebugService();
    Svc.NotifyException( m_PID, &exceptionData );
}

//==============================================================================

void Process::NotifyUser( const char* pMessage )
{
//    TMA_TRACE( "Process::NotifyUser", "Sending message %s", pMessage );
    DebugService& Svc = GetDebugService();
    Svc.NotifyMessage( m_PID, pMessage );
}

//==============================================================================

nn::Result Process::ContinueDebugProcess( u64 ThreadId )
{
    nn::Bit32 Size = 0;
    nn::Bit32 Flags = 0;
    nn::Bit64* ThreadIds = NULL;
    Flags |= nn::svc::ContinueFlag_ExceptionHandled | nn::svc::ContinueFlag_EnableExceptionEvent;
    if( ThreadId == 0 )
    {
        ThreadIds = tma::threadfrozen::GetFrozenThreadIds( Size );
        Flags |= nn::svc::ContinueFlag_ContinueOthers;
    }
    else
    {
        Size = 1;
        ThreadIds = (nn::Bit64*)s_Allocate( sizeof( nn::Bit64 ) );
        ThreadIds[0] = ThreadId;

        bool Frozen = false;
        tma::threadfrozen::GetThreadFrozen( ThreadId, Frozen );
        if( Frozen )
        {
            // Not running any threads here deadlocks the system.
            // Whatever state this call is trying to resolve can wait until this thread is unfrozen.
            Flags |= nn::svc::ContinueFlag_ContinueOthers;
        }
    }

    m_NeedUpdate = true;

    nn::Result ret = dbghlp::ContinueDebugEvent( m_Handle, Flags, ThreadIds, Size );

    if( ThreadIds != NULL )
    {
        s_Deallocate( ThreadIds, Size * sizeof( nn::Bit64 ) );
        ThreadIds = NULL;
    }

    return ret;
}

//==============================================================================
}} // namespace
//==============================================================================
