﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "coredump_service.h"
#include "coredump_task.h"

#if DEBUG
#define COREDUMP_LOG(...) NN_SDK_LOG( "[tma][coredump] - " ); NN_SDK_LOG( __VA_ARGS__ ); NN_SDK_LOG( "\n" )
#else
#define COREDUMP_LOG(...)
#endif

//==============================================================================
namespace tma { namespace coredump {
//==============================================================================

struct working_task
{
    working_task*   m_pNext;
    CoredumpTask*   m_pTask;

    //==============================================================================

    explicit working_task( CoredumpTask* pTask )
    {
        COREDUMP_LOG( "working_task %d", pTask->GetTaskId() );
        m_pTask = pTask;
        m_pNext = NULL;
    }

    virtual ~working_task()
    {
        COREDUMP_LOG( "~working_task %d", m_pTask->GetTaskId() );
        delete m_pTask;
    }

    //==============================================================================

    uint32_t GetId()
    {
        return m_pTask->GetTaskId();
    }

    //==============================================================================

    bool Is( uint32_t TaskId )
    {
        return ( m_pTask->GetTaskId() == TaskId );
    }

    //==============================================================================
    // Static list maintenance functionality
    //==============================================================================

    static uint32_t add( working_task* pAddTask, working_task** pAddToList )
    {
        COREDUMP_LOG( "add %d", pAddTask->GetId() );
        working_task* pList = *pAddToList;
        if( pList == NULL )
        {
            *pAddToList = pAddTask;
        }
        else
        {
            while( pList->m_pNext != NULL )
            {
                pList = pList->m_pNext;
            }
            pList->m_pNext = pAddTask;
        }
        return pAddTask->GetId();
    }

    //==============================================================================

    static working_task* find( uint32_t Find, working_task* pList )
    {
        COREDUMP_LOG( "find %d", Find );
        while( pList != NULL )
        {
            if( pList->Is( Find ) )
            {
                break;
            }
            pList = pList->m_pNext;
        }

        return pList;
    }

    //==============================================================================

    static working_task* find( tmipc::Task* pTask, working_task* pList )
    {
        return find( pTask->GetTaskId(), pList );
    }

    //==============================================================================

    static working_task* remove( uint32_t Find, working_task** pFromList )
    {
        COREDUMP_LOG( "remove %d", Find );
        working_task* pList = *pFromList;
        if( pList->Is( Find ) )
        {
            *pFromList = pList->m_pNext;
        }
        else
        {
            working_task* pPrev = pList;
            pList = pList->m_pNext;
            while( pList  )
            {
                if( pList->Is( Find ) )
                {
                    break;
                }

                pPrev = pList;
                pList = pList->m_pNext;
            }
            if( pPrev != NULL )
            {
                pPrev->m_pNext = pList->m_pNext;
            }
        }
        return pList;
    }

    //==============================================================================

    static void clear( working_task** pClearList )
    {
        COREDUMP_LOG( "clear" );
        working_task* pList = *pClearList;
        while( pList != NULL )
        {
            working_task* pNext = pList->m_pNext;
            delete pList;
            pList = pNext;
        }

        *pClearList = NULL;
    }

    //==============================================================================
};

//==============================================================================

Service::Service()
:   m_pProcessMgr( NULL )
,   m_pWorkingTasks( NULL )
,   m_pCompletedTasks( NULL )
{
    m_ServiceId = HashString( "CoredumpService" );
    m_TaskListMutex.Create();
    COREDUMP_LOG( "Service::Service" );
}

//==============================================================================

Service::~Service()
{
    m_TaskListMutex.Destroy();
    COREDUMP_LOG( "Service::~Service" );
}

//==============================================================================

void Service::Init( tma::dbg::ProcessMgr* pProcessMgr )
{
    COREDUMP_LOG( "Service::Init" );
    Create();
    m_pProcessMgr = pProcessMgr;
}

//==============================================================================

void Service::Kill()
{
    COREDUMP_LOG( "Service::Kill" );
    Destroy();

    // Kill all our tasks
    working_task::clear( &m_pWorkingTasks );
    working_task::clear( &m_pCompletedTasks );
}

//==============================================================================
// Task list maintenance functionality
//==============================================================================

void Service::StartedTask( tmipc::Task* pTask )
{
    COREDUMP_LOG( "Service::StartedTask:  %d", pTask->GetTaskId() );
    ScopedLock Lock( m_TaskListMutex );
    working_task::add( new working_task( (CoredumpTask*)pTask ), &m_pWorkingTasks );
    working_task::clear( &m_pCompletedTasks );
}

//==============================================================================

void Service::OnComplete( tmipc::Task* pTask )
{
    COREDUMP_LOG( "Service::OnComplete:  %d", pTask->GetTaskId() );
    ScopedLock Lock( m_TaskListMutex );
    working_task::clear( &m_pCompletedTasks );
    working_task* pCompleted = working_task::remove( pTask->GetTaskId(), &m_pWorkingTasks );
    if( pCompleted != NULL )
    {
        working_task::add( pCompleted, &m_pCompletedTasks );
    }
}

//==============================================================================

tmipc::Task* Service::OnNewTask( tmipc::Packet* pPacket )
{
    COREDUMP_LOG( "Service::OnNewTask:  %d", pPacket->GetTaskType() );
    tmipc::Task* pRet = NULL;

    switch( pPacket->GetTaskType() )
    {
        case tmipc::TaskType_CoredumpStartTask:
            {
                void* pMem = s_Allocate( sizeof( CoredumpTask ) );
                pRet = new (pMem) CoredumpTask( this );
            }
            break;

        default:
            DEJA_ERROR( "tma::coredump::Service::OnNewTask", "Unrecognized task type 0x%08x", pPacket->GetTaskType() );
            ASSERT( 0 );
            break;
    }

    if( pRet )
    {
        pRet->SetServicesManager( m_pServicesManager );
        pRet->SetServiceId( m_ServiceId );
        pRet->SetTaskId( pPacket->GetTaskId() );
        pRet->SetTaskType( (s16)pPacket->GetTaskType() );
        StartedTask( pRet );
        pRet->OnInitiate( pPacket );
    }

    return( pRet );
}

//==============================================================================
}} // namespace
//==============================================================================
