﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/tz/timesrv_TimeZoneDetail.h>

#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_ResultPrivate.h>
#include <nn/util/util_StringUtil.h>
#include <nn/result/result_HandlingUtility.h>

#include <siglo/Include/time_TzLocaltime.h> // tz database
#include <siglo/Include/time_TzApiForTimeZoneBinary.h>

namespace nn { namespace timesrv { namespace detail { namespace tz {

using ::nn::time::PosixTime;
using ::nn::time::CalendarTime;
using ::nn::timesrv::TimeZoneRuleInner;

namespace
{
    void ConvertStructTm(nne::tz::tm* pOut, const nn::time::CalendarTime in, DaylightSavingTimeType daylightSavingTimeType) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        pOut->tm_year = in.year - 1900;
        pOut->tm_mon = in.month - 1;
        pOut->tm_mday = in.day;

        pOut->tm_hour = in.hour;
        pOut->tm_min = in.minute;
        pOut->tm_sec = in.second;

        pOut->tm_isdst = static_cast<int32_t>( daylightSavingTimeType );
    }

    void ConvertCalendarTime(nn::time::CalendarTime* pOut, const nne::tz::tm in) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        pOut->year = static_cast<uint16_t>(in.tm_year + 1900);
        pOut->month = static_cast<uint8_t>(in.tm_mon + 1);
        pOut->day = static_cast<uint8_t>(in.tm_mday);

        pOut->hour = static_cast<uint8_t>(in.tm_hour);
        pOut->minute = static_cast<uint8_t>(in.tm_min);
        pOut->second = static_cast<uint8_t>(in.tm_sec);
    }

    void ConvertCalendarAdditionalInfo(nn::time::sf::CalendarAdditionalInfo* pOut, const nne::tz::tm in) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOut);

        pOut->dayOfWeek = static_cast<int8_t>( in.tm_wday );
        pOut->yearDay = in.tm_yday;

        pOut->timeZone.utcOffsetSeconds = in.tm_gmtoffset_second;

        nn::util::Strlcpy(pOut->timeZone.standardTimeName, in.tm_zonename, nn::time::TimeZone::StandardTimeNameSize);

        NN_SDK_ASSERT(in.tm_isdst >= 0);
        pOut->timeZone.isDaylightSavingTime = in.tm_isdst == 0 ? false : true;
    }


    // nne::tz::tm の日時部分だけの一致性チェック
    bool EqualCalendar(const nne::tz::tm& lhs, const nne::tz::tm& rhs)
    {
        return
            lhs.tm_sec  == rhs.tm_sec &&
            lhs.tm_min  == rhs.tm_min &&
            lhs.tm_hour == rhs.tm_hour &&
            lhs.tm_mday == rhs.tm_mday &&
            lhs.tm_mon  == rhs.tm_mon &&
            lhs.tm_year == rhs.tm_year;
    }

    nn::Result ValidateTimeZoneRule(const TimeZoneRuleInner& rule) NN_NOEXCEPT
    {
        // 範囲チェック
        NN_RESULT_THROW_UNLESS(0 < rule.typecnt && rule.typecnt <= static_cast<int32_t>(TypesCountMax), nn::time::ResultArgumentOutOfRange());
        NN_RESULT_THROW_UNLESS(0 < rule.timecnt && rule.timecnt <= static_cast<int32_t>(TimesCountMax), nn::time::ResultArgumentOutOfRange());
        NN_RESULT_THROW_UNLESS(0 < rule.charcnt && rule.charcnt <= static_cast<int32_t>(CharsCountMax), nn::time::ResultArgumentOutOfRange());
        for (int i = 0; i < rule.timecnt; ++i)
        {
            NN_RESULT_THROW_UNLESS(static_cast<int32_t>(rule.types[i]) < rule.typecnt, nn::time::ResultArgumentOutOfRange());
        }
        for (int i = 0; i < rule.typecnt; ++i)
        {
            NN_RESULT_THROW_UNLESS(0 <= rule.ttis[i].tt_abbrind && rule.ttis[i].tt_abbrind < static_cast<int32_t>(CharsCountMax), nn::time::ResultArgumentOutOfRange());
        }
        NN_RESULT_SUCCESS;
    }
}

nn::Result ParseTimeZoneBinary(TimeZoneRuleInner* pOut, const char *pBinary, size_t size) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOut);
    NN_ABORT_UNLESS_NOT_NULL(pBinary);

    // バイナリパースして TimeZoneRule に入れる
    NN_RESULT_THROW_UNLESS(
        nne::tz::ParseTimeZoneBinary(pOut, pBinary, size) == nne::tz::ResultType::Success,
        nn::time::ResultInvalidTimeZoneBinary());

    NN_RESULT_SUCCESS;
}

nn::Result ToCalendarTime(
    CalendarTime* pOutCalendar,
    nn::time::sf::CalendarAdditionalInfo* pOutCalendarAdditionalInfo,
    const PosixTime& posixTime,
    const TimeZoneRuleInner& rule) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutCalendar, nn::time::ResultInvalidPointer());

    NN_RESULT_DO(ValidateTimeZoneRule(rule));

    nne::tz::time_t timeval = posixTime.value;
    nne::tz::tm calendar;

    auto result = nne::tz::Localtime(&calendar, &rule, &timeval);

    // nne::tz::ResultType::Overflow 以外は発生しない想定
    NN_SDK_ASSERT(result == nne::tz::ResultType::Success || result == nne::tz::ResultType::Overflow);
    NN_RESULT_THROW_UNLESS(result == nne::tz::ResultType::Success, nn::time::ResultOverflowed());

    // pOutCalendar へ格納
    ConvertCalendarTime(pOutCalendar, calendar);

    if(pOutCalendarAdditionalInfo)
    {
       // pOutCalendarAdditionalInfo へ格納
       ConvertCalendarAdditionalInfo(pOutCalendarAdditionalInfo, calendar);
    }

    NN_RESULT_SUCCESS;
}

namespace
{
    // nne::tz::Mktime をラップして nn::Result を返す
    nn::Result MktimeFunc(  PosixTime* pOut,
                            int32_t* pOutTimeTypeIndex,
                            const TimeZoneRuleInner& rule,
                            const CalendarTime& inCalendar,
                            DaylightSavingTimeType daylightSavingTimeType)
    {
        nne::tz::time_t time;
        nne::tz::tm outputTm;
        ConvertStructTm(&outputTm, inCalendar, daylightSavingTimeType);

        const nne::tz::tm inputTm = outputTm;

        auto result = nne::tz::Mktime(&time, &rule, &outputTm);

        if(result == nne::tz::ResultType::Overflow)
        {
            NN_RESULT_THROW(nn::time::ResultOverflowed());
        }
        else if(result == nne::tz::ResultType::NotFound)
        {
            NN_RESULT_THROW(nn::time::ResultNotFound());
        }
        else if(!EqualCalendar(inputTm, outputTm))
        {
            // tm が補正されていたら存在しない日付を入力している
            NN_RESULT_THROW(nn::time::ResultNotFound());
        }

        // 上記以外のエラーは発生しない想定
        NN_ABORT_UNLESS_EQUAL(result, nne::tz::ResultType::Success);

        pOut->value = time;
        *pOutTimeTypeIndex = outputTm.tm_use_type_index;
        NN_RESULT_SUCCESS;
    };

    /**
     * @brief   rule.types[] の index を指定して、 rule.ttis にある時差を取得する
     * @param[in] rule
     * @param[in] timeTypeIndex
     */
    int64_t GetUtcOffset(const TimeZoneRuleInner& rule, int timeTypeIndex)
    {
        NN_SDK_ASSERT_NOT_EQUAL(0, rule.timecnt);
        NN_SDK_ASSERT_RANGE(timeTypeIndex, 0, rule.timecnt);

        int index = rule.types[timeTypeIndex];
        return static_cast<int64_t>( rule.ttis[index].tt_gmtoff );
    }

    /**
     * @brief   絶対時刻に対して採用される rule.types[] の index を返す
     * @param[in] rule
     * @param[in] posix
     */
    int SearchTimeTypeIndex(const TimeZoneRuleInner& rule, const PosixTime posix)
    {
        const nne::tz::time_t t = posix.value;
        const TimeZoneRuleInner* sp = &rule;

        if (sp->timecnt == 0 || t < sp->ats[0])
        {
            return 0;
        }
        else
        {
            int lo = 1;
            int hi = sp->timecnt;

            while (lo < hi) {
                int mid = (lo + hi) >> 1;

                if (t < sp->ats[mid])
                {
                    hi = mid;
                }
                else
                {
                    lo = mid + 1;
                }
            }
            return lo - 1;
        }
    }
}

nn::Result ToPosixTime(
    int* pOutCount,
    PosixTime* pOutPosixTimeList,
    size_t count,
    const CalendarTime& calendarTime,
    const TimeZoneRuleInner& rule,
    DaylightSavingTimeType daylightSavingTimeType) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutCount, nn::time::ResultInvalidPointer());
    NN_RESULT_THROW_UNLESS(pOutPosixTimeList, nn::time::ResultInvalidPointer());

    NN_RESULT_DO(ValidateTimeZoneRule(rule));

    int outCount = 0;

    PosixTime posix;
    int32_t timeTypeIndex;
    NN_RESULT_DO(MktimeFunc(&posix, &timeTypeIndex, rule, calendarTime, daylightSavingTimeType));

    // 1つ目の答えを保存
    outCount = 1;
    pOutPosixTimeList[0] = posix;

    if(count > 1)
    {
        // 2つ目の答えがあるか探す.
        // 1つめの答えで使用した timeTypeIndex の前後の時差切り替わりで、同じカレンダーが存在するかチェック.


        /**
         * @brief timeTypeIndex のオフセットを指定して、time type info 違いのもう1つ PosixTime を取得する
         * @param[out]  pOut            もう1つの答え
         * @param[in]   rule            タイムゾーンルール
         * @param[in]   nowPosix        1つ目の答え
         * @param[in]   timeTypeIndex   1つ目の答えを求めるのに使用した time type info の index
         * @param[in]   offsetIndex     チェックする time type info の index のオフセット
         *
         * @return  もう1つの答えが見つかったどうか
         * @retval  true    見つかった
         * @retval  false   見つからなかった
         */
        auto FindOtherPosixTime = [](
            PosixTime* pOut, const TimeZoneRuleInner& rule, const PosixTime nowPosix,
            int32_t timeTypeIndex , int32_t offsetIndex) -> bool
        {
            // 最初に求まった答えに使われた timeTypeIndex の時差と、前後の TimeType の時差を利用して PosixTime を計算。
            // その PosixTime が tiemTypeIndex に属していれば、狭義タイムゾーン違いのもう1つの答え.

            NN_SDK_ASSERT_NOT_NULL(pOut);

            int64_t nowTimeDifference = GetUtcOffset(rule, timeTypeIndex); // 1つ目の答えの、狭義タイムゾーンでの時差
            int64_t otherTimeDifference = GetUtcOffset(rule, timeTypeIndex + offsetIndex); // 他の time type の時差

            nn::time::PosixTime candidatePosix =
                {nowPosix.value + nowTimeDifference - otherTimeDifference};

            // 候補の candidatePosix が本当に該当の TimeTypeIndex として存在するかどうか
            if( SearchTimeTypeIndex(rule, candidatePosix) == (timeTypeIndex + offsetIndex) )
            {
                *pOut = candidatePosix;
                return true;
            }
            return false;
        };

        if(timeTypeIndex > 0) // 1つ前の時差切り替わりを探す
        {
            PosixTime other;
            if(FindOtherPosixTime(&other, rule, posix, timeTypeIndex, -1))
            {
                outCount = 2;
                pOutPosixTimeList[1] = other;
            }
        }
        if(outCount == 1 && (timeTypeIndex + 1) < rule.timecnt) // 1つ次の時差切り替わりを探す
        {
            PosixTime other;
            if(FindOtherPosixTime(&other, rule, posix, timeTypeIndex, +1))
            {
                outCount = 2;
                pOutPosixTimeList[1] = other;
            }
        }
    }

    // ここまでくれば成功

    NN_SDK_ASSERT_GREATER_EQUAL(count, static_cast<size_t>(outCount));
    NN_SDK_ASSERT_MINMAX(outCount, 0, 2);

    *pOutCount = outCount;
    NN_RESULT_SUCCESS;
}

// いったん作らない
nn::Result AdjustCalendarTime(
    int* pOutCount,
    CalendarTime* pOutCalendarTimeList,
    int count,
    const CalendarTime& calendarTime,
    const TimeZoneRuleInner& rule) NN_NOEXCEPT
{
    // MEMO:ToPosixTime とほぼ同様に、mktime で補正される strcut tm を pOutCorrectCalendarTimeList に入れる

    NN_ABORT_UNLESS_NOT_NULL(pOutCount);
    NN_ABORT_UNLESS_NOT_NULL(pOutCalendarTimeList);
    NN_UNUSED(count);
    NN_UNUSED(calendarTime);
    NN_UNUSED(rule);

    NN_RESULT_THROW(nn::time::ResultNotImplemented());
}

}}}} // nn::timesrv::detail::tz
