﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/timesrv/detail/service/timesrv_StaticServiceServer.h>
#include <nn/time/detail/time_ClockSnapshotInitialType.h>
#include <nn/time/detail/util/time_UtilApi.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/timesrv/detail/service/timesrv_ServiceProvider.h>
#include <nn/timesrv/detail/service/timesrv_SystemClockServer.h>
#include <nn/timesrv/detail/service/timesrv_SteadyClockServer.h>
#include <nn/timesrv/detail/service/timesrv_TimeZoneServiceServer.h>

#include <nn/timesrv/detail/core/timesrv_EphemeralNetworkSystemClockCore.h>
#include <nn/timesrv/detail/core/timesrv_StandardLocalSystemClockCore.h>
#include <nn/timesrv/detail/core/timesrv_StandardUserSystemClockCore.h>
#include <nn/timesrv/detail/core/timesrv_StandardNetworkSystemClockCore.h>
#include <nn/timesrv/detail/core/timesrv_StandardSteadyClockCore.h>

#include <nn/timesrv/detail/settings/timesrv_ClockSettings.h>

namespace nn
{
namespace timesrv
{
namespace detail
{
namespace service
{

namespace
{
    bool IsStandardNetworkSystemClockAvailable(const nn::time::sf::ClockSnapshot& value) NN_NOEXCEPT
    {
        return value.netSystemClockContext.timeStamp.sourceId == value.steadyClockTimePoint.sourceId;
    }
}

StaticServiceServer::StaticServiceServer(
    const Capabilities& capabilities,
    ServiceProvider* pServiceProvider,
    nn::sf::ExpHeapAllocator* pExpHeapAllocator
) NN_NOEXCEPT
    : m_Capabilities(capabilities),
      m_pExpHeapAllocator(pExpHeapAllocator),
      m_pStandardLocalSystemClockCore(&pServiceProvider->m_StandardLocalSystemClockCore),
      m_pStandardUserSystemClockCore(&pServiceProvider->m_StandardUserSystemClockCore),
      m_pStandardNetworkSystemClockCore(&pServiceProvider->m_StandardNetworkSystemClockCore),
      m_pStandardSteadyClockCore(&pServiceProvider->m_StandardSteadyClockCore),
      m_pTimeZoneServiceCore(&pServiceProvider->m_TimeZoneServiceCore),
      m_pEphemeralNetworkSystemClockCore(&pServiceProvider->m_EphemeralNetworkSystemClockCore),
      m_pSharedMemoryManager(&pServiceProvider->m_SharedMemoryManager)
{
}

StaticServiceServer::~StaticServiceServer() NN_NOEXCEPT
{
}

nn::Result StaticServiceServer::GetStandardUserSystemClock(nn::sf::Out<nn::sf::SharedPointer<ISystemClock>> outStandardUserSystemClock) NN_NOEXCEPT
{
    auto pIStandardUserSystemClock =
        nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<ISystemClock, SystemClockServer>(
            m_pExpHeapAllocator, m_pStandardUserSystemClockCore, m_Capabilities.isUserClockAdministrator
        );

    NN_RESULT_THROW_UNLESS(pIStandardUserSystemClock != nullptr, nn::time::ResultOutOfMemory());

    outStandardUserSystemClock.Set(pIStandardUserSystemClock);
    NN_RESULT_SUCCESS;
}

nn::Result StaticServiceServer::GetStandardNetworkSystemClock(nn::sf::Out<nn::sf::SharedPointer<ISystemClock>> outStandardNetworkSystemClock) NN_NOEXCEPT
{
    auto pIStandardNetworkSystemClock =
        nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<ISystemClock, SystemClockServer>(
            m_pExpHeapAllocator, m_pStandardNetworkSystemClockCore, m_Capabilities.isNetworkClockAdministrator
        );

    NN_RESULT_THROW_UNLESS(pIStandardNetworkSystemClock != nullptr, nn::time::ResultOutOfMemory());

    outStandardNetworkSystemClock.Set(pIStandardNetworkSystemClock);
    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::GetEphemeralNetworkSystemClock(
    nn::sf::Out<nn::sf::SharedPointer<ISystemClock>> outEphemeralNetworkSystemClock) NN_NOEXCEPT
{
    auto pIStandardNetworkSystemClock =
        nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<ISystemClock, SystemClockServer>(
            m_pExpHeapAllocator, m_pEphemeralNetworkSystemClockCore, m_Capabilities.isNetworkClockAdministrator
        );

    NN_RESULT_THROW_UNLESS(pIStandardNetworkSystemClock != nullptr, nn::time::ResultOutOfMemory());

    outEphemeralNetworkSystemClock.Set(pIStandardNetworkSystemClock);
    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::GetSharedMemoryNativeHandle(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    outHandle.Set(nn::sf::NativeHandle(m_pSharedMemoryManager->GetHandle(), false));
    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::GetStandardLocalSystemClock(
    nn::sf::Out<nn::sf::SharedPointer<ISystemClock>> outStandardLocalSystemClock) NN_NOEXCEPT
{
    auto pIStandardLocalSystemClock =
        nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<ISystemClock, SystemClockServer>(
            m_pExpHeapAllocator, m_pStandardLocalSystemClockCore, m_Capabilities.isLocalClockAdministrator
        );

    NN_RESULT_THROW_UNLESS(pIStandardLocalSystemClock != nullptr, nn::time::ResultOutOfMemory());

    outStandardLocalSystemClock.Set(pIStandardLocalSystemClock);
    NN_RESULT_SUCCESS;
}

nn::Result StaticServiceServer::GetStandardSteadyClock(nn::sf::Out<nn::sf::SharedPointer<ISteadyClock>> outStandardSteadyClock) NN_NOEXCEPT
{
    auto pIStandardSteadyClock =
        nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<ISteadyClock, SteadyClockServer>(
            m_pExpHeapAllocator, m_pStandardSteadyClockCore, m_Capabilities.isSteadyClockAdministrator
        );

    NN_RESULT_THROW_UNLESS(pIStandardSteadyClock != nullptr, nn::time::ResultOutOfMemory());

    outStandardSteadyClock.Set(pIStandardSteadyClock);
    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::GetTimeZoneService(
    nn::sf::Out<nn::sf::SharedPointer<ITimeZoneService>> outTimeZoneService) NN_NOEXCEPT
{
    auto pITimeZoneService =
        nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<ITimeZoneService, TimeZoneServiceServer>(
            m_pExpHeapAllocator, m_pStandardSteadyClockCore, m_pTimeZoneServiceCore, m_Capabilities.isTimeZoneServiceAdministrator
        );

    NN_RESULT_THROW_UNLESS(pITimeZoneService != nullptr, nn::time::ResultOutOfMemory());

    outTimeZoneService.Set(pITimeZoneService);
    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::SetStandardSteadyClockInternalOffset(nn::TimeSpanType internalOffset) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Capabilities.isSteadyClockAdministrator, nn::time::ResultNoCapability());

    settings::WriteSteadyClockInternalOffset(internalOffset);

    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::IsStandardUserSystemClockAutomaticCorrectionEnabled(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    // 自動補正フラグは m_pStandardUserSystemClockCore に丸投げ
    outValue.Set(m_pStandardUserSystemClockCore->IsAutomaticCorrectionEnabled());

    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::SetStandardUserSystemClockAutomaticCorrectionEnabled(bool value) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_Capabilities.isUserClockAdministrator, nn::time::ResultNoCapability());

    // 自動補正フラグは m_pStandardUserSystemClockCore に丸投げ
    NN_RESULT_DO(m_pStandardUserSystemClockCore->SetAutomaticCorrectionEnabled(value));

    m_pSharedMemoryManager->SetStandardUserSystemClockAutomaticCorrectionEnabled(value);

    // フラグ変更時間を更新
    nn::time::SteadyClockTimePoint currentTimePoint;
    NN_RESULT_DO(m_pStandardSteadyClockCore->GetCurrentTimePoint(&currentTimePoint));
    m_pStandardUserSystemClockCore->SetAutomaticCorrectionUpdatedTime(currentTimePoint);

    // 永続化
    settings::WriteUserSystemClockAutomaticCorrectionEnabled(value);
    settings::WriteUserSystemClockAutomaticCorrectionUpdatedTime(currentTimePoint);

    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::IsStandardNetworkSystemClockAccuracySufficient(nn::sf::Out<bool> outValue) NN_NOEXCEPT
{
    outValue.Set(m_pStandardNetworkSystemClockCore->IsStandardNetworkSystemClockAccuracySufficient());
    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::GetStandardUserSystemClockAutomaticCorrectionUpdatedTime(
    nn::sf::Out<nn::time::SteadyClockTimePoint> outValue) NN_NOEXCEPT
{
    outValue.Set(m_pStandardUserSystemClockCore->GetAutomaticCorrectionUpdatedTime());
    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::CalculateMonotonicSystemClockBaseTimePoint(
    nn::sf::Out<int64_t> outValue,
    const nn::time::SystemClockContext& systemClockContext) NN_NOEXCEPT
{
    nn::time::SteadyClockTimePoint steadyClockTimePoint;
    NN_RESULT_DO(m_pStandardSteadyClockCore->GetCurrentTimePoint(&steadyClockTimePoint));

    NN_RESULT_THROW_UNLESS(steadyClockTimePoint.sourceId == systemClockContext.timeStamp.sourceId, nn::time::ResultOffsetInvalid());

    int64_t baseTime =
            steadyClockTimePoint.value + systemClockContext.offset - nn::os::GetSystemTick().ToTimeSpan().GetSeconds();

    outValue.Set(baseTime);
    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::GetClockSnapshot(nn::sf::Out<nn::time::sf::ClockSnapshot> outValue, uint8_t initialType) NN_NOEXCEPT
{
    nn::time::SystemClockContext userSystemClockContext, netSystemClockContext;
    NN_RESULT_DO(m_pStandardUserSystemClockCore->GetSystemClockContext(&userSystemClockContext));
    NN_RESULT_DO(m_pStandardNetworkSystemClockCore->GetSystemClockContext(&netSystemClockContext));

    NN_RESULT_DO(GetClockSnapshotFromSystemClockContext(outValue, userSystemClockContext, netSystemClockContext, initialType));

    NN_RESULT_SUCCESS;
}

// version=0
Result StaticServiceServer::GetClockSnapshotFromSystemClockContext(
    nn::sf::Out<nn::time::sf::ClockSnapshot> outValue,
    const nn::time::SystemClockContext& userSystemClockContext,
    const nn::time::SystemClockContext& netSystemClockContext,
    uint8_t initialType) NN_NOEXCEPT
{
    const uint8_t Version = 0;

    outValue->userSystemClockContext = userSystemClockContext;
    outValue->netSystemClockContext = netSystemClockContext;

    NN_RESULT_DO(m_pStandardSteadyClockCore->GetCurrentTimePoint(&outValue->steadyClockTimePoint));

    outValue->isAutomaticCorrectionEnabled = m_pStandardUserSystemClockCore->IsAutomaticCorrectionEnabled();
    m_pTimeZoneServiceCore->GetDeviceLocationName(&outValue->locationName);

    // SteadyClockTimePoint は常に進むので、一度取得した値から PosixTime を計算する
    NN_RESULT_DO(core::SystemClockCore::CalculatePosixTime(&outValue->userSystemClockPosixTime, outValue->steadyClockTimePoint, outValue->userSystemClockContext));
    NN_RESULT_DO(m_pTimeZoneServiceCore->ToCalendarTimeWithMyRule(&outValue->userCalendarTime, &outValue->userCalendarAdditionalInfo, outValue->userSystemClockPosixTime));

    auto result = core::SystemClockCore::CalculatePosixTime(&outValue->netSystemClockPosixTime, outValue->steadyClockTimePoint, outValue->netSystemClockContext);
    if (result.IsFailure()) // ネットワーク時計未補正
    {
        outValue->netSystemClockPosixTime.value = 0;
    }
    NN_RESULT_DO(m_pTimeZoneServiceCore->ToCalendarTimeWithMyRule(&outValue->netCalendarTime, &outValue->netCalendarAdditionalInfo, outValue->netSystemClockPosixTime));

    outValue->initialType = initialType;
    outValue->version = Version;
    outValue->padding[0] = 0;

    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::CalculateStandardUserSystemClockDifferenceByUser(
    nn::sf::Out<nn::TimeSpanType> outValue,
    const nn::time::sf::ClockSnapshot& from,
    const nn::time::sf::ClockSnapshot& to) NN_NOEXCEPT
{
    if(from.userSystemClockContext == to.userSystemClockContext)
    {
        // まったく操作されていない
        outValue.Set(nn::TimeSpan(0));
        NN_RESULT_SUCCESS;
    }

    // sourceId 違いはユーザーによる意図的な操作とはしない仕様
    if(from.userSystemClockContext.timeStamp.sourceId != to.userSystemClockContext.timeStamp.sourceId)
    {
        // 以下のいずれか
        // - 単調増加クロックの電池切れによってユーザー時計の時刻が初期値に戻った
        // - セーブデータが別デバイスへ移行された
        outValue.Set(nn::TimeSpan(0));
        NN_RESULT_SUCCESS;
    }

    if(from.isAutomaticCorrectionEnabled && to.isAutomaticCorrectionEnabled
        && (IsStandardNetworkSystemClockAvailable(from) || IsStandardNetworkSystemClockAvailable(to)))
    {
        // 2つとも自動補正ONで、どちらか一方でもネットワーク時計が補正済であれば、ユーザー時計の操作はユーザーによるものとは決めつけられず 0 を返す
        // IsStandardNetworkSystemClockAvailable() の判定を && にすると、自動補正ONがずっとONで、
        // from と to の間に初めてネットワーク時計の補正が行われた際にユーザーが操作したことになってしまうのでダメ。
        outValue.Set(nn::TimeSpan(0));
        NN_RESULT_SUCCESS;
    }

    // from から to の間に操作された量を計算
    auto operationAmountSeconds =
        to.userSystemClockContext.offset - from.userSystemClockContext.offset; // SystemClockContext.offset は秒単位

    outValue.Set(nn::TimeSpan::FromSeconds(operationAmountSeconds));
    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::CalculateSpanBetween(
    nn::sf::Out<nn::TimeSpanType> outValue,
    const nn::time::sf::ClockSnapshot& from,
    const nn::time::sf::ClockSnapshot& to) NN_NOEXCEPT
{
    // 単調増加で計算結果の安定する steadyClockTimePoint を使ってまずは計算を試みる
    int64_t seconds;
    auto result = nn::time::detail::util::GetSpanBetween(&seconds, from.steadyClockTimePoint, to.steadyClockTimePoint);
    if (result.IsSuccess())
    {
        outValue.Set(nn::TimeSpan::FromSeconds(seconds));
        NN_RESULT_SUCCESS;
    }

    // nn::time::detail::util::GetSpanBetween 失敗時、ネットワーク時計を使って計算を試みる
    // from, to の両者がネットワーク時計を保持してないと計算不可
    NN_RESULT_THROW_UNLESS(
        from.netSystemClockPosixTime.value != 0 && to.netSystemClockPosixTime.value != 0,
        nn::time::ResultNotComparable());

    auto span = to.netSystemClockPosixTime - from.netSystemClockPosixTime;
    outValue.Set(span);
    NN_RESULT_SUCCESS;
}

Result StaticServiceServer::GetStandardUserSystemClockInitialYear(nn::sf::Out<int32_t> outValue) NN_NOEXCEPT
{
    int value = settings::ReadStandardUserSystemClockInitialYear();
    outValue.Set(static_cast<int32_t>(value));
    NN_RESULT_SUCCESS;
}

}
}
}
}
