﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/time_StandardUserSystemClock.h>

#include <nn/time/detail/time_CommonDetail.h>
#include <nn/time/detail/time_SharedMemoryClient.h>
#include <nn/timesrv/detail/service/timesrv_ISystemClock.sfdl.h>

#include <nn/util/util_Optional.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn
{
namespace time
{

extern nn::util::optional<nn::time::detail::SharedMemoryClient> g_SharedMemoryClient;

Result StandardUserSystemClock::GetCurrentTime(PosixTime* pOutPosixTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPosixTime);
    NN_RESULT_THROW_UNLESS(g_SharedMemoryClient == true, ResultNotInitialized());

    g_SharedMemoryClient->GetStandardUserSystemClockCurrentTime(pOutPosixTime);

    NN_RESULT_SUCCESS;
}

Result StandardUserSystemClock::GetSystemClockContext(SystemClockContext* pOutSystemClockContext) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSystemClockContext);
    NN_RESULT_THROW_UNLESS(g_SharedMemoryClient == true, ResultNotInitialized());

    g_SharedMemoryClient->GetStandardUserSystemClockContext(pOutSystemClockContext);

    NN_RESULT_SUCCESS;
}

StandardUserSystemClock::time_point StandardUserSystemClock::now() NN_NOEXCEPT
{
    PosixTime posixTime = {0};

    Result result = StandardUserSystemClock::GetCurrentTime(&posixTime);
    if(result.IsFailure())
    {
        // ResultNotInitialized だけは返る可能性があるのでそのときは 0 を返して濁す
        posixTime.value = 0;
    }

    return time_point(duration(posixTime.value));
}

std::time_t StandardUserSystemClock::to_time_t(const StandardUserSystemClock::time_point& t) NN_NOEXCEPT
{
    // time_point::time_since_epoch() については、
    // NX向けビルド(clang)、Windowsビルド(VC++)にて time_point 生成時にセットする duration を
    // そのまま返す実装になっている.
    // time_since_epoch() で内部のduration に何らかの計算をした結果が返ってくる場合には、
    // 実装分岐する必要があり、その場合は from_time_t でもその逆の変換を行う必要が出てくる.

    // TODO: time_since_epoch() の内部仕様が想定外であったときの検知(SIGLO-19154)
    // TODO: std::time_t が 32bit でも十分な範囲の時間が扱えるように(SIGLO-19156)

    return static_cast<std::time_t>( std::chrono::duration_cast<std::chrono::seconds>( t.time_since_epoch() ).count() );
}

StandardUserSystemClock::time_point StandardUserSystemClock::from_time_t(std::time_t t) NN_NOEXCEPT
{
    return time_point(duration(t));
}

}
}

