﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/time_ApiForClockSnapshot.h>
#include <nn/time/detail/time_ClockSnapshotPrivateApi.h>
#include <nn/timesrv/detail/service/timesrv_IStaticService.sfdl.h>
#include <nn/time/detail/time_ClaimPeriodicBenefit.h>

namespace nn { namespace time {

extern nn::sf::SharedPointer<nn::timesrv::detail::service::IStaticService> g_pStaticService;

#if !defined( NN_BUILD_CONFIG_COMPILER_GCC ) // gcc に std::is_trivially_copyable がないので回避
NN_STATIC_ASSERT(std::is_trivially_copyable<nn::time::PeriodicBenefitClaimContext>::value);
#endif
NN_STATIC_ASSERT(std::is_standard_layout<nn::time::PeriodicBenefitClaimContext>::value);
NN_STATIC_ASSERT(sizeof(PeriodicBenefitClaimContext) == 768);

bool PeriodicBenefitClaimContext::GetLastReceivedStandardUserSystemClockCalendarTime(CalendarTime* pOut) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    if(!detail::HasLastBenefitReceivedInfo(*this))
    {
        return false;
    }

    *pOut = this->_lastBenefitReceivedSnapshot.GetStandardUserSystemClockCalendarTime();
    return true;
}

bool PeriodicBenefitClaimContext::GetEndOfPenaltyAtLastClaimForDebug(SteadyClockTimePoint* pOut) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    if(!detail::HasPenaltyInfo(*this))
    {
        return false;
    }

    *pOut = this->_penaltyEndSteadyClockTimePoint;
    return true;
}

bool operator == (const PeriodicBenefitClaimContext& lhs, const PeriodicBenefitClaimContext& rhs) NN_NOEXCEPT
{
    return true
        && lhs._snapshot                        == rhs._snapshot
        && lhs._lastBenefitReceivedSnapshot     == rhs._lastBenefitReceivedSnapshot
        && lhs._penaltyEndSteadyClockTimePoint  == rhs._penaltyEndSteadyClockTimePoint
        && lhs._isMaxPenalatyOccurred           == rhs._isMaxPenalatyOccurred
    ;
}

bool operator != (const PeriodicBenefitClaimContext& lhs, const PeriodicBenefitClaimContext& rhs) NN_NOEXCEPT
{
    return !(lhs == rhs);
}

PeriodicBenefitClaimResult  ClaimPeriodicBenefitWithUserSystemClock(
    PeriodicBenefitClaimContext* pHandover,
    const ClockSnapshot& currentSnapshot,
    const nn::TimeSpan& maxPenaltyTimeSpan,
    const nn::TimeSpan& acceptableOperationTimeSpan,
    const CheckCalendarTimeCallback* pCheckCalendarTimeCallback) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pStaticService != nullptr, "[TIME] The time library is not initialized.");

    NN_SDK_REQUIRES_NOT_NULL(pHandover);

    auto pCurrent = nn::time::detail::GetSfClockSnapshotPtr(&currentSnapshot);
    NN_UNUSED(pCurrent);

    NN_SDK_REQUIRES(
        pCurrent->initialType != static_cast<uint8_t>(nn::time::detail::ClockSnapshotInitialType::Constructor),
        "[TIME] The current ClockSnapshot is not initialized.");
    NN_SDK_REQUIRES_GREATER(maxPenaltyTimeSpan, nn::TimeSpan(0));
    NN_SDK_REQUIRES_GREATER_EQUAL(acceptableOperationTimeSpan, nn::TimeSpan(0));
    NN_SDK_REQUIRES_NOT_NULL(pCheckCalendarTimeCallback);

    return detail::ClaimPeriodicBenefitWithUserSystemClock(
                pHandover, currentSnapshot,
                maxPenaltyTimeSpan, acceptableOperationTimeSpan,
                pCheckCalendarTimeCallback
                );
}

}}

