﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/time/time_AdjustableUserSystemClock.h>

#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/detail/time_CommonDetail.h>
#include <nn/time/detail/time_MonotonicSystemClock.h>

namespace nn
{
namespace time
{

namespace
{
    detail::MonotonicSystemClock g_MonotonicSystemClock;
}

Result AdjustableUserSystemClock::GetCurrentTime(PosixTime* pOutPosixTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPosixTime);

    return g_MonotonicSystemClock.GetCurrentTime(pOutPosixTime);
}

void AdjustableUserSystemClock::Adjust() NN_NOEXCEPT
{
    SystemClockContext context;
    NN_ABORT_UNLESS_RESULT_SUCCESS(StandardUserSystemClock::GetSystemClockContext(&context));
    g_MonotonicSystemClock.SetSystemClockContext(context);
}

void AdjustableUserSystemClock::GetSystemClockContext(nn::time::SystemClockContext* pOutSystemClockContext) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSystemClockContext);

    *pOutSystemClockContext = g_MonotonicSystemClock.GetSystemClockContext();
}

AdjustableUserSystemClock::time_point AdjustableUserSystemClock::now() NN_NOEXCEPT
{
    PosixTime posixTime = {0};

    Result result = GetCurrentTime(&posixTime);
    if(result.IsFailure())
    {
        posixTime.value = 0;
    }

    return time_point(duration(posixTime.value));
}

std::time_t AdjustableUserSystemClock::to_time_t(const AdjustableUserSystemClock::time_point& t) NN_NOEXCEPT
{
    // time_point::time_since_epoch() については、
    // NX向けビルド(clang)、Windowsビルド(VC++)にて time_point 生成時にセットする duration を
    // そのまま返す実装になっている.
    // time_since_epoch() で内部のduration に何らかの計算をした結果が返ってくる場合には、
    // 実装分岐する必要があり、その場合は from_time_t でもその逆の変換を行う必要が出てくる.
    return static_cast<std::time_t>( std::chrono::duration_cast<std::chrono::seconds>( t.time_since_epoch() ).count() );
}

AdjustableUserSystemClock::time_point AdjustableUserSystemClock::from_time_t(std::time_t t) NN_NOEXCEPT
{
    return time_point(duration(t));
}

}
}

