﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>

#include <nn/erpt.h>
#include <nn/err.h>

#include <nn/err/detail/err_ErrorCodeConvert.h>
#include <nn/hid/system/hid_InputDetection.h>
#include <nn/tc/tc_TemperatureApi.h>
#include <nn/tcap/detail/tcap_Log.h>
#include <nn/tcap/tcap_OperationModeTypes.h>
#include <nn/tcap/tcap_Result.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include "tcap_ContinuousHighSkinTemperatureHandler.h"

#define NN_DETAIL_TCAP_WARN_UNLESS_RESULT_SUCCESS(expression) \
    do \
    { \
        auto result_ = (expression); \
        if ( result_.IsFailure() ) \
        { \
            NN_DETAIL_TCAP_WARN("Failed: %s\n  Module: %d\n  Description: %d\n  InnerValue: 0x%08x\n", \
                NN_MACRO_STRINGIZE(expression), result_.GetModule(), result_.GetDescription(), result_.GetInnerValueForDebug()); \
        } \
    } while ( NN_STATIC_CONDITION(false) )

namespace nn { namespace tcap { namespace server {

bool ContinuousHighSkinTemperatureHandler::DetectValidInput() NN_NOEXCEPT
{
    nn::hid::system::InputSourceState inputSourceState;
    nn::hid::system::GetInputSourceState(&inputSourceState,
        nn::hid::system::InputSourceIdSet().Set()
        ^ nn::hid::system::InputSourceId::Sensor::Mask
        ^ nn::hid::system::InputSourceId::TouchScreen::Mask
        ^ nn::hid::system::InputSourceId::AnalogStick::Mask
        ^ nn::hid::system::InputSourceId::ConsoleSixAxisSensor::Mask);

    if ( inputSourceState.timestamp != m_LatestTimestamp )
    {
        m_LatestTimestamp = inputSourceState.timestamp;
        return true;
    }
    else
    {
        return false;
    }
}

bool ContinuousHighSkinTemperatureHandler::IsSkinTemperatureUnderThreshold() NN_NOEXCEPT
{
    return (nn::tc::GetSkinTemperatureMilliC() < m_ThresholdTemperatureMilliC);
}

void ContinuousHighSkinTemperatureHandler::SubmitErrorReport(nn::Result result) NN_NOEXCEPT
{
    nn::erpt::Context context(nn::erpt::ErrorInfo);

    auto errorCode = nn::err::detail::ConvertResultToErrorCode(result);
    char errorCodeString[nn::err::ErrorCode::StringLengthMax];
    nn::util::SNPrintf(errorCodeString, sizeof(errorCodeString), "%04d-%04d", errorCode.category, errorCode.number);

    NN_DETAIL_TCAP_WARN_UNLESS_RESULT_SUCCESS(context.Add(nn::erpt::ErrorCode, errorCodeString, static_cast<uint32_t>(sizeof(errorCodeString))));
    NN_DETAIL_TCAP_WARN_UNLESS_RESULT_SUCCESS(context.CreateReport(nn::erpt::ReportType_Invisible));
}

void ContinuousHighSkinTemperatureHandler::Initialize(int milliSeconds) NN_NOEXCEPT
{
    if ( milliSeconds > 0 )
    {
        m_ThresholdTime = nn::TimeSpan::FromMilliSeconds(milliSeconds);
    }
    else
    {
        m_ThresholdTime = nn::TimeSpan(0);
    }
}

void ContinuousHighSkinTemperatureHandler::Update(nn::TimeSpan elapsedTime) NN_NOEXCEPT
{
    if ( DetectValidInput() || IsSkinTemperatureUnderThreshold() || m_ThresholdTime == nn::TimeSpan(0) || m_OperationMode != OperationMode_Handheld )
    {
        m_ElapsedTime = nn::TimeSpan(0);
        return;
    }

    m_ElapsedTime += elapsedTime;

    if ( m_ElapsedTime >= m_ThresholdTime )
    {
        nn::os::SignalSystemEvent(&m_SystemEvent);
        SubmitErrorReport(ResultContinuousHighSkinTemperature());
        m_ElapsedTime = nn::TimeSpan(0);
    }
}

void ContinuousHighSkinTemperatureHandler::ClearElapsedTime() NN_NOEXCEPT
{
    m_ElapsedTime = nn::TimeSpan(0);
}

nn::os::SystemEventType* ContinuousHighSkinTemperatureHandler::GetSystemEvent() NN_NOEXCEPT
{
    return &m_SystemEvent;
}

void ContinuousHighSkinTemperatureHandler::SetOperationMode(OperationMode operationMode) NN_NOEXCEPT
{
    m_OperationMode = operationMode;
}

}}} // namespace nn::tcap::server
