﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <string>

#include <nn/nn_SdkAssert.h>

#include <nn/tc/detail/tc_Log.h>

namespace nn { namespace tc { namespace impl { namespace detail {

namespace {

class Parser
{

public:

    Parser()
    : m_TargetNest(0)
    , m_Nest(0)
    , m_BeginFlag(false)
    , m_pBegin(nullptr)
    , m_pEnd(nullptr)
    {
    }

    int Parse(int* pOutList, int listSize, const char* pBuffer, size_t bufferSize, int m_TargetNest) NN_NOEXCEPT;

private:
    void Initialize(int targetNest) NN_NOEXCEPT;
    void IncrementNest() NN_NOEXCEPT;
    void DecrementNest() NN_NOEXCEPT;
    void BeginToken() NN_NOEXCEPT;
    void SetBeginPtr(const char* pCurrent) NN_NOEXCEPT;
    void SetEndPtr(const char* pCurrent) NN_NOEXCEPT;
    bool EndToken(int* pOutValue) NN_NOEXCEPT;

private:

    // 値の含まれるネストの深度
    int m_TargetNest;

    // 現在のネストの深度。
    int m_Nest;

    // トークン毎の処理状態。
    bool m_BeginFlag;
    const char* m_pBegin;
    const char* m_pEnd;
};

void Parser::Initialize(int targetNest) NN_NOEXCEPT
{
    m_TargetNest = targetNest;
    m_Nest = 0;
    m_BeginFlag = false;
    m_pBegin = nullptr;
    m_pEnd = nullptr;
}

void Parser::IncrementNest() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Nest < m_TargetNest);

    m_Nest++;
}

void Parser::DecrementNest() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Nest > 0);

    m_Nest--;
}

void Parser::BeginToken() NN_NOEXCEPT
{
    if ( m_Nest == m_TargetNest )
    {
        m_BeginFlag = true;
    }
}

void Parser::SetBeginPtr(const char* pCurrent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCurrent);

    if ( m_Nest == m_TargetNest )
    {
        if ( m_BeginFlag )
        {
            m_BeginFlag = false;
            m_pBegin = pCurrent;
        }
    }
}

void Parser::SetEndPtr(const char* pCurrent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCurrent);

    if ( m_Nest == m_TargetNest )
    {
        if ( m_pBegin != nullptr && m_pEnd == nullptr )
        {
            m_pEnd = pCurrent;
        }
    }
}

bool Parser::EndToken(int* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    const size_t LocalBufferSize = 64;

    char buffer[LocalBufferSize];

    bool value = false;

    if ( m_Nest == m_TargetNest )
    {
        if ( m_pBegin != nullptr && m_pEnd != nullptr )
        {
            std::memset(buffer, '\0', LocalBufferSize);
            std::memcpy(buffer, m_pBegin, (static_cast<size_t>(m_pEnd - m_pBegin) > LocalBufferSize) ? LocalBufferSize : static_cast<size_t>(m_pEnd - m_pBegin));
            *pOutValue = std::atoi(buffer);
            value = true;
        }
    }

    m_BeginFlag = false;
    m_pBegin = nullptr;
    m_pEnd = nullptr;

    return value;
}

int Parser::Parse(int* pOutList, int listSize, const char* pBuffer, size_t bufferSize, int targetNest) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutList);
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    Initialize(targetNest);

    int listIndex = 0;

    const char* pEndBuffer = pBuffer + bufferSize;

    for ( const char* ptr = pBuffer; ptr < pEndBuffer && *ptr != '\0' && listIndex < listSize; ptr++ )
    {
        switch ( *ptr )
        {
        case '[':
            IncrementNest();
            BeginToken();
            break;

        case ']':
            SetEndPtr(ptr);
            if ( EndToken(pOutList + listIndex) )
            {
                listIndex++;
            }
            DecrementNest();
            break;

        case ',':
            SetEndPtr(ptr);
            if ( EndToken(pOutList + listIndex) )
            {
                listIndex++;
            }
            BeginToken();
            break;

        case ' ':
            SetEndPtr(ptr);
            break;

        default:
            SetBeginPtr(ptr);
            break;
        }
    }

    return listIndex;
}

} // namespace

// ネストされたシーケンスの文字列を一次元配列にして返す。
int ParseStringToSequence(int* pOutList, int listSize, const char* pBuffer, size_t bufferSize, int targetNest) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutList);
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);

    Parser localParser;
    return localParser.Parse(pOutList, listSize, pBuffer, bufferSize, targetNest);
}

}}}} // namespace nn::tc::impl::detail
