﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ts.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "tc_LocationAccessor.h"

namespace nn { namespace tc { namespace impl { namespace detail {

const TemperatureMilliC TemperatureJunctionError = 1000;

void LocationAccessor::Initialize() NN_NOEXCEPT
{
    nn::ts::Initialize();
}

void LocationAccessor::Finalize() NN_NOEXCEPT
{
    nn::ts::Finalize();
}

nn::Result LocationAccessor::GetRealTemperature(TemperatureMilliC* pOutTemperature) NN_NOEXCEPT
{
    int tsTemperatureMilliC = 0;
    NN_RESULT_DO(nn::ts::GetTemperatureMilliC(&tsTemperatureMilliC, m_Location));

    // TODO: 値がオーバーフローするケースをハンドリングする温度異常エラーが必要。（センサ故障時）
    *pOutTemperature = static_cast<TemperatureMilliC>(tsTemperatureMilliC);

    NN_RESULT_SUCCESS;
}

nn::Result LocationAccessor::GetTemperature(TemperatureMilliC* pOutTemperature) NN_NOEXCEPT
{
    if ( m_VirtualTemperatureEnabled )
    {
        *pOutTemperature = m_VirtualTemperature;
        return nn::ResultSuccess();
    }
    else
    {
        NN_RESULT_DO(GetRealTemperature(static_cast<int*>(pOutTemperature)));

        // GetRealTemperature ではセンサの即値を取得するべき。
        // センサ精度の考慮は GetTemperature で行う。
        *pOutTemperature += TemperatureJunctionError;

        return nn::ResultSuccess();
    }
}

void LocationAccessor::SetVirtualTemperature(TemperatureMilliC temperature) NN_NOEXCEPT
{
    m_VirtualTemperature = temperature;
}

void LocationAccessor::SetVirtualTemperatureEnabled(bool enabled) NN_NOEXCEPT
{
    m_VirtualTemperatureEnabled = enabled;
}

void LocationAccessor::SetPowerMode(PowerMode powerMode) NN_NOEXCEPT
{
    switch(powerMode)
    {
    case PowerMode_FullAwake:
    case PowerMode_MinimumAwake:
        nn::ts::SetMeasurementMode(m_Location, nn::ts::MeasurementMode::Continuous);
        break;
    case PowerMode_SleepReady:
        nn::ts::SetMeasurementMode(m_Location, nn::ts::MeasurementMode::Standby);

        // 仮想温度でスリープに入った場合を考慮して、復帰させるために仮想温度設定を初期値に戻す。
        m_VirtualTemperature = DefaultVirtualTemperature;
        m_VirtualTemperatureEnabled = false;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}}}} // namespace nn::tc::impl::detail
