﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/tc/detail/tc_Log.h>

namespace nn { namespace tc { namespace impl { namespace detail {

class FirmwareDebugSettingsAccessor {

public:
    template <typename T>
    void ReadSetting(T* pOutValue, const char* pName, const char* pKey) NN_NOEXCEPT
    {
        size_t size = nn::settings::fwdbg::GetSettingsItemValueSize(pName, pKey);
        if ( 0 == size )
        {
            NN_DETAIL_TC_WARN("Failed to read %s.%s (Value is not written)\n", pName, pKey);
            return;
        }
        else if ( sizeof(T) != size )
        {
            NN_DETAIL_TC_WARN("Failed to read %s.%s (data size does not match). expected=%d, actual=%d\n", pName, pKey, sizeof(T), size);
            return;
        }

        T temporaryValue;
        if ( sizeof(T) != (nn::settings::fwdbg::GetSettingsItemValue(&temporaryValue, sizeof(T), pName, pKey)) )
        {
            NN_DETAIL_TC_WARN("Failed to read %s.%s.\n", pName, pKey);
        }
        else
        {
            *pOutValue = temporaryValue;
        }
    }

    template <typename T>
    void ReadSetting(T* pOutValue, const char* pKey) NN_NOEXCEPT
    {
        ReadSetting(pOutValue, "tc", pKey);
    }

    size_t ReadSettingString(char* pOutBuffer, size_t bufferSize, const char* pName, const char* pKey) NN_NOEXCEPT
    {
        size_t size = nn::settings::fwdbg::GetSettingsItemValueSize(pName, pKey);
        if ( 0 == size )
        {
            NN_DETAIL_TC_WARN("Failed to read %s.%s (Value is not written)\n", pName, pKey);
            return 0;
        }
        if ( bufferSize < size )
        {
            NN_DETAIL_TC_WARN("Failed to read %s.%s (output buffer size is not sufficient). buffer size=%d, data size=%d\n", pName, pKey, bufferSize, size);
            return 0;
        }

        size = nn::settings::fwdbg::GetSettingsItemValue(pOutBuffer, bufferSize, pName, pKey);
        if ( bufferSize < size || 0 == size )
        {
            NN_DETAIL_TC_WARN("Failed to read %s.%s.\n", pName, pKey);
            return 0;
        }

        return size;
    }

    size_t ReadSettingString(char* pOutBuffer, size_t bufferSize, const char* pKey) NN_NOEXCEPT
    {
        return ReadSettingString(pOutBuffer, bufferSize, "tc", pKey);
    }
};

}}}} // namespace nn::tc::impl::detail
