﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ソフトウェアキーボード呼び出しに関する公開ヘッダ
 */

#include <nn/swkbd/swkbd_Api.h>
#include <nn/la/la_Api.h>
#include <string>

#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/nn_StaticAssert.h>

#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_CharacterEncodingResult.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_BitUtil.h>

#include <nn/swkbd/swkbd_PrivateTypes.h>
#include <nn/swkbd/swkbd_Result.h>

namespace nn { namespace swkbd {

// enum のバイト数チェック
NN_STATIC_ASSERT( sizeof( Preset ) == 4 );
NN_STATIC_ASSERT( sizeof( KeyboardMode ) == 4 );
NN_STATIC_ASSERT( sizeof( InvalidChar ) == 4 );
NN_STATIC_ASSERT( sizeof( PasswordMode ) == 4 );
NN_STATIC_ASSERT( sizeof( InputFormMode ) == 4 );
NN_STATIC_ASSERT( sizeof( InitialCursorPos ) == 4 );
NN_STATIC_ASSERT( sizeof( TextCheckResult ) == 4 );

// (開発用) swkbd 終了理由のキャッシュ
nn::applet::LibraryAppletExitReason g_ExitReason = nn::applet::LibraryAppletExitReason_Normal;

//------------------------------------------------------------------------------
// UTF-8の文字列をUTF-16に変換します。
void ConvertUtf8ToUtf16( void* dst, int dstLength, const char* src, int srcLength ) NN_NOEXCEPT
{
    nn::util::CharacterEncodingResult encoding_result;
    encoding_result = nn::util::ConvertStringUtf8ToUtf16Native( (uint16_t*)dst, dstLength, src, srcLength );
    if( encoding_result != nn::util::CharacterEncodingResult_Success )
    {
        if( encoding_result == nn::util::CharacterEncodingResult_InsufficientLength )
        {
            NN_ABORT( "Insufficient Length." );
        }
        else if( encoding_result == nn::util::CharacterEncodingResult_InvalidFormat )
        {
            NN_ABORT( "Invalid Format." );
        }
    }
}


//------------------------------------------------------------------------------
// UTF-8の文字列をUTF-16に変換します。終端文字まで変換されます。
void ConvertUtf8ToUtf16( void* dst, int dstLength, const char* src ) NN_NOEXCEPT
{
    nn::util::CharacterEncodingResult encoding_result;
    encoding_result = nn::util::ConvertStringUtf8ToUtf16Native( (uint16_t*)dst, dstLength, src );
    if( encoding_result != nn::util::CharacterEncodingResult_Success )
    {
        if( encoding_result == nn::util::CharacterEncodingResult_InsufficientLength )
        {
            NN_ABORT( "Insufficient Length." );
        }
        else if( encoding_result == nn::util::CharacterEncodingResult_InvalidFormat )
        {
            NN_ABORT( "Invalid Format." );
        }
    }
}


//------------------------------------------------------------------------------
// UTF-8の文字列をUTF-16に変換するために必要な変換先の長さを返します
int GetLengthOfConvertedStringUtf8ToUtf16( const char* pSrc ) NN_NOEXCEPT
{
    int src_length = 0;
    while( true )
    {
        if( pSrc[ src_length ] == 0 )
        {
            break;
        }
        ++src_length;
    }

    int str_length = 0;

    nn::util::CharacterEncodingResult encoding_result = nn::util::GetLengthOfConvertedStringUtf8ToUtf16Native(
        &str_length, pSrc, src_length );

    if( encoding_result != nn::util::CharacterEncodingResult_Success )
    {
        if( encoding_result == nn::util::CharacterEncodingResult_InvalidFormat )
        {
            NN_ABORT( "Invalid Format." );
        }
    }

    return str_length;
}

//------------------------------------------------------------------------------
//
NN_FORCEINLINE
void* AddOffset( void* ptr, ptrdiff_t offset ) NN_NOEXCEPT
{
    // ptr は非破壊
    return static_cast<char*>( ptr ) + offset;
}


//---------------------------------------------------------------------------
//
void MakePresetDefault( KeyboardConfig* pOutKeyboardConfig ) NN_NOEXCEPT
{
    std::memset(pOutKeyboardConfig, 0, sizeof(*pOutKeyboardConfig) );
    pOutKeyboardConfig->keyboardMode = KeyboardMode_ASCII;
    pOutKeyboardConfig->leftOptionalSymbolKey = 0;
    pOutKeyboardConfig->rightOptionalSymbolKey = 0;
    pOutKeyboardConfig->isPredictionEnabled = false;
    pOutKeyboardConfig->invalidCharFlag = 0;
    pOutKeyboardConfig->initialCursorPos = InitialCursorPos_Last;
    pOutKeyboardConfig->textMaxLength = 0;
    pOutKeyboardConfig->textMinLength = 0;
    pOutKeyboardConfig->passwordMode = PasswordMode_Show;
    pOutKeyboardConfig->inputFormMode = InputFormMode_MultiLine;
    pOutKeyboardConfig->isUseNewLine = true;
    pOutKeyboardConfig->isUseUtf8 = false;
    pOutKeyboardConfig->isUseBlurBackground = true;
    // (未設定用の負の値を設定する)
    for(int i = 0; i < SepareteTextPosMax; ++i)
    {
        pOutKeyboardConfig->separateTextPos[i] = -1;
    }
}

//---------------------------------------------------------------------------
//
void MakePresetPassword( KeyboardConfig* pOutKeyboardConfig ) NN_NOEXCEPT
{
    std::memset(pOutKeyboardConfig, 0, sizeof(*pOutKeyboardConfig));
    pOutKeyboardConfig->keyboardMode = KeyboardMode_ASCII;
    pOutKeyboardConfig->leftOptionalSymbolKey = 0;
    pOutKeyboardConfig->rightOptionalSymbolKey = 0;
    pOutKeyboardConfig->isPredictionEnabled = false;
    pOutKeyboardConfig->invalidCharFlag = 0;
    pOutKeyboardConfig->initialCursorPos = InitialCursorPos_Last;
    pOutKeyboardConfig->textMaxLength = 0;
    pOutKeyboardConfig->textMinLength = 0;
    pOutKeyboardConfig->passwordMode = PasswordMode_Hide;
    pOutKeyboardConfig->inputFormMode = InputFormMode_OneLine;
    pOutKeyboardConfig->isUseNewLine = false;
    pOutKeyboardConfig->isUseUtf8 = false;
    pOutKeyboardConfig->isUseBlurBackground = true;
    // (未設定用の負の値を設定する)
    for(int i = 0; i < SepareteTextPosMax; ++i)
    {
        pOutKeyboardConfig->separateTextPos[i] = -1;
    }
}

//---------------------------------------------------------------------------
//
void MakePresetUserName( KeyboardConfig* pOutKeyboardConfig ) NN_NOEXCEPT
{
    std::memset(pOutKeyboardConfig, 0, sizeof(*pOutKeyboardConfig));
    pOutKeyboardConfig->keyboardMode = KeyboardMode_Full;
    pOutKeyboardConfig->leftOptionalSymbolKey = 0;
    pOutKeyboardConfig->rightOptionalSymbolKey = 0;
    pOutKeyboardConfig->isPredictionEnabled = false;
    pOutKeyboardConfig->invalidCharFlag = InvalidCharFlag_OutsideOfMiiNickName;
    pOutKeyboardConfig->initialCursorPos = InitialCursorPos_Last;
    pOutKeyboardConfig->textMaxLength = 0;
    pOutKeyboardConfig->textMinLength = 0;
    pOutKeyboardConfig->passwordMode = PasswordMode_Show;
    pOutKeyboardConfig->inputFormMode = InputFormMode_OneLine;
    pOutKeyboardConfig->isUseNewLine = false;
    pOutKeyboardConfig->isUseUtf8 = false;
    pOutKeyboardConfig->isUseBlurBackground = true;
    // (未設定用の負の値を設定する)
    for(int i = 0; i < SepareteTextPosMax; ++i)
    {
        pOutKeyboardConfig->separateTextPos[i] = -1;
    }
}

//---------------------------------------------------------------------------
//
void MakePresetDownloadCode( KeyboardConfig* pOutKeyboardConfig ) NN_NOEXCEPT
{
    std::memset(pOutKeyboardConfig, 0, sizeof(*pOutKeyboardConfig));
    pOutKeyboardConfig->keyboardMode = KeyboardMode_ASCII;
    pOutKeyboardConfig->leftOptionalSymbolKey = 0;
    pOutKeyboardConfig->rightOptionalSymbolKey = 0;
    pOutKeyboardConfig->isPredictionEnabled = false;
    pOutKeyboardConfig->invalidCharFlag = InvalidCharFlag_OutsideOfDownloadCode;
    pOutKeyboardConfig->initialCursorPos = InitialCursorPos_Last;
    pOutKeyboardConfig->textMaxLength = 16;
    pOutKeyboardConfig->textMinLength = 16;
    pOutKeyboardConfig->passwordMode = PasswordMode_Show;
    pOutKeyboardConfig->inputFormMode = InputFormMode_Separate;
    pOutKeyboardConfig->isUseNewLine = false;
    pOutKeyboardConfig->isUseUtf8 = false;
    pOutKeyboardConfig->isUseBlurBackground = true;
    // (未設定用の負の値を設定する)
    for(int i = 0; i < SepareteTextPosMax; ++i)
    {
        pOutKeyboardConfig->separateTextPos[i] = -1;
    }
    pOutKeyboardConfig->separateTextPos[0] = 4 * 1 - 1;
    pOutKeyboardConfig->separateTextPos[1] = 4 * 2 - 1;
    pOutKeyboardConfig->separateTextPos[2] = 4 * 3 - 1;
}


//---------------------------------------------------------------------------
//
void GetInteractiveOutStorageCallback( nn::applet::LibraryAppletHandle laHandle,
    String* pOutResultString,
    const ShowKeyboardArg& showKeyboardArg ) NN_NOEXCEPT
{
    // 溜まっているストレージを取得する

    nn::applet::StorageHandle handle;

    // ストレージの取得を試みる
    size_t buf_size = 0; // ヌル終端を含む入力テキストのサイズ
    if( nn::applet::TryPopFromInteractiveOutChannel( &handle, laHandle ) )
    {
        // データがあるなら読み込む
        uint64_t size = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage( handle,
            0, &size, sizeof( uint64_t ) ) );
        // NX32 / NX64 時に size_t のバイトサイズが異なるため、そろえる( buf_size は数千byte程度なので本対応で吸収可能 )
        buf_size = static_cast<size_t>(size);
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage( handle,
            sizeof( uint64_t ), pOutResultString->ptr, buf_size) );

        nn::applet::ReleaseStorage( handle );
    }
    else
    {
        // ストレージがあると言われているのに、取得に失敗した場合、
        // 何か内部でエラーが起きている場合がある
        // ここでは、ひとまず処理を終わらせて、次のメッセージを待ってみる
        return;
    }

    TextCheckResult result = TextCheckResult_ShowFailureDialog; // チェック結果
    size_t text_buffer_size = 0; // ダイアログの文字数

    // テキストチェック有効時のみ、チェックを行う
    if( showKeyboardArg.keyboardConfig._isUseTextCheck &&
        showKeyboardArg.keyboardConfig._textCheckCallback != nullptr )
    {
        // アプリ側でテキストチェック開始
        auto original_size = pOutResultString->bufSize;
        pOutResultString->bufSize = buf_size;       // 通知中はバッファサイズを一時的に入力テキストサイズに変更
        result = showKeyboardArg.keyboardConfig._textCheckCallback(
            showKeyboardArg.textCheckWorkBuf, &text_buffer_size, pOutResultString );
        pOutResultString->bufSize = original_size;  // バッファサイズに戻しておく
    }
    // なんらかの拍子でテキストチェックを素通りしたら、何も言わずエラーのダイアログを出す

    // swkbd 側に結果を伝える（パラメータを渡す）
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateStorage( &handle, sizeof( TextCheckResult ) + GetRequiredTextCheckWorkBufferSize() ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage( handle, 0, &result, sizeof( TextCheckResult ) ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::WriteToStorage( handle, sizeof( TextCheckResult ), showKeyboardArg.textCheckWorkBuf, GetRequiredTextCheckWorkBufferSize() ) );
    nn::applet::PushToInteractiveInChannel( laHandle, handle );
}


//---------------------------------------------------------------------------
//
void ReadCloseResultAndString( nn::applet::LibraryAppletHandle laHandle,
    CloseResult* pOutCloseResult,
    String* pOutResultString ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pOutCloseResult != nullptr,  "[swkbd] nn::swkbd::CloseResult must not be null." );
    NN_SDK_ASSERT( pOutResultString != nullptr, "[swkbd] nn::swkbd::String must not be null." );

    nn::applet::StorageHandle out_handle;
    // ストレージの取得を試みる
    if( nn::applet::TryPopFromOutChannel( &out_handle, laHandle ) )
    {
        // 終了理由を読み込む
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage( out_handle, 0,
            pOutCloseResult, sizeof( CloseResult ) ) );

        if( pOutResultString->ptr != nullptr && pOutResultString->bufSize > 0 )
        {
            // データがあるなら読み込む ( オフセット値をずらしておく )
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::ReadFromStorage( out_handle, sizeof( CloseResult ),
                pOutResultString->ptr, pOutResultString->bufSize ) );
        }
        nn::applet::ReleaseStorage( out_handle );
    }
}


//---------------------------------------------------------------------------
// WorkBufHeader に書き込む初期文字列情報を記憶しておく
void CopyInitialStringInfo( ShowKeyboardArg* pOutShowKeyboardArg, int32_t length ) NN_NOEXCEPT
{
    pOutShowKeyboardArg->keyboardConfig._initialStringLength = length;

    // 範囲外の文字数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( pOutShowKeyboardArg->keyboardConfig._initialStringLength >= 0 &&
        pOutShowKeyboardArg->keyboardConfig._initialStringLength <= TextMaxLength ),
        "[swkbd] string size is out of range.\n" );

    // 上記の ASSERT 該当時は、Release 時に何もしないようにする
    if( ( pOutShowKeyboardArg->keyboardConfig._initialStringLength < 0 ) )
    {
        return;
    }
    if( ( pOutShowKeyboardArg->keyboardConfig._initialStringLength > TextMaxLength ) )
    {
        return;
    }

    // 文字列開始位置を計算
    pOutShowKeyboardArg->keyboardConfig._initialStringOffset = sizeof( WorkBufHeaderV3 );
}

//---------------------------------------------------------------------------
// WorkBufHeader に書き込むユーザー辞書情報を記憶しておく
void CopyUserDictionaryInfo( ShowKeyboardArg* pOutShowKeyboardArg, int32_t wordNum ) NN_NOEXCEPT
{
    // ユーザー辞書の総数を記憶しておく
    pOutShowKeyboardArg->keyboardConfig._userDictionaryNum = wordNum;

    // ユーザー辞書用のバッファを事前に確保しているかどうか
    NN_SDK_ASSERT( ( pOutShowKeyboardArg->workBuf != nullptr ), "[swkbd] workBuf must not be null.\n" );
    // ユーザー辞書分の十分なメモリサイズを確保しているかどうか
    NN_SDK_ASSERT( ( pOutShowKeyboardArg->workBufSize >= GetRequiredWorkBufferSize( wordNum ) ),
        "[swkbd] workBufSize < nn::swkbd::GetRequiredWorkBufferSize( wordNum )" );

    // 上記の ASSERT 該当時は、Release 時に何もしないようにする
    if( pOutShowKeyboardArg->workBuf == nullptr )
    {
        return;
    }
    if( pOutShowKeyboardArg->workBufSize < GetRequiredWorkBufferSize( wordNum ) )
    {
        // 要求バッファより少ないバッファサイズになっている
        return;
    }

    // userDictionaryOffset に入れるオフセット値を計算
    pOutShowKeyboardArg->keyboardConfig._userDictionaryOffset = sizeof( WorkBufHeaderV3 ) + c_FixedStringUtf8BufSize;
}


//---------------------------------------------------------------------------
//
nn::Result ShowKeyboard( String* pOutResultString, const ShowKeyboardArg& showKeyboardArg ) NN_NOEXCEPT
{
    // ワークバッファは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( showKeyboardArg.workBuf != nullptr ), "[swkbd] workBuf not be null." );
    NN_SDK_ASSERT( ( pOutResultString->ptr != nullptr && pOutResultString->bufSize > 0 ), "[swkbd] nn::swkbd::String must not be null." );

    // この時点でパラメータの設定に矛盾があったらエラーを返す
    // 予測変換と隠蔽処理が両方有効だと ASSERT
    if( showKeyboardArg.keyboardConfig.isPredictionEnabled == true &&
        showKeyboardArg.keyboardConfig.passwordMode == PasswordMode_Hide )
    {
        NN_SDK_ASSERT( false, "[swkbd] prediction mode and password mode must not set it at the same time." );
    }
    // 区切り文字かつ Full/FullLatin モードで止めたい場合は以下のコードを仕込む
    if( showKeyboardArg.keyboardConfig.inputFormMode == InputFormMode_Separate &&
        ( showKeyboardArg.keyboardConfig.keyboardMode == KeyboardMode_Full ||
            showKeyboardArg.keyboardConfig.keyboardMode == KeyboardMode_FullLatin ) )
    {
        //NN_SDK_ASSERT( false, "[swkbd] separate input form mode and Full/FullLatin keyboard mode must not set it at the same time." );
    }

    // LA 共通パラメータを設定する
    // 共通パラメータは独立したストレージとして扱う
    // 引数には swkbd shim のバージョンを埋め込む
    nn::la::CommonArgumentsWriter commonArg( c_CurrentMajorVersion, c_CurrentMinorVersion );

    // swkbd 起動用のアプレットハンドル
    nn::applet::LibraryAppletHandle laHandle;
    // swkbd 起動準備を行う
    // ここでは FG 起動をする
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateLibraryApplet(
        &laHandle,
        nn::applet::AppletId_LibraryAppletSwkbd,
        nn::applet::LibraryAppletMode_AllForeground ) );

    // まず初めに LA 共通パラメータ用ストレージを送信する
    commonArg.PushToInChannel( laHandle );
    // 次に swkbd 固有のパラメータを送信する
    nn::la::PushToInChannel( laHandle,
        &showKeyboardArg.keyboardConfig, sizeof( KeyboardConfig ) );

    // 辞書などの大きなデータをラージストレージとして送信する
    nn::applet::StorageHandle large_storage_handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::CreateLargeStorage( &large_storage_handle,
        showKeyboardArg.workBuf, showKeyboardArg.workBufSize, true ) );
    nn::applet::PushToInChannel( laHandle, large_storage_handle );

    // 呼び出し元 LA に渡すパラメータの準備
    nn::la::LibraryAppletStartHookUserArg userArg;
    userArg.isExtremity = true;

    // swkbd 終了理由を格納する変数
    // ただし、この変数は「LAが正しく終了したか」の判定にのみ使う
    // swkbd 内でキャンセル入力があったかどうかは、別で判定する
    nn::Result la_result;

    // swkbd の起動
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::applet::StartLibraryApplet( laHandle, &userArg ) );

    // swkbd の終了待機
    // 途中、場合によってはアプリ側に処理を返す時があるので、その対応
    // 以下の while 文は、swkbd 終了時に false を返す
    while( nn::applet::WaitPopFromInteractiveOutChannelEvent( laHandle ) )
    {
        // ここに来る場合は、インタラクティブ出力チャンネルに pop できるデータがあることを意味する
        // 現時点では、テキストチェックを行う場合のみ、ここを通る
        GetInteractiveOutStorageCallback( laHandle, pOutResultString, showKeyboardArg );
    }

    // ライブラリアプレットの終了確認
    nn::applet::JoinLibraryApplet( laHandle );

    // ライブラリアプレット終了原因の確認
    g_ExitReason = nn::applet::GetLibraryAppletExitReason( laHandle );

    // SAがLAを強制終了させたとき LibraryAppletExitReason_Canceledになる
    // それは許容する必要がある。
    // 異常終了のときはABORTでいいはず。
    // ポリシー：http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=164702960
    NN_ABORT_UNLESS( g_ExitReason != nn::applet::LibraryAppletExitReason_Abnormal );
    // Unexpectedはデバッグで撲滅する必要がある
    NN_ABORT_UNLESS( g_ExitReason != nn::applet::LibraryAppletExitReason_Unexpected );
    if( g_ExitReason == nn::applet::LibraryAppletExitReason_Canceled )
    {
        la_result = nn::la::ResultLibraryAppletCanceled();
    }
    else
    {
        // ここまでで問題なければ、成功とする
        la_result = nn::ResultSuccess();
    }

    // 終了理由格納用変数の準備
    CloseResult close_result = CloseResult_Cancel;

    // swkbd は終了時にパラメータを渡しているはずなので、それを受け取る
    if( la_result.IsSuccess() )
    {
        ReadCloseResultAndString( laHandle, &close_result, pOutResultString );
    }

    // これ以上 swkbd からデータを受け取る必要はない
    // swkbd を終了する
    nn::applet::CloseLibraryApplet( laHandle );

    // swkbd が返した終了理由に従って、返り値を決める
    if( la_result.IsSuccess() ||
        nn::applet::ResultLibraryAppletExited::Includes( la_result ) )
    {
        // 正常終了の場合は条件判定を行う
        if( close_result == CloseResult_Enter )
        {
            return nn::ResultSuccess();
        }
        else
        {
            // キャンセルボタンを押したり、何らかの異常終了の場合はキャンセルとする
            return nn::swkbd::ResultCanceled();
        }
    }
    else
    {
        // 異常終了の場合はキャンセルとする
        return nn::swkbd::ResultCanceled();
    }
}


//---------------------------------------------------------------------------
//
void InitializeKeyboardConfig( KeyboardConfig* pOutKeyboardConfig ) NN_NOEXCEPT
{
    pOutKeyboardConfig->_initialStringOffset = 0;
    pOutKeyboardConfig->_initialStringLength = 0;
    pOutKeyboardConfig->_userDictionaryOffset = 0;
    pOutKeyboardConfig->_userDictionaryNum = 0;
    pOutKeyboardConfig->_isUseTextCheck = false;
    pOutKeyboardConfig->_textCheckCallback = nullptr;
}


//---------------------------------------------------------------------------
//
void MakePreset( KeyboardConfig* pOutKeyboardConfig, Preset preset ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    switch( preset )
    {
    case Preset_Default:
        MakePresetDefault( pOutKeyboardConfig );
        break;
    case Preset_Password:
        MakePresetPassword( pOutKeyboardConfig );
        break;
    case Preset_UserName:
        MakePresetUserName( pOutKeyboardConfig );
        break;
    case Preset_DownloadCode:
        MakePresetDownloadCode( pOutKeyboardConfig );
        break;
    default:
        // プリセットにないものを選択している
        NN_ABORT( "Unknown Preset.\n" );
        break;
    }

    // 文字列バッファの初期化
    std::memset( pOutKeyboardConfig->okText, 0, OkTextMaxLength * sizeof( char16_t ) );
    std::memset( pOutKeyboardConfig->headerText, 0, HeaderTextMaxLength * sizeof( char16_t ) );
    std::memset( pOutKeyboardConfig->subText, 0, SubTextMaxLength * sizeof( char16_t ) );
    std::memset( pOutKeyboardConfig->guideText, 0, GuideTextMaxLength * sizeof( char16_t ) );

    // 内部的なパラメータを初期化する
    InitializeKeyboardConfig( pOutKeyboardConfig );
}


//---------------------------------------------------------------------------
//
size_t GetRequiredWorkBufferSize( bool useDictionary ) NN_NOEXCEPT
{
    // ユーザー辞書を使う場合、
    // 500 単語を指定した場合のバッファサイズを返す
    if( useDictionary )
    {
        return GetRequiredWorkBufferSize( 500 );
    }
    else
    {
        return GetRequiredWorkBufferSize( 0 );
    }
}


//---------------------------------------------------------------------------
//
size_t GetRequiredWorkBufferSize( const int userWordNum ) NN_NOEXCEPT
{
    // 最大値より大きな単語数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( userWordNum >= 0 && userWordNum <= UserWordMax ),
        "[swkbd] user word size is out of range.\n" );
    if( userWordNum < 0 || userWordNum > UserWordMax )
    {
        return 0;
    }

    // 文字必要バッファ：UTF-8 を考慮して、( 確定文字列 + 文字終端 )の4倍分は確保しておく
    size_t required_size = c_FixedStringUtf8BufSize;

    // 辞書必要バッファ：ユーザー辞書の最大サイズ
    if( userWordNum > 0 )
    {
        required_size += sizeof( UserWord ) * userWordNum;
    }

    // nn::os::MemoryPageSize でアライメント
    required_size = nn::util::align_up( required_size, nn::os::MemoryPageSize );

    return required_size;
}


//---------------------------------------------------------------------------
//
size_t GetRequiredTextCheckWorkBufferSize() NN_NOEXCEPT
{
    // 文字必要バッファ：UTF-8 を考慮して、( ダイアログ用文字数 + 文字終端 )の4倍分は確保しておく
    size_t required_size = ( DialogTextMaxLength + 1 ) * 4;

    return required_size;
}


//---------------------------------------------------------------------------
//
size_t GetRequiredStringBufferSize() NN_NOEXCEPT
{
    // 文字必要バッファ：UTF-8 を考慮して、( 確定文字列 + 文字終端 )の4倍分は確保しておく
    size_t required_size = c_FixedStringUtf8BufSize;

    return required_size;
}


//---------------------------------------------------------------------------
//
void SetOkText( KeyboardConfig* pOutKeyboardConfig, const char16_t* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 先頭にヌル終端をつけて終了する
        pOutKeyboardConfig->okText[ 0 ] = 0;
        return;
    }

    // 文字列の長さを取得する
    int str_length = nn::util::Strnlen( pStr, OkTextMaxLength + 1 );

    // 最大値より大きな文字数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( str_length > 0 && str_length < OkTextMaxLength + 1 ), "[swkbd] string size is out of range.\n" );

    if( str_length == OkTextMaxLength + 1 )
    {
        str_length = OkTextMaxLength;
    }

    // 最大値より大きな文字数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( str_length > 0 && str_length <= OkTextMaxLength ), "[swkbd] string size is out of range.\n" );

    std::memcpy( pOutKeyboardConfig->okText, pStr, str_length * sizeof( char16_t ) );

    // 末尾にヌル終端をつける
    pOutKeyboardConfig->okText[ str_length ] = 0;
}


//---------------------------------------------------------------------------
//
void SetOkTextUtf8( KeyboardConfig* pOutKeyboardConfig, const char* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 先頭にヌル終端をつけて終了する
        pOutKeyboardConfig->okText[ 0 ] = 0;
        return;
    }

    // 文字列の長さを計算する
    int str_length = GetLengthOfConvertedStringUtf8ToUtf16( pStr );

    // 最大値より大きな文字数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( str_length > 0 && str_length <= OkTextMaxLength ), "[swkbd] string size is out of range.\n" );

    ConvertUtf8ToUtf16( &( pOutKeyboardConfig->okText[ 0 ] ), OkTextMaxLength + 1, pStr );

    // 末尾にヌル終端をつける
    pOutKeyboardConfig->okText[ str_length ] = 0;
}


//---------------------------------------------------------------------------
//
void SetLeftOptionalSymbolKey( KeyboardConfig* pOutKeyboardConfig, const char16_t code ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    pOutKeyboardConfig->leftOptionalSymbolKey = code;
}


//---------------------------------------------------------------------------
//
void SetLeftOptionalSymbolKeyUtf8( KeyboardConfig* pOutKeyboardConfig, const char* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    ConvertUtf8ToUtf16( &pOutKeyboardConfig->leftOptionalSymbolKey, 1, pStr, std::strlen( pStr ) );
}


//---------------------------------------------------------------------------
//
void SetRightOptionalSymbolKey( KeyboardConfig* pOutKeyboardConfig, const char16_t code ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    pOutKeyboardConfig->rightOptionalSymbolKey = code;
}


//---------------------------------------------------------------------------
//
void SetRightOptionalSymbolKeyUtf8( KeyboardConfig* pOutKeyboardConfig, const char* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    ConvertUtf8ToUtf16( &pOutKeyboardConfig->rightOptionalSymbolKey, 1, pStr, std::strlen( pStr ) );
}


//---------------------------------------------------------------------------
//
void SetHeaderText( KeyboardConfig* pOutKeyboardConfig, const char16_t* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 先頭にヌル終端をつけて終了する
        pOutKeyboardConfig->headerText[ 0 ] = 0;
        return;
    }

    // 文字列の長さを取得する
    int str_length = nn::util::Strnlen( pStr, HeaderTextMaxLength + 1 );

    // 最大値より大きな文字数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( str_length > 0 && str_length < HeaderTextMaxLength + 1 ), "[swkbd] string size is out of range.\n" );

    if( str_length == HeaderTextMaxLength + 1 )
    {
        str_length = HeaderTextMaxLength;
    }

    std::memcpy( pOutKeyboardConfig->headerText, pStr, str_length * sizeof( char16_t ) );

    // 末尾にヌル終端をつける
    pOutKeyboardConfig->headerText[ str_length ] = 0;
}


//---------------------------------------------------------------------------
//
void SetHeaderTextUtf8( KeyboardConfig* pOutKeyboardConfig, const char* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 先頭にヌル終端をつけて終了する
        pOutKeyboardConfig->headerText[ 0 ] = 0;
        return;
    }

    // 文字列の長さを計算する
    int str_length = GetLengthOfConvertedStringUtf8ToUtf16( pStr );

    // 最大値より大きな文字数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( str_length > 0 && str_length <= HeaderTextMaxLength ), "[swkbd] string size is out of range.\n" );

    ConvertUtf8ToUtf16( &( pOutKeyboardConfig->headerText[ 0 ] ), HeaderTextMaxLength + 1, pStr );

    // 末尾にヌル終端をつける
    pOutKeyboardConfig->headerText[ str_length ] = 0;
}


//---------------------------------------------------------------------------
//
void SetSubText( KeyboardConfig* pOutKeyboardConfig, const char16_t* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 先頭にヌル終端をつけて終了する
        pOutKeyboardConfig->subText[ 0 ] = 0;
        return;
    }

    // 文字列の長さを取得する
    int str_length = nn::util::Strnlen( pStr, SubTextMaxLength + 1 );

    // 最大値より大きな文字数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( str_length > 0 && str_length < SubTextMaxLength + 1 ), "[swkbd] string size is out of range.\n" );

    if( str_length == SubTextMaxLength + 1 )
    {
        str_length = SubTextMaxLength;
    }

    std::memcpy( pOutKeyboardConfig->subText, pStr, str_length * sizeof( char16_t ) );

    // 末尾にヌル終端をつける
    pOutKeyboardConfig->subText[ str_length ] = 0;
}


//---------------------------------------------------------------------------
//
void SetSubTextUtf8( KeyboardConfig* pOutKeyboardConfig, const char* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 先頭にヌル終端をつけて終了する
        pOutKeyboardConfig->subText[ 0 ] = 0;
        return;
    }

    // 文字列の長さを計算する
    int str_length = GetLengthOfConvertedStringUtf8ToUtf16( pStr );

    // 最大値より大きな文字数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( str_length > 0 && str_length <= SubTextMaxLength ), "[swkbd] string size is out of range.\n" );

    ConvertUtf8ToUtf16( &( pOutKeyboardConfig->subText[ 0 ] ), SubTextMaxLength + 1, pStr );

    // 末尾にヌル終端をつける
    pOutKeyboardConfig->subText[ str_length ] = 0;
}


//---------------------------------------------------------------------------
//
void SetGuideText( KeyboardConfig* pOutKeyboardConfig, const char16_t* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 先頭にヌル終端をつけて終了する
        pOutKeyboardConfig->guideText[ 0 ] = 0;
        return;
    }

    // 文字列の長さを取得する
    int str_length = nn::util::Strnlen( pStr, GuideTextMaxLength + 1 );

    // 最大値より大きな文字数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( str_length > 0 && str_length < GuideTextMaxLength + 1 ), "[swkbd] string size is out of range.\n" );

    if( str_length == GuideTextMaxLength + 1 )
    {
        str_length = GuideTextMaxLength;
    }

    std::memcpy( pOutKeyboardConfig->guideText, pStr, str_length * sizeof( char16_t ) );

    // 末尾にヌル終端をつける
    pOutKeyboardConfig->guideText[ str_length ] = 0;
}


//---------------------------------------------------------------------------
//
void SetGuideTextUtf8( KeyboardConfig* pOutKeyboardConfig, const char* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutKeyboardConfig != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 先頭にヌル終端をつけて終了する
        pOutKeyboardConfig->guideText[ 0 ] = 0;
        return;
    }

    // 文字列の長さを計算する
    int str_length = GetLengthOfConvertedStringUtf8ToUtf16( pStr );

    // 最大値より大きな文字数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( str_length > 0 && str_length <= GuideTextMaxLength ), "[swkbd] string size is out of range.\n" );

    ConvertUtf8ToUtf16( &( pOutKeyboardConfig->guideText[ 0 ] ), GuideTextMaxLength + 1, pStr );

    // 末尾にヌル終端をつける
    pOutKeyboardConfig->guideText[ str_length ] = 0;
}


//---------------------------------------------------------------------------
//
void SetInitialText( ShowKeyboardArg* pOutShowKeyboardArg, const char16_t* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutShowKeyboardArg != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );
    // ワークバッファの設定は必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutShowKeyboardArg->workBuf != nullptr ), "[swkbd] workBuf must not be null.\n" );

    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 文字サイズを 0 にする
        // workBuf はユーザー辞書を使う可能性もあるので nullptr にしない
        pOutShowKeyboardArg->keyboardConfig._initialStringLength = 0;
        return;
    }

    // 文字列の長さを取得する
    int32_t length = nn::util::Strnlen( pStr, TextMaxLength );

    CopyInitialStringInfo( pOutShowKeyboardArg, length );

    // ヘッダ分位置をずらす
    void* ptr = AddOffset( pOutShowKeyboardArg->workBuf, pOutShowKeyboardArg->keyboardConfig._initialStringOffset );

    // 文字列をバッファに追加
    std::memcpy( ptr, pStr, pOutShowKeyboardArg->keyboardConfig._initialStringLength * sizeof( char16_t ) );
}


//---------------------------------------------------------------------------
//
void SetInitialTextUtf8( ShowKeyboardArg* pOutShowKeyboardArg, const char* pStr ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutShowKeyboardArg != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );
    // ワークバッファの設定は必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutShowKeyboardArg->workBuf != nullptr ), "[swkbd] workBuf must not be null.\n" );

    // 未指定の場合は初期化する
    if( pStr == nullptr )
    {
        // 文字サイズを 0 にする
        // workBuf はユーザー辞書を使う可能性もあるので nullptr にしない
        pOutShowKeyboardArg->keyboardConfig._initialStringLength = 0;
        return;
    }

    // 文字列の長さを計算する
    int32_t length = GetLengthOfConvertedStringUtf8ToUtf16( pStr );

    CopyInitialStringInfo( pOutShowKeyboardArg, length );

    // ヘッダ分位置をずらす
    void* ptr = AddOffset( pOutShowKeyboardArg->workBuf, pOutShowKeyboardArg->keyboardConfig._initialStringOffset );

    // 文字列をバッファに追加
    ConvertUtf8ToUtf16( ptr, TextMaxLength + 1, pStr );
}


//---------------------------------------------------------------------------
//
void SetUserWordList( ShowKeyboardArg* pOutShowKeyboardArg, const UserWord* pUserWord, const int32_t userWordNum ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutShowKeyboardArg != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    // 辞書登録をやめるようなパラメータを渡された場合の対応
    if( pUserWord == nullptr || userWordNum <= 0 )
    {
        pOutShowKeyboardArg->keyboardConfig._userDictionaryNum = 0;
        return;
    }

    // 最大値より大きな単語数を指定していたら、エラーを返す
    NN_SDK_ASSERT( ( userWordNum >= 0 && userWordNum <= UserWordMax ),
        "[swkbd] user word size is out of range.\n" );

    CopyUserDictionaryInfo( pOutShowKeyboardArg, userWordNum );

    // ユーザー辞書用データを workBuf に挿入する
    // GetRequiredWorkBufferSize の最大バイト数 (c_FixedStringUtf8BufSize) 以降に挿入を想定
    void* ptr = AddOffset( pOutShowKeyboardArg->workBuf, pOutShowKeyboardArg->keyboardConfig._userDictionaryOffset );
    std::memcpy( ptr, pUserWord, sizeof( UserWord ) * userWordNum );
}

//---------------------------------------------------------------------------
//
void SetTextCheckCallback( ShowKeyboardArg* pOutShowKeyboardArg, TextCheckCallback pCallback ) NN_NOEXCEPT
{
    // キーボード用パラメータは必須なので、なければエラーを返す
    NN_SDK_ASSERT( ( pOutShowKeyboardArg != nullptr ), "[swkbd] KeyboardArg must not be null.\n" );

    if( pCallback == nullptr )
    {
        pOutShowKeyboardArg->keyboardConfig._isUseTextCheck = false;
        return;
    }

    pOutShowKeyboardArg->keyboardConfig._textCheckCallback = pCallback;
    pOutShowKeyboardArg->keyboardConfig._isUseTextCheck = true;
}

}} // namespace nn::swkbd

