﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nn {
namespace stdfu {

/*
The DFU prefix placed as a header file is the first part read by the software application, used
to retrieve the file context, and enable valid DFU files to be recognized. The Prefix buffer is
represented in Big Endian order.

- The szSignature field, five-byte coded, presents the file identifier that enables valid
  DFU files to be recognized, and incompatible changes detected. This identifier should
  be updated when major changes are made to the file format. This field is set to
  “DfuSe”.
- The bVersion field, one-byte coded, presents the DFU format revision, The value will be
  incremented if extra fields are added to one of the three sections. Software exploring
  the file can either treat the file depending on its specified revision, or just test for valid
  value.
- The DFUImageSize field, four-byte coded, presents the total DFU file length in bytes.
- the bTargets field, one-byte coded, presents the number of DFU image stored in the
  file.
*/
struct DfuPrefix
{
    uint8_t szSignature[5];
    uint8_t bVersion;
    uint8_t DFUImageSize[4];
    uint8_t bTargets;
};


/*
The DFU Suffix, as specified in the DFU specification, allows the host software to detect and
prevent attempts to download incompatible firmware. The Suffix buffer is represented in
Little Endian order.

- The bcdDevice field gives the firmware version contained in the file, or 0xFFFF if
  ignored.
- The idProduct and idVendor field give the Product ID and Vendor ID respectively of the
  device that the file is intended for, or 0xFFFF if the field is ignored.
- The bcdDFU field, fixed to 0x011A, gives the DFU specification number. This value
  differs from that specified in standard DFU rev1.1.
- The ucSignature field contains a fixed string of three unsigned characters (44h, 46h,
  55h). In the file they appear in reverse order, allowing valid DFU files to be recognized.
- The bLength field, currently fixed to 16, gives the length of the DFU Suffix itself in bytes.
- The dwCRC (Cyclic Redundancy Check) field is the CRC calculated over the whole file
  except for the dwCRC data itself.
*/
struct DfuSuffix
{
    uint8_t bcdDeviceLo;
    uint8_t bcdDeviceHi;
    uint8_t idProductLo;
    uint8_t idProductHi;
    uint8_t idVendorLo;
    uint8_t idVendorHi;
    uint8_t bcdDFULo;
    uint8_t bcdDFUHi;
    uint8_t ucDfuSignature[3];
    uint8_t bLength;
    uint8_t dwCRC[4];
};


/*
The target prefix record is used to describe the associated image. The Target Prefix buffer is
represented in Big Endian order.

- The szSignature field, 6-byte coded, fixed to “Target”.
- The bAlternateSetting field gives the device alternate setting for which the associated
  image can be used.
- The bTargetNamed field is a boolean value which indicates if the target is named or
  not.
- The szTargetName field gives the target name.
- The dwTargetSize field gives the whole length of the associated image excluding the
  Target prefix.
- The dwNbElements field gives the number of elements in the associated image.
*/
struct DfuTargetPrefix
{
    uint8_t szSignature[6];
    uint8_t bAlternatesetting;
    uint8_t bTargetNamed[4];
    uint8_t szTargetName[255];
    uint8_t dwTargetSize[4];
    uint8_t dwNbElements[4];
};


/*
The Image element structured as follows, provides a data record containing the effective
firmware data preceded by the data address and data size. The Image Element buffer is
represented in Big Endian order.

- The dwElementAddress field gives the 4-byte starting address of the data.
- The dwElementSize field gives the size of the contained data.
- The Data field present the effective data and immedately follows DfuImageElement.
*/
struct DfuImageElement
{
    uint8_t dwElementAddres[4];
    uint8_t dwElementSize[4];
};


class DfuFile
{
public:

    /**
     * @brief Initialize the DfuFile binary
     *
     * @param[in] pBinary
     *   Pointer to DFU binary
     *
     * @param[in] totalBytes
     *   Total number of bytes for DFU binary
     *
     * @param[out] pOutVid
     *   Pointer to store vender ID, the DFU binary intends to update device with this vendor ID
     *
     * @param[out] pOutPid
     *   Pointer to store product ID, the DFU binary intends to update device with this device ID
     *
     * @param[out] pOutBcd
     *   Pointer to store version number, the DFU binary intends to update device with this version number
     *
     * @param[out] pOutImageCount
     *   Pointer to store number of images inside the DFU binary
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     *   @handleresult{nn::usb::ResultInvalidDfuPrefixSignature, the DFU binary has invalid DFU prefix signature}
     *   @handleresult{nn::usb::ResultInvalidDfuPrefixVersion, the DFU binary has invalid DFU prefix version number}
     *   @handleresult{nn::usb::ResultInvalidDfuPrefixImageSize, the DFU binary has invalid DFU image size}
     *   @handleresult{nn::usb::ResultInvalidDfuPrefixTargets, the DFU binary has no targets}
     *   @handleresult{nn::usb::ResultInvalidDfuSuffixSignature, the DFU binary has invalid DFU suffix signature}
     *   @handleresult{nn::usb::ResultInvalidDfuSuffixCrc, the DFU binary has invalid CRC}
     * @endretresult
     *
     * @pre
     *   None
     *
     * @post
     *   The DFU binary header and suffix is verfied
     *
     * @details
     *   This call verifies the DFU binary prefix, suffix and CRC. It also stores the inteded Vendor ID, Product ID, version, and number of images to caller specified location.
     */
    Result Initialize(void *pBinary, uint32_t totalBytes, uint16_t *pOutVid, uint16_t *pOutPid, uint16_t *pOutBcd, uint8_t *pOutImageCount) NN_NOEXCEPT;

    /**
     * @brief Retreive specified DFU image from DfuFile
     *
     * @param[out] pOutDfuTargetPrefix
     *   Pointer to store DfuTargetPrefix for the specified image
     *
     * @param[out] pOutAlternateSetting
     *   Pointer to store alternate setting for the specified image
     *
     * @param[out] pOutTargetName
     *   Pointer to store zero termainated string for the specified image
     *
     * @param[out] pOutElementsCount
     *   Pointer to store number of elements for the specified image
     *
     * @param[in] imageIndex
     *   Index of image to retreive from DfuFile
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     *   @handleresult{nn::usb::ResultInvalidDfuImageSignature, the specified image has invalid signature}
     *   @handleresult{nn::usb::ResultDfuImageNotFound, the specified image is not found in the DfuFile}
     * @endretresult
     *
     * @pre
     *   The DfuFile is initialized
     *
     * @post
     *   The target prefix, alternate setting, target name, and number of elemants for the image has been stored to caller provided storage.
     *
     * @details
     *   This call retreives specified image from the DfuFile. It also stores the target prefix, alternate setting, target name, and number of elemants to caller specified location.
     */
    Result GetImage(DfuTargetPrefix **pOutDfuTargetPrefix, uint8_t *pOutAlternateSetting, uint8_t **pOutTargetName, uint32_t *pOutElementsCount, uint8_t imageIndex) NN_NOEXCEPT;

    /**
     * @brief Retreive specified element from DFU image
     *
     * @param[out] pOutAddress
     *   Address of DFU element
     *
     * @param[out] pOutSize
     *   Size of DFU element
     *
     * @param[out] pOutData
     *   Pointer to DFU element data
     *
     * @param[in] iElement
     *   Index of DFU element to retreive
     *
     * @param[in] pDfuTargetPrefix
     *   Pointer to DfuTargetPrefix for the DFU image
     *
     * @retresult
     *   @handleresult{nn::ResultSuccess, Initialize succeeded}
     *   @handleresult{nn::usb::ResultDfuElementNotFound, the specified element is not found in the DFU image}
     * @endretresult
     *
     * @pre
     *   The DfuFile is initialized
     *
     * @post
     *   The element address, size in bytes, location of element data for the element has been stored to caller provided storage.
     *
     * @details
     *   This call retreives specified image from the DfuFile. It also stores the element address, size in bytes, location of element data to caller specified location.
     */
    Result GetElement(uint32_t *pOutAddress, uint32_t *pOutSize, uint8_t **pOutData, uint32_t iElement, DfuTargetPrefix *pDfuTargetPrefix) NN_NOEXCEPT;

private:

    DfuPrefix  *m_pDfuPrefix;
    DfuSuffix  *m_pDfuSuffix;
    uint8_t    *m_pImage;

    Result ValidateDfuPrefix(uint32_t totalBytes) NN_NOEXCEPT;
    Result ValidateDfuSuffix() NN_NOEXCEPT;

    bool IsValidCrc() NN_NOEXCEPT;
    uint32_t ComputeCrc() NN_NOEXCEPT;
    uint32_t SwapBytes32(uint8_t *p) NN_NOEXCEPT;
    uint16_t SwapBytes16(uint8_t *p) NN_NOEXCEPT;

};


} // namespace stdfu
} // namespace nn
