﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>

#include <nn/stdfu/stdfu_Result.public.h>
#include <nn/stdfu/stdfu_DfuUpdateStatus.h>

#include "../detail/stdfu_Log.h"
#include "stdfu_DfuDevice.h"

namespace nn {
namespace stdfu {
//////////////////////////////////////////////////////////////////////////////
// Public
//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::PerformUpdate(uint16_t vid, uint16_t pid, uint16_t bcd, DfuFile *pDfuFile, uint32_t imageCount, DfuUpdateStatus *pDfuUpdateStatus) NN_NOEXCEPT
{
    Result result = m_UsbHost.Initialize();

    if (result.IsSuccess())
    {
        result = GetDfuDevice(vid, pid, bcd);

        if (result.IsSuccess())
        {
            result = m_UsbHostInterface.Initialize(&m_UsbHost, m_Handle);

            if (result.IsSuccess())
            {
                result = GetDfuDescriptor();

                if (result.IsSuccess())
                {
                    /*
                    NN_DETAIL_STDFU_INFO("DfuDescriptor\n");
                    NN_DETAIL_STDFU_INFO("    DownloadCapableBit:         %x\n",      (m_DfuDescriptor.bmAttributes & DfuAttributes::DfuAttributes_DownloadCapableBit) >> (DfuAttributes::DfuAttributes_DownloadCapableBit - 1));
                    NN_DETAIL_STDFU_INFO("    UploadCapableBit:           %x\n",      (m_DfuDescriptor.bmAttributes & DfuAttributes::DfuAttributes_UploadCapableBit) >> (DfuAttributes::DfuAttributes_UploadCapableBit - 1));
                    NN_DETAIL_STDFU_INFO("    ManifestationToerantBit:    %x\n",      (m_DfuDescriptor.bmAttributes & DfuAttributes::DfuAttributes_ManifestationToerantBit) >> (DfuAttributes::DfuAttributes_ManifestationToerantBit - 1));
                    NN_DETAIL_STDFU_INFO("    wDetachTimeout:             %dms\n",    m_DfuDescriptor.wDetachTimeout);
                    NN_DETAIL_STDFU_INFO("    wTransferSize:              %d\n",      m_DfuDescriptor.wTransferSize);
                    NN_DETAIL_STDFU_INFO("    bcdDfuVersion:              %04x\n",    m_DfuDescriptor.bcdDfuVersion);
                     */

                    NN_DETAIL_STDFU_INFO("Start FW update.\n");

                    for (uint32_t iImage = 0; iImage < imageCount; iImage++)
                    {
                        pDfuUpdateStatus->SetImage(iImage);

                        result = pDfuFile->GetImage(&m_pDfuTargetPrefix, &m_AlternateSetting, &m_pImageName, &m_ElementsCount, iImage);

                        if (result.IsSuccess())
                        {
//                            NN_DETAIL_STDFU_INFO("image: %d, m_AlternateSetting: %d, name: %s, elements: %d\n", iImage, m_AlternateSetting, m_pImageName, m_ElementsCount);

                            result = DownloadElementAddresses(pDfuFile, pDfuUpdateStatus);

                            if (result.IsSuccess())
                            {
                                result = DownloadElements(pDfuFile, pDfuUpdateStatus);

                                if (result.IsSuccess())
                                {
                                    result = VerifyElements(pDfuFile, pDfuUpdateStatus);
                                }
                            }
                        }

                        if (!result.IsSuccess())
                        {
                            break;
                        }
                    }

                    NN_DETAIL_STDFU_INFO("Finished FW update.\n");

                    EndDfuDownload();
                    m_UsbHostInterface.ResetDevice();
                }

                m_UsbHostInterface.Finalize();
            }
        }

        m_UsbHost.Finalize();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
// Private
//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::GetDfuDevice(uint16_t vid, uint16_t pid, uint16_t bcd) NN_NOEXCEPT
{
    NN_DETAIL_STDFU_INFO("Looking for DFU Device vid: %04x pid:%04x bcd: %04x\n", vid, pid, bcd);

    int32_t ifCount;
    nn::usb::DeviceFilter filter;
    nn::usb::InterfaceQueryOutput ifList;

    memset(&filter, 0, sizeof(filter));

    filter.matchFlags   = nn::usb::DeviceFilterMatchFlags_Vendor | nn::usb::DeviceFilterMatchFlags_Product | nn::usb::DeviceFilterMatchFlags_DeviceLo;
    filter.idVendor     = vid;
    filter.idProduct    = pid;
    filter.bcdDeviceLo  = bcd;

    Result result = m_UsbHost.QueryAvailableInterfaces(&ifCount, &ifList, sizeof(ifList), &filter);

    if (result.IsSuccess() && ifCount)
    {
        m_Handle = ifList.ifProfile.handle;
        NN_DETAIL_STDFU_WARN("DFU device found\n");
    }
    else
    {
        m_Handle = 0;
        NN_DETAIL_STDFU_WARN("DFU device not found\n");
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::GetDfuDescriptor() NN_NOEXCEPT
{
    Result result;
    size_t bytesTransferred;

    result = m_UsbHostInterface.ControlRequest(
                                                &bytesTransferred,
                                                m_TransferBuffer,
                                                0x80,
                                                0x06,
                                                0x0200,
                                                0x0000,
                                                sizeof(nn::usb::UsbConfigDescriptor)
                                                );

    if (result.IsSuccess() && (bytesTransferred == sizeof(nn::usb::UsbConfigDescriptor)))
    {
        nn::usb::UsbConfigDescriptor *pUsbConfigDescriptor = reinterpret_cast<nn::usb::UsbConfigDescriptor*>(m_TransferBuffer);

        if (pUsbConfigDescriptor->wTotalLength <= sizeof(m_TransferBuffer))
        {
            uint16_t totalSize = pUsbConfigDescriptor->wTotalLength;

            result = m_UsbHostInterface.ControlRequest(
                                                    &bytesTransferred,
                                                    m_TransferBuffer,
                                                    0x80,
                                                    0x06,
                                                    0x0200,
                                                    0x0000,
                                                    totalSize
                                                    );

            if (result.IsSuccess() && (bytesTransferred == totalSize))
            {
                result = ResultDfuDescriptorNotFound();
                uint8_t *p = m_TransferBuffer;

                while (totalSize)
                {
                    uint8_t bLength         = p[0];
                    uint8_t bDescriptorType = p[1];

                    if ((bLength == sizeof(DfuDescriptor)) && (bDescriptorType == 0x21))
                    {
                        result = ResultSuccess();
                        memcpy(&m_DfuDescriptor, p, sizeof(DfuDescriptor));
                        break;
                    }

                    p += bLength;
                    totalSize -= bLength;
                }
            }
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::DownloadElementAddresses(DfuFile *pDfuFile, DfuUpdateStatus *pDfuUpdateStatus) NN_NOEXCEPT
{
    Result result = PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuIdle, 100);

    if (result.IsSuccess())
    {
        for (uint32_t iElement = 0; iElement < m_ElementsCount; iElement++)
        {
            uint32_t elementAddress;
            uint32_t elementSize;
            uint8_t *pData;

            pDfuUpdateStatus->SetPhase(UpdatePhase::UpdatePhase_Address);
            pDfuUpdateStatus->SetElement(iElement);
            pDfuUpdateStatus->SetProgress(0);

            result = pDfuFile->GetElement(&elementAddress, &elementSize, &pData, iElement, m_pDfuTargetPrefix);

            if (result.IsSuccess())
            {
                NN_DETAIL_STDFU_INFO("element: %d, address: %08x, size: %d\n", iElement, elementAddress, elementSize);

                // Make sure state is DfuFidle or DfuDownloadIdle
                uint8_t state;
                result = GetState(&state);

                // TODO, check state

                uint8_t data[5];

                data[0] = 0x41; // ????
                memcpy(&data[1], &elementAddress, 4);

                result = Download(data, sizeof(data), 0);

                if (result.IsSuccess())
                {
                    PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuDownloadIdle, 100);
                }

                pDfuUpdateStatus->SetProgress(100);
            }

            if (!result.IsSuccess())
            {
                break;
            }
        }
    }

    Abort();
    PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuIdle, 100);

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::DownloadElements(DfuFile *pDfuFile, DfuUpdateStatus *pDfuUpdateStatus) NN_NOEXCEPT
{
    Result result = PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuIdle, 100);

    if (result.IsSuccess())
    {
        for (uint32_t iElement = 0; iElement < m_ElementsCount; iElement++)
        {
            uint32_t elementAddress;
            uint32_t elementSize;
            uint8_t *pData;

            pDfuUpdateStatus->SetPhase(UpdatePhase::UpdatePhase_Write);
            pDfuUpdateStatus->SetElement(iElement);
            pDfuUpdateStatus->SetProgress(0);

            result = pDfuFile->GetElement(&elementAddress, &elementSize, &pData, iElement, m_pDfuTargetPrefix);

            if (result.IsSuccess())
            {
                NN_DETAIL_STDFU_INFO("write element: %d, address: %08x, size: %d\n", iElement, elementAddress, elementSize);

                // Make sure state is DfuIdle or DfuDownloadIdle
                uint8_t state;
                result = GetState(&state);

                if ((state != DfuStateCode::DfuStateCode_DfuIdle) && (state != DfuStateCode::DfuStateCode_DfuDownloadIdle))
                {
                    return ResultInvalidDfuState();
                }

                uint8_t data[5];

                data[0] = 0x21; // ????
                memcpy(&data[1], &elementAddress, 4);

                result = Download(data, sizeof(data), 0);

                if (result.IsSuccess())
                {
                    result = PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuDownloadIdle, 100);
                }

                if (result.IsSuccess())
                {
                    // Compute blocks
                    int blocks = (elementSize / m_DfuDescriptor.wTransferSize) + 1;

                    if (elementSize % m_DfuDescriptor.wTransferSize)
                    {
                        blocks++;
                    }

                    // Download data
                    uint16_t iBlock = 2;    // for some reason blocks start at 2 :0

                    while (elementSize)
                    {
                        uint16_t bytes;

                        if (elementSize >= m_DfuDescriptor.wTransferSize)
                        {
                            bytes = m_DfuDescriptor.wTransferSize;
                        }
                        else
                        {
                            bytes = elementSize;
                        }

                        pDfuUpdateStatus->SetProgress((iBlock * 100) / blocks);

//                        NN_DETAIL_STDFU_INFO("Block %d / %d\n", iBlock, blocks);
                        result = Download(pData, bytes, iBlock);

                        if (result.IsSuccess())
                        {
                            elementSize -= bytes;
                            pData += bytes;
                            iBlock++;

                            result = PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuDownloadIdle, 100);
                        }

                        if (!result.IsSuccess())
                        {
                            break;
                        }
                    }
                }
            }

            if (!result.IsSuccess())
            {
                break;
            }


        }
    }

    Abort();
    PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuIdle, 100);

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::VerifyElements(DfuFile *pDfuFile, DfuUpdateStatus *pDfuUpdateStatus) NN_NOEXCEPT
{
    Result result = PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuIdle, 100);

    if (result.IsSuccess())
    {
        for (uint32_t iElement = 0; iElement < m_ElementsCount; iElement++)
        {
            uint32_t elementAddress;
            uint32_t elementSize;
            uint8_t *pData;

            pDfuUpdateStatus->SetPhase(UpdatePhase::UpdatePhase_Verify);
            pDfuUpdateStatus->SetElement(iElement);
            pDfuUpdateStatus->SetProgress(0);

            result = pDfuFile->GetElement(&elementAddress, &elementSize, &pData, iElement, m_pDfuTargetPrefix);

            if (result.IsSuccess())
            {
                NN_DETAIL_STDFU_INFO("verify element: %d, address: %08x, size: %d\n", iElement, elementAddress, elementSize);

                // Make sure state is DfuIdle or DfuDownloadIdle
                uint8_t state;
                result = GetState(&state);

                if ((state != DfuStateCode::DfuStateCode_DfuIdle) && (state != DfuStateCode::DfuStateCode_DfuUploadIdle))
                {
                    return ResultInvalidDfuState();
                }

                uint8_t data[5];

                data[0] = 0x21; // ????
                memcpy(&data[1], &elementAddress, 4);

                result = Download(data, sizeof(data), 0);

                if (result.IsSuccess())
                {
                    result = PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuDownloadIdle, 100);
                }

                if (result.IsSuccess())
                {
                    result = Abort();

                    if (result.IsSuccess())
                    {
                        // Compute blocks
                        int blocks = (elementSize / m_DfuDescriptor.wTransferSize) + 1;

                        if (elementSize % m_DfuDescriptor.wTransferSize)
                        {
                            blocks++;
                        }

                        // Upload data
                        uint16_t iBlock = 2;    // for some reason blocks start at 2 :0

                        while (elementSize)
                        {
                            uint16_t bytes;

                            if (elementSize >= m_DfuDescriptor.wTransferSize)
                            {
                                bytes = m_DfuDescriptor.wTransferSize;
                            }
                            else
                            {
                                bytes = elementSize;
                            }

                            pDfuUpdateStatus->SetProgress((iBlock * 100) / blocks);

//                            NN_DETAIL_STDFU_INFO("Block %d / %d\n", iBlock, blocks);
                            result = Upload(m_TransferBuffer, bytes, iBlock);

                            if (result.IsSuccess())
                            {
                                // Verify bits here
                                if (memcmp(m_TransferBuffer, pData, bytes))
                                {
                                    result = ResultVerificationFailed();
                                }

                                if (result.IsSuccess())
                                {
                                    elementSize -= bytes;
                                    pData += bytes;
                                    iBlock++;

                                    result = PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuUploadIdle, 100);
                                }
                            }

                            if (!result.IsSuccess())
                            {
                                break;
                            }
                        }

                        if (result.IsSuccess())
                        {
                            result = PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuUploadIdle, 100);

                            if (result.IsSuccess())
                            {
                                result = Abort();
                            }
                        }
                    }
                }
            }

            if (!result.IsSuccess())
            {
                break;
            }
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::EndDfuDownload() NN_NOEXCEPT
{
    Result result = PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuIdle, 100);

    if (result.IsSuccess())
    {
        result = Download(NULL, 0, 0);

        if (result.IsSuccess())
        {
            result = PollForStatus(DfuStatusCode::DfuStatusCode_Ok, DfuStateCode::DfuStateCode_DfuManifest, 100);
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::PollForStatus(uint8_t bStatus, uint8_t bState, uint32_t retries) NN_NOEXCEPT
{
    Result result = ResultDfuStatusTimeout();

    while (retries--)   // Might want to implement retires and return error
    {
        DfuStatus status;
        Result result = GetStaus(&status);

        if (result.IsSuccess())
        {
            if ((status.bStatus == bStatus) && (status.bState == bState))
            {
                return ResultSuccess();
            }
        }
        else
        {
            break;
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::GetStaus(DfuStatus *pDfuStatus) NN_NOEXCEPT
{
    return ControlRequest(
                            pDfuStatus,                         // pData
                            0xa1,                               // bmRequestType
                            0x03,                               // bRequest
                            0,                                  // wValue
                            0,                                  // wIndex
                            sizeof(DfuStatus)                   // wLength
                            );
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::GetState(uint8_t *pState) NN_NOEXCEPT
{
    return ControlRequest(
                            pState,                             // pData
                            0xa1,                               // bmRequestType
                            0x05,                               // bRequest
                            0,                                  // wValue
                            0,                                  // wIndex
                            sizeof(uint8_t)                     // wLength
                            );
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::Download(uint8_t *pData, uint32_t bytes, uint16_t blockNumber) NN_NOEXCEPT
{
    return ControlRequest(
                            pData,                              // pData
                            0x21,                               // bmRequestType
                            0x01,                               // bRequest
                            blockNumber,                        // wValue
                            0,                                  // wIndex
                            bytes                               // wLength
                            );
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::Upload(uint8_t *pData, uint32_t bytes, uint16_t blockNumber) NN_NOEXCEPT
{
    return ControlRequest(
                            pData,                              // pData
                            0xa1,                               // bmRequestType
                            0x02,                               // bRequest
                            blockNumber,                        // wValue
                            0,                                  // wIndex
                            bytes                               // wLength
                            );
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::Abort() NN_NOEXCEPT
{
    return ControlRequest(
                            NULL,                               // pData
                            0x21,                               // bmRequestType
                            0x06,                               // bRequest
                            0,                                  // wValue
                            0,                                  // wIndex
                            0                                   // wLength
                            );
}


//////////////////////////////////////////////////////////////////////////////
Result DfuDevice::ControlRequest(
                                    void      *pData,
                                    uint8_t    bmRequestType,
                                    uint8_t    bRequest,
                                    uint16_t   wValue,
                                    uint16_t   wIndex,
                                    uint16_t   wLength
                                    ) NN_NOEXCEPT
{
    Result result;
    size_t bytesTransferred;

    // For OUT requests, copy data to transaction buffer
    if ((bmRequestType & 0x80) == 0)
    {
        if (pData && wLength)
        {
            if (pData != m_TransferBuffer)
            {
                memcpy(m_TransferBuffer, pData, wLength);
            }
        }
    }

    result = m_UsbHostInterface.ControlRequest(
                                                &bytesTransferred,
                                                m_TransferBuffer,
                                                bmRequestType,
                                                bRequest,
                                                wValue,
                                                wIndex,
                                                wLength
                                                );

    if (result.IsSuccess())
    {
        // For IN requests, copy data from transaction buffer
        if (bmRequestType & 0x80)
        {
            if (pData && (wLength == bytesTransferred))
            {
                if (pData != m_TransferBuffer)
                {
                    memcpy(pData, m_TransferBuffer, wLength);
                }
            }
        }
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
} // namespace stdfu
} // namespace nn
