﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nn/ssl/detail/ssl_Common.h>

#include "server/ssl_MemoryManager.h"
#include "detail/ssl_ServiceSession.h"
#include "server/ssl_Util.h"
#include "detail/ssl_ApiBase.h"

using namespace nn::ssl::detail;

namespace nn { namespace ssl {

namespace
{
    std::aligned_storage<BuildParameter_MemoryManagerHeapSizeHorizonLibrary>::type  g_HeapBuf;

// ------------------------------------------------------------------------------------------------
// class SslApi
// ------------------------------------------------------------------------------------------------
    class SslApi : public SslApiBase
    {
    public:
        nn::Result Initialize() NN_NOEXCEPT;
        nn::Result Initialize(uint32_t concurLimit) NN_NOEXCEPT;
        nn::Result Finalize() NN_NOEXCEPT;
    };

    nn::Result SslApi::Initialize() NN_NOEXCEPT
    {
        return SslApi::Initialize(nn::ssl::DefaultConcurrencyLimit);
    }

    nn::Result SslApi::Initialize(uint32_t concurLimit) NN_NOEXCEPT
    {
        nn::Result                  ret = nn::ResultSuccess();

        do
        {
            //  Check manupilate reference counter and returns here if the library is already
            //  initialized
            ret = SslApiBase::Initialize();
            if (nn::ssl::ResultLibraryAlreadyInitialized::Includes(ret))
            {
                return nn::ssl::ResultLibraryAlreadyInitialized();
            }

            //  Init memory management
            SslMemoryManager::Initialize(
                reinterpret_cast<char*>(&g_HeapBuf),
                BuildParameter_MemoryManagerHeapSizeHorizonLibrary);

            //  Init our connection to the service
            ret = ServiceSession::Initialize(concurLimit);
            if (ret.IsFailure())
            {
                NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to init service session\n");
                break;
            }
        } while (NN_STATIC_CONDITION(false));

        return ret;
    }

    nn::Result SslApi::Finalize() NN_NOEXCEPT
    {
        nn::Result                  ret = nn::ResultSuccess();

        do
        {
            //  Manupilate reference counter and returns here if the library is not ready
            //  to finalize, proceed only when there's no reference anymore
            ret = SslApiBase::Finalize();
            if (ret.IsFailure())
            {
                if (nn::ssl::ResultLibraryNotInitialized::Includes(ret))
                {
                    break;
                }

                if (nn::ssl::ResultInternalReferenceRemains::Includes(ret))
                {
                    //  Someone is still referencing the library, just return success
                    ret = nn::ResultSuccess();
                    break;
                }
            }


            if (!ServiceSession::IsInitialized())
            {
                ret = ResultLibraryNotInitialized();
                break;
            }

            ServiceSession::Finalize();

            ret = SslMemoryManager::Finalize();
            if (ret.IsFailure())
            {
                NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to finalize memory mgmt\n");
                break;
            }

            NN_DETAIL_SSL_DBG_PRINT("[ssl] Finalized library.\n");
        } while (NN_STATIC_CONDITION(false));

        return ret;
    }

    SslApi g_SslLibrary;
}

// ------------------------------------------------------------------------------------------------
// Public interface
// ------------------------------------------------------------------------------------------------
nn::Result Initialize() NN_NOEXCEPT
{
    return g_SslLibrary.Initialize(nn::ssl::DefaultConcurrencyLimit);
}

nn::Result Initialize(uint32_t concurLimit) NN_NOEXCEPT
{
    return g_SslLibrary.Initialize(concurLimit);
}

nn::Result Finalize() NN_NOEXCEPT
{
    return g_SslLibrary.Finalize();
}

nn::Result GetSslResultFromValue(
    nn::Result* pOutResult,
    const char* pInValue,
    uint32_t inValueSize) NN_NOEXCEPT
{
    return SslApiBase::GetSslResultFromValue(pOutResult, pInValue, inValueSize);
}

}}
