﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "ssl_SystemDataReader.h"
#include "ssl_Util.h"

#include "prprf.h"    //  For PR_snprintf


namespace nn { namespace ssl { namespace detail {




const char                      *SystemDataReader::g_TrustedCertMountPoint = "ssltc";
const char                      *SystemDataReader::g_TcfName = "ssl_TrustedCerts.bdf";
const char                      *SystemDataReader::g_CrlName = "ssl_Crl.bdf";
const char                      *SystemDataReader::g_TestCrlName = "ssl_TestNtdServer1Crl.bdf";
const nn::ncm::SystemDataId     SystemDataReader::g_SslTrustedCertManagerId = { 0x0100000000000800UL };
uint8_t                         *SystemDataReader::g_pCacheBuf = nullptr;


bool SystemDataReader::IsInitialized()
{
    //  If we have a cache buffer, then the SystemData area was mounted
    //  (this is intrinsic to our Initialize() method).
    return (SystemDataReader::g_pCacheBuf != nullptr);
}


SystemDataReader::SystemDataReader(BdfReader::BdfId id) : m_id(id)
{
    m_File.handle = nullptr;
}


SystemDataReader::~SystemDataReader()
{
    if (m_File.handle != nullptr)
    {
        nn::fs::CloseFile(m_File);
        m_File.handle = nullptr;
    }
}


nn::Result SystemDataReader::Initialize()
{
    nn::Result                  result = ResultSuccess();
    size_t                      cacheSize;
    uint8_t                     *pCacheBuf = nullptr;

    do
    {
        if (SystemDataReader::g_pCacheBuf != nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SystemDataReader::Initialize] Already initialized!\n");
            break;
        }

        result = nn::fs::QueryMountSystemDataCacheSize(&cacheSize,
                                                       SystemDataReader::g_SslTrustedCertManagerId);
        if (!result.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SystemDataReader::Initialize] ERROR: failed to get needed cache size\n");
            break;
        }

        pCacheBuf = new uint8_t[cacheSize];
        if (pCacheBuf == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SystemDataReader::Initialize] ERROR: failed to create cache buf\n");
            result = ResultInsufficientMemory();
            break;
        }

        //  Mount the system save data area containing the TCF
        result = nn::fs::MountSystemData(SystemDataReader::g_TrustedCertMountPoint,
                                         SystemDataReader::g_SslTrustedCertManagerId,
                                         pCacheBuf,
                                         cacheSize);
        if (!result.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SystemDataReader::Initialize] ERROR: failed to mount trusted cert store: %d-%d\n",
                                    result.GetModule(),
                                    result.GetDescription());
            break;
        }

        //  Retain the cache buffer in our static data so it can be released later
        //  during Finalize()
        SystemDataReader::g_pCacheBuf = pCacheBuf;
        pCacheBuf = nullptr;
    } while (NN_STATIC_CONDITION(false));

    if (pCacheBuf != nullptr)
    {
        delete[] pCacheBuf;
        pCacheBuf = nullptr;
    }

    return result;
}


nn::Result SystemDataReader::Finalize()
{
    nn::Result                  result = ResultSuccess();

    do
    {
        if (SystemDataReader::g_pCacheBuf == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SystemDataReader::Finalize] Cache buf not present, not initialized!\n");
            result = ResultLibraryNotInitialized();
            break;
        }

        //  Unmount the save data area *first* so the cache buffer is no longer
        //  being used.  Then destroy our instance.
        nn::fs::Unmount(g_TrustedCertMountPoint);

        delete[] SystemDataReader::g_pCacheBuf;
        SystemDataReader::g_pCacheBuf = nullptr;
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result SystemDataReader::Open() NN_NOEXCEPT
{
    nn::Result                  result = ResultSuccess();
    uint32_t                    charsWritten;
    uint32_t                    expected;
    char                        scratchBuf[SystemDataReader::g_ScratchBufSize];
    const char                  *pFilename = nullptr;

    do
    {
        //  Double check that we are initialized first, then try to
        //  open the file.
        if (!SystemDataReader::IsInitialized())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SystemDataReader::Open] not initialized!\n");
            result = ResultLibraryNotInitialized();
            break;
        }

        switch (m_id)
        {
            case BdfReader::BdfId_TrustedCerts:
                pFilename = SystemDataReader::g_TcfName;
                break;

            case BdfReader::BdfId_Crl:
                pFilename = SystemDataReader::g_CrlName;
                break;

            case BdfReader::BdfId_TestCrl:
                pFilename = SystemDataReader::g_TestCrlName;
                break;

            default:
                NN_DETAIL_SSL_DBG_PRINT("[SystemDataReader::Open] invalid id: %8.8X\n", m_id);
                result = ResultErrorLower();
                break;
        }

        if (pFilename == nullptr)
        {
            break;
        }

        //  The expected name is the mount point, directory separator, TCF name
        //  and a null terminator.
        expected =
            static_cast<uint32_t>(strlen(SystemDataReader::g_TrustedCertMountPoint) +
                                  strlen(pFilename) +
                                  2);

        charsWritten = PR_snprintf(scratchBuf,
                                   sizeof(scratchBuf),
                                   "%s:/%s",
                                   SystemDataReader::g_TrustedCertMountPoint,
                                   pFilename);
        if (charsWritten < expected)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SystemDataReader::Open] failed to gen name (%u, %u)\n",
                                    charsWritten,
                                    expected);
            result = ResultInsufficientMemory();
            break;
        }

        //  Try to open the TCF for processing
        result = nn::fs::OpenFile(&m_File, scratchBuf, nn::fs::OpenMode_Read);
        if (!result.IsSuccess())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SystemDataReader::Open] ERROR failed to open BDF id %8.8X: \'%s\', %d-%d\n",
                                    m_id,
                                    scratchBuf,
                                    result.GetModule(),
                                    result.GetDescription());
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}


nn::Result SystemDataReader::Read(size_t   *pOutBytesRead,
                                  int64_t  offset,
                                  uint8_t  *pOutBuf,
                                  size_t   size) NN_NOEXCEPT
{
    nn::Result                  result = ResultSuccess();

    do
    {
        if (!SystemDataReader::IsInitialized())
        {
            NN_DETAIL_SSL_DBG_PRINT("[SystemDataReader::Read] not initialized!\n");
            result = ResultLibraryNotInitialized();
            break;
        }

        //  This is just a simple wrapper around nn::fs::ReadFile
        result = nn::fs::ReadFile(pOutBytesRead,
                                  m_File,
                                  offset,
                                  pOutBuf,
                                  size);
    } while (NN_STATIC_CONDITION(false));

    return result;
}

} } }    //  nn::ssl::detail
