﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ssl/ssl_Types.h>
#include <nn/ssl/detail/ssl_Build.h>

#include "server/ssl_CertStore.h"
#include "server/ssl_InternalPki.h"
#include "server/ssl_Util.h"
#include "server/ssl_NssUtil.h"
#include "server/ssl_NssConfigurator.h"
#include "server/ssl_NssPkcsUtil.h"
#include "server/ssl_NssCommon.h"

namespace nn { namespace ssl { namespace detail {

// ------------------------------------------------------------------------------------------------
//
// Pk12ModuleUtil
//
// ------------------------------------------------------------------------------------------------
SECItem* Pk12ModuleUtil::PrepPassword(const char *pInPasswordData, uint32_t passwordDataSize)
{
    SECItem* pScratchItem        = nullptr;
    SECItem* pRetItem            = nullptr;
    uint32_t unicodePasswordSize = 0;

    do {
        pScratchItem = SECITEM_AllocItem(nullptr, nullptr, passwordDataSize + 1);
        if(pScratchItem == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(PrepPassword) failed to alloc pScratchItem buf\n");
            break;
        }

        memset(pScratchItem->data, 0, pScratchItem->len);
        memcpy(pScratchItem->data, pInPasswordData, passwordDataSize);
        unicodePasswordSize = pScratchItem->len << 2;

        pRetItem = SECITEM_AllocItem(nullptr, nullptr, unicodePasswordSize);
        if (pRetItem == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(PrepPassword) failed to alloc pRetItem buf\n");
            break;
        }
        memset(pRetItem->data, 0, unicodePasswordSize);

        if (PORT_UCS2_ASCIIConversion(
            PR_TRUE,
            pScratchItem->data,
            pScratchItem->len,
            pRetItem->data,
            unicodePasswordSize,
            &pRetItem->len,
            PR_TRUE) == PR_FALSE)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(PrepPassword) failed to convert pw data\n");
            SECITEM_ZfreeItem(pRetItem, PR_TRUE);
            pRetItem = nullptr;
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    if(pScratchItem != nullptr)
    {
        SECITEM_ZfreeItem(pScratchItem, PR_TRUE);
    }

    return pRetItem;
}

SECStatus Pk12ModuleUtil::ProcessP12Data(PK11SlotInfo      *pInPk11Slot,
                                         SECItem           *pInPassword,
                                         SECItem           *p12File,
                                         char              *pInOutNickname,
                                         uint32_t          maxNicknameLen,
                                         CERTCertificate   **pOutCert,
                                         SECKEYPrivateKey  **pOutPrivKey)
{
    SECStatus                 status = SECFailure;
    PRBool                    isAgain;
    SEC_PKCS12DecoderContext* p12DecCtx = nullptr;
    SECItem                   unicodePassword;
    SECMODListLock            *lock = nullptr;
    char                      *pTmpNick = nullptr;
    Pk12RenameDetails         renameDetails;

    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) started for slot %p, nickname %s\n",
                                      pInPk11Slot,
                                      pInOutNickname);

    renameDetails.pDerCert = nullptr;
    renameDetails.pNickname = nullptr;

    unicodePassword.data = pInPassword->data;
    unicodePassword.len = pInPassword->len;

    do {
        lock = SECMOD_GetDefaultModuleListLock();
        SECMOD_GetWriteLock(lock);

        do {
            isAgain = PR_FALSE;
            status = SECFailure;
            p12DecCtx = SEC_PKCS12DecoderStart(&unicodePassword,
                                               pInPk11Slot,
                                               nullptr,
                                               nullptr,
                                               nullptr,
                                               nullptr,
                                               nullptr,
                                               nullptr);
            if (p12DecCtx == nullptr)
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) failed to start P12 decoder\n");
                break;
            }

            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) decoder started\n");
            status = SEC_PKCS12DecoderUpdate(p12DecCtx, p12File->data, p12File->len);
            if (status != SECSuccess)
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) failed to decode\n");
                break;
            }

            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) verifying decoder\n");
            status = SEC_PKCS12DecoderVerify(p12DecCtx);
            if (status != SECSuccess)
            {
                // If the length of the unicode password is 2, try it again
                // with a 0 length password.
                if (unicodePassword.len == 2)
                {
                    unicodePassword.len = 0;
                    isAgain = PR_TRUE;
                    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) try 0 len pInPassword\n");

                    // Finish the current decoder context so we can start again
                    SEC_PKCS12DecoderFinish(p12DecCtx);
                    p12DecCtx = NULL;
                }
                else
                {
                    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) decode not verified\n");
                    break;
                }
            }
        } while (isAgain == PR_TRUE);

        if (status != SECSuccess)
        {
            break;
        }

        //  Get the DER cert data for the client's certificate.  This will allow
        //  us to match it when we rename later while letting others (such as
        //  a CA chain) pass through.
        renameDetails.pDerCert =
            Pk12ModuleUtil::FindDERCertForPrivKey(p12DecCtx);
        if (renameDetails.pDerCert == nullptr)
        {
             NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) bundle contains no private key\n");
             status = SECFailure;
             break;
        }

        //  Use the "rename cert nicknames" to override the nickname which is
        //  present in the PKCS#12 bundle.  Our caller MUST ensure that the
        //  nickname provided is UNIQUE.
        //
        //  It would have been better to do this in CallbackP12NicknameCollision, but
        //  NSS passes the cert as the argument to the callback rather than a pointer
        //  of our choosing.  So we let it pick the nickname first, then override it.
        //  Additionally, this call has the side effect of altering the nickname
        //  of any associated private key and must be done *before* validation is
        //  done.
        renameDetails.pNickname = pInOutNickname;
        status = SEC_PKCS12DecoderRenameCertNicknames(p12DecCtx,
                                                      Pk12ModuleUtil::CallbackP12NicknameRename,
                                                      reinterpret_cast<void *>(&renameDetails));
        if (status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) failed to rename nickname: %d\n", PORT_GetError());
            break;
        }

        // Do a deeper validation of the P12 data, making sure nicknames are ok, etc.
        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) validate p12 bags\n");
        status =
            SEC_PKCS12DecoderValidateBags(p12DecCtx,
                                          Pk12ModuleUtil::CallbackP12NicknameCollision);
        if (status != SECSuccess)
        {
            int err = PORT_GetError();
            if (err != SEC_ERROR_PKCS12_DUPLICATE_DATA)
            {
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) decode validate failed: %d\n", err);
                NN_UNUSED(err);
                break;
            }
        }

        //  At this point the PKCS#12 bundle has been decoded, verified and
        //  validated.  Before we go on, we need to check if this
        //  cert/key is already installed in our cert/key store.  If it is,
        //  then get that existing cert, key and nickname from the store
        //  and return it.  If it is NOT, then import it.
        status = Pk12ModuleUtil::FindExistingPrivKeyAndCert(pInPk11Slot,
                                                            renameDetails.pDerCert,
                                                            pOutCert,
                                                            pOutPrivKey);
        if (status == SECSuccess)
        {
            //  We found a matching cert/key in the slot, so no need to
            //  finish this by doing an install.  However, we do need to
            //  extract the nickname from the provided cert so the caller
            //  gets the existing nickname.
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) cert/key already installed, return existing\n");
            pTmpNick = CertStore::ExtractNickname(*pOutCert);
            if (pTmpNick == nullptr)
            {
                //  This should *NEVER* happen, but do not fail if it does.
                //  The failures will come later when something tries to
                //  destroy a cert from the store with the name passed
                //  into this method; but, those can be safely ignored.
                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) WARNING no nickname for existing cert %p!\n", *pOutCert);
            }
            else
            {
                //  Copy out the nickname, as long as it fits.  If it
                //  does not then return a ResultErrorLower as there is
                //  no good way for us to convey this.
                uint32_t    nickLen = PL_strlen(pTmpNick) + 1;
                if (nickLen > maxNicknameLen)
                {
                    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) ERROR: not enough room for existing nickname (need \'%s\', %u; have \'%s\', %u)\n",
                                                      pTmpNick,
                                                      nickLen,
                                                      pInOutNickname,
                                                      maxNicknameLen);
                    status = SECFailure;
                    CERT_DestroyCertificate(*pOutCert);
                    SECKEY_DestroyPrivateKey(*pOutPrivKey);
                    *pOutCert = nullptr;
                    *pOutPrivKey = nullptr;
                    break;
                }

                NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) existig cert/key has nickname \'%s\'\n",
                                                  pTmpNick);
                memset(pInOutNickname, 0, maxNicknameLen);
                PL_strncpy(pInOutNickname, pTmpNick, nickLen);
            }

            //  Status is already success, so just break out
            break;
        }

        // The cert/key are not already installed, so import them
        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) import p12 bags\n");
        status = SEC_PKCS12DecoderImportBags(p12DecCtx);
        if (status != SECSuccess)
        {
            int err = PORT_GetError();
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) failed to import: %d\n", err);
            NN_UNUSED(err);
            break;
        }

        //  Get the certificate and key objects for this import so we can
        //  provide them to the caller.
        status = Pk12ModuleUtil::FindExistingPrivKeyAndCert(pInPk11Slot,
                                                            renameDetails.pDerCert,
                                                            pOutCert,
                                                            pOutPrivKey);
        if (status != SECSuccess)
        {
            //  This should *NEVER* happen as the cert/key should have been
            //  imported above with our nickname.
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) ERROR: cert/key imported, but can't find?!\n");
            status = SECFailure;
            break;
        }

        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ProcessP12Data) done for slot %p, nickname \'%s\'\n",
                                          pInPk11Slot,
                                          pInOutNickname);

    } while (NN_STATIC_CONDITION(false));

    if (lock != nullptr)
    {
        SECMOD_ReleaseWriteLock(lock);
    }

    if (renameDetails.pDerCert != nullptr)
    {
        SECITEM_FreeItem(renameDetails.pDerCert, PR_TRUE);
        renameDetails.pDerCert = nullptr;
        renameDetails.pNickname = nullptr;
    }

    if (p12DecCtx != NULL)
    {
        SEC_PKCS12DecoderFinish(p12DecCtx);
    }

    if (pTmpNick != nullptr)
    {
        PL_strfree(pTmpNick);
        pTmpNick = nullptr;
    }

    return status;
}    //  NOLINT(impl/function_size)


SECItem* Pk12ModuleUtil::CallbackP12NicknameCollision(
    SECItem* pInOldNickname,
    PRBool* pIsCancel,
    void* pArg)
{
    SECItem* pRetItem = NULL;
    PRBool   isShouldCancel = PR_TRUE;

    if(pInOldNickname == nullptr)
    {
        CERTCertificate* pCert = reinterpret_cast<CERTCertificate*>(pArg);
        char*            pTmpPtr;

        // Try common name first
        if(pCert != NULL)
        {
            pTmpPtr = CERT_GetCommonName(const_cast<const CERTName*>(&pCert->subject));
            if(pTmpPtr != NULL)
            {
                pRetItem = SECITEM_AllocItem(NULL, NULL, static_cast<unsigned int>(PORT_Strlen(pTmpPtr) + 1));
                if(pRetItem != nullptr)
                {
                    PORT_Memset(pRetItem->data, 0, pRetItem->len);
                    PORT_Memcpy(pRetItem->data, pTmpPtr, PORT_Strlen(pTmpPtr));
                    isShouldCancel = PR_FALSE;
                    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(CallbackP12NicknameCollision) no nickname for cert, using %s\n",
                        pTmpPtr);
                }
                PORT_Free(pTmpPtr);
            }
        }
    }
    else
    {
        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(CallbackP12NicknameCollision) not implemented, skipping nickname %s\n",
            pInOldNickname);
    }

    *pIsCancel = isShouldCancel;
    return pRetItem;
}


SECStatus Pk12ModuleUtil::CallbackP12NicknameRename(const CERTCertificate*  cert,
                                                    const SECItem*          defaultNickname,
                                                    SECItem**               newNickname,
                                                    void*                   arg)
{
    SECStatus                   ret = SECSuccess;
    Pk12RenameDetails           *renameDetails = reinterpret_cast<Pk12RenameDetails *>(arg);
    uint32_t                    nicknameOverrideLen;
    SECItem                     *newNickHolder = nullptr;
    char                        *dfltNick = nullptr;

    do
    {
        if (arg == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(CallbackP12NicknameRename) no override provided!\n");
            ret = SECFailure;
            break;
        }

        if (defaultNickname != nullptr)
        {
            dfltNick = reinterpret_cast<char *>(defaultNickname->data);
        }

        //  See if this cert matches what we need/expect by checking against
        //  the DER cert.
        if (!SECITEM_ItemsAreEqual(&cert->derCert, renameDetails->pDerCert))
        {
            //  Cert is not the one we are interested in, take whatever the
            //  default is.
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(CallbackP12NicknameRename) cert %p (%s) is not the one we are looking for\n",
                                              cert,
                                              (cert->nickname != nullptr) ? cert->nickname : "{null}");
            *newNickname = nullptr;
            break;
        }

        nicknameOverrideLen = PL_strlen(renameDetails->pNickname);

        //  We don't care what the current nickname is, we need to override it
        //  to whatever was provided by our P12 import routine.  Do this by
        //  allocating a new SECItem and copying out the data.
        newNickHolder = SECITEM_AllocItem(NULL, NULL, nicknameOverrideLen + 1);
        if (newNickHolder == nullptr)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(CallbackP12NicknameRename) unable to alloc mem for override\n");
            ret = SECFailure;
            break;
        }

        newNickHolder->type = siAsciiString;
        memcpy(newNickHolder->data,
               renameDetails->pNickname,
               nicknameOverrideLen);
        newNickHolder->data[nicknameOverrideLen] = 0;
        newNickHolder->len = nicknameOverrideLen;

        //  We are good to go, provide the new holder to the caller.  NSS will release
        //  it as needed.
        *newNickname = newNickHolder;
        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(CallbackP12NicknameRename) renamed cert %p from \'%s\' to \'%s\'\n",
                                          cert,
                                          (dfltNick != nullptr) ? dfltNick : "{null}",
                                          renameDetails->pNickname);
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


SECStatus Pk12ModuleUtil::FindExistingPrivKeyAndCert(PK11SlotInfo      *pSlot,
                                                     SECItem           *pDerCert,
                                                     CERTCertificate   **pCert,
                                                     SECKEYPrivateKey  **pKey)
{
    SECStatus                   ret = SECFailure;

    do
    {
        //  This cert has a key associated with it.  Get the cert
        //  based on the DER data, then get the private key.  If both
        //  are present then we need to give them to the caller.  If not,
        //  continue (though there should *not* be more than 1
        //  cert/key combo in a single PKCS#12 bundle for us.)
        CERTCertificate     *pTmpCert =
            PK11_FindCertFromDERCertItem(pSlot, pDerCert, nullptr);
        if (pTmpCert == nullptr)
        {
            //  The cert is not in the slot, error out
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(FindExistingPrivKeyAndCert) cert not in slot\n");
            break;
        }

        //  We have the cert, find the matching private key
        SECKEYPrivateKey    *pTmpKey =
            PK11_FindPrivateKeyFromCert(pSlot, pTmpCert, nullptr);
        if (pTmpKey == nullptr)
        {
            //  This should *NEVER* happen!
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(FindExistingPrivKeyAndCert) WARNING: public key found, but not private key\n");
            CERT_DestroyCertificate(pTmpCert);
            pTmpCert = nullptr;
            break;
        }

        //  Excellent!  We have a match.  Give the cert and key
        //  to the caller and update the return value.
        NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(FindExistingPrivKeyAndCert) cert/key already imported, return cert %p, key %p\n",
                                          pTmpCert,
                                          pTmpKey);
        *pCert = pTmpCert;
        *pKey  = pTmpKey;
        ret = SECSuccess;
    } while (NN_STATIC_CONDITION(false));

    return ret;
}


SECItem* Pk12ModuleUtil::FindDERCertForPrivKey(SEC_PKCS12DecoderContext *pCtx)
{
    SECItem                     *pRet = nullptr;

    SECStatus                   status;

    do
    {
        const SEC_PKCS12DecoderItem     *pItem = nullptr;

        status = SEC_PKCS12DecoderIterateInit(pCtx);
        if (status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(FindExistingPrivKeyAndCert) failed to init decoder iterator\n");
            break;
        }

        for (status = SEC_PKCS12DecoderIterateNext(pCtx, &pItem);
             status == SECSuccess;
             status = SEC_PKCS12DecoderIterateNext(pCtx, &pItem))
        {
            if (pItem->hasKey == 0)
            {
                //  Cannot be our cert, there is no key with it
                continue;
            }

            //  We found the cert that has a private key included in it.
            //  Just grab the DER data pointer and return it.
            pRet = SECITEM_DupItem(pItem->der);
            break;
        }

    } while (NN_STATIC_CONDITION(false));

    return pRet;
}


// ------------------------------------------------------------------------------------------------
//
// Pk11ModuleUtil
//
// ------------------------------------------------------------------------------------------------

// ------------------------------------------------------------------------------------------------
// Public method
// ------------------------------------------------------------------------------------------------
nn::Result Pk11ModuleUtil::CheckAndSetupDefaultSlot()
{
    nn::Result                  ret = ResultSuccess();
    PK11SlotInfo                *slot = nullptr;

    do
    {
        slot = PK11_GetInternalKeySlot();
        if (PK11_NeedUserInit(slot))
        {
            NN_DETAIL_SSL_DBG_PRINT("[ssl] slot needs setup\n");
            SECStatus status = SECU_ChangePW(slot, NULL, NULL);
            if (status != SECSuccess)
            {
                NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to init pw on slot: %d (%d)\n",
                                        status,
                                        PORT_GetError());
                ret = ResultErrorLower();
                break;
            }
            else
            {
                NN_DETAIL_SSL_DBG_PRINT("[ssl] slot pw init, now auth\n");
            }

            status = PK11_Authenticate(slot, PR_TRUE, NULL);
            if (status != SECSuccess)
            {
                NN_DETAIL_SSL_DBG_PRINT("[ssl] failed to auth slot: %d (%d)\n",
                                        status,
                                        PORT_GetError());
                ret = ResultErrorLower();
                break;
            }
            else
            {
                NN_DETAIL_SSL_DBG_PRINT("[ssl] slot auth done\n");
            }
        }
    } while (NN_STATIC_CONDITION(false));

    if (slot != nullptr)
    {
        PK11_FreeSlot(slot);
        slot = nullptr;
    }

    return ret;
}


// ------------------------------------------------------------------------------------------------
// Public method
// ------------------------------------------------------------------------------------------------
nn::Result Pk11ModuleUtil::ImportPk12(Pk12ImportDetails  *pInDetails,
                                      char               *pInOutNickname,
                                      uint32_t           maxNameLen,
                                      CERTCertificate    **pOutCert,
                                      SECKEYPrivateKey   **pOutPrivKey)
{
    PK11SlotInfo                *pPkcs11Slot;

    NN_SDK_REQUIRES_NOT_NULL(pInDetails);
    if (pInDetails == nullptr)
    {
        return ResultInvalidReference();
    }

    NN_SDK_REQUIRES_NOT_NULL(pInOutNickname);
    if ((pInOutNickname == nullptr) || (maxNameLen == 0))
    {
        return ResultInvalidPointer();
    }

    if (pInDetails->pInP12Data == nullptr)
    {
        return ResultInvalidReference();
    }

    SECItem*      pUnicodePwItem = nullptr;
    SECItem*      pP12FileItem   = nullptr;
    nn::Result    result         = nn::ResultSuccess();
    const char    *pInP12Data    = pInDetails->pInP12Data;
    const char    *pInPwData     = pInDetails->pInPwData;
    uint32_t      p12DataSize    = pInDetails->p12DataLen;
    uint32_t      pwDataSize     = pInDetails->pwDataLen;

    pPkcs11Slot = PK11_GetInternalKeySlot();
    if (pPkcs11Slot == nullptr)
    {
        return ResultErrorLower();
    }

    do {
        SECStatus secStatus;

        // ----------------------------------------------------------------------------------------
        // PK12 setup
        // ----------------------------------------------------------------------------------------
        pUnicodePwItem = Pk12ModuleUtil::PrepPassword(pInPwData, pwDataSize);
        if (pUnicodePwItem == nullptr)
        {
            result = ResultInsufficientMemory();
            break;
        }

        pP12FileItem = SECITEM_AllocItem(NULL, NULL, p12DataSize);
        if (pP12FileItem == nullptr)
        {
            result = ResultInsufficientMemory();
            break;
        }

        memcpy(pP12FileItem->data, pInP12Data, p12DataSize);
        pP12FileItem->len = p12DataSize;

        secStatus = Pk12ModuleUtil::ProcessP12Data(pPkcs11Slot,
                                                   pUnicodePwItem,
                                                   pP12FileItem,
                                                   pInOutNickname,
                                                   maxNameLen,
                                                   pOutCert,
                                                   pOutPrivKey);
        if (secStatus != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("(ImportPk12) failed to process P12 file: %d\n", secStatus);
            result = ResultSslErrorReceivedUnkownData();
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    if (pP12FileItem != nullptr)
    {
        SECITEM_ZfreeItem(pP12FileItem, PR_TRUE);
    }

    if (pUnicodePwItem != nullptr)
    {
        SECITEM_ZfreeItem(pUnicodePwItem, PR_TRUE);
    }

    if (pPkcs11Slot != nullptr)
    {
        PK11_FreeSlot(pPkcs11Slot);
        pPkcs11Slot = nullptr;
    }

    return result;
}


void Pk11ModuleUtil::DumpSlotInformation()
{
    PK11SlotInfo* pInSlot = PK11_GetInternalKeySlot();

    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT_RAW("-----PK11SlotInfo----------------------------------\n");
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_GetTokenName: %s\n",
        PK11_GetTokenName(pInSlot));
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_GetSlotName: %s\n",
        PK11_GetSlotName(pInSlot));
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_GetTokenName: %s\n",
        PK11_GetTokenName(pInSlot));


    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("IsReadOnly: %s\n",
        (PK11_IsReadOnly(pInSlot) == PR_TRUE)?"TRUE":"FALSE");
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_IsInternal: %s\n",
        (PK11_IsInternal(pInSlot) == PR_TRUE)?"TRUE":"FALSE");
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_IsInternalKeySlot: %s\n",
        (PK11_IsInternalKeySlot(pInSlot) == PR_TRUE)?"TRUE":"FALSE");

    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_NeedLogin: %s\n",
        (PK11_NeedLogin(pInSlot) == PR_TRUE)?"TRUE":"FALSE");
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_IsFriendly: %s\n",
        (PK11_IsFriendly(pInSlot) == PR_TRUE)?"TRUE":"FALSE");
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_IsHW: %s\n",
        (PK11_IsHW(pInSlot) == PR_TRUE)?"TRUE":"FALSE");
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_IsRemovable: %s\n",
        (PK11_IsRemovable(pInSlot) == PR_TRUE)?"TRUE":"FALSE");

    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_NeedUserInit: %s\n",
        (PK11_NeedUserInit(pInSlot) == PR_TRUE)?"TRUE":"FALSE");
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_ProtectedAuthenticationPath: %s\n",
        (PK11_ProtectedAuthenticationPath(pInSlot) == PR_TRUE)?"TRUE":"FALSE");
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_IsDisabled: %s\n",
        (PK11_IsDisabled(pInSlot) == PR_TRUE)?"TRUE":"FALSE");
    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT("PK11_HasRootCerts: %s\n",
        (PK11_HasRootCerts(pInSlot) == PR_TRUE)?"TRUE":"FALSE");

    NN_DETAIL_SSL_DBG_CERTSTORE_PRINT_RAW("---------------------------------------------------\n");

    if (pInSlot != nullptr)
    {
        PK11_FreeSlot(pInSlot);
        pInSlot = nullptr;
    }
}


SECStatus Pk11ModuleUtil::DecodeCertCb(void     *pArg,
                                       SECItem  **pCertItems,
                                       int      certsCount)
{
    SECStatus                   status = SECFailure;
    CertDecoderDetail           *pDecDetail = (CertDecoderDetail *)pArg;
    SECItem                     tmpItem;

    memset(&tmpItem, 0, sizeof(tmpItem));

    do
    {
        //  We expect to only receive a single decoded item (a certificate)
        if ((certsCount != 1) || (pCertItems == nullptr))
        {
            NN_DETAIL_SSL_DBG_PRINT("(DecodeCertCb) Invalid cert (certsCount:%d) (pCertItems:%s)\n",
                certsCount, (pCertItems)?("EXISTS"):("NULL"));
            break;
        }

        status = SECITEM_CopyItem(nullptr, &tmpItem, *pCertItems);
        if (status != SECSuccess)
        {
            break;
        }

        //  Use the default "cert DB", which is really the default trust
        //  domain and backing crypto context.  The same backing trust and
        //  crypto context is used for temp PKCS#11 slots, so this is ok.
        CERTCertDBHandle *certDbHandle = CERT_GetDefaultCertDB();
        pDecDetail->pCert =
            CERT_NewTempCertificate(certDbHandle,
                                    &tmpItem,
                                    (char *)pDecDetail->pNickname,
                                    PR_FALSE,
                                    PR_TRUE);
        if (pDecDetail->pCert == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("(DecodeCertCb) failed to create new cert (%d|%s)",
                PR_GetError(), PR_ErrorToName(PR_GetError()));
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT("(DecodeCertCb) Created temp cert!\n");
        status = SECSuccess;
    } while (NN_STATIC_CONDITION(false));

    if (tmpItem.data != nullptr)
    {
        SECITEM_FreeItem(&tmpItem, PR_FALSE);
    }

    return status;
}


SECStatus Pk11ModuleUtil::SslCertTraverseCb(CERTCertificate *pCert,
                                            void            *arg)
{
    CERTCertificate             **pOutCert = (CERTCertificate **)arg;

    do
    {
        //  This traveral will only give us a cert which matches the
        //  nickname we specified.  There should be only one, so
        //  as long as we have not already set the cert, grab it.
        if (*pOutCert != nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[SslCertTraverseCb] already found match, skip this\n");
            break;
        }

        *pOutCert = CERT_DupCertificate(pCert);
    } while(NN_STATIC_CONDITION(false));

    //  Always return success, the caller of the traversal will
    //  handle if we didn't get a cert.
    return SECSuccess;
}


nn::Result Pk11ModuleUtil::GetClientPki(const char       *pInNickname,
                                        CERTCertificate  **pOutCert,
                                        SECKEYPrivateKey **pOutPrivKey)
{
    nn::Result                  ret = ResultSuccess();
    SECMODListLock              *listLock = nullptr;
    PK11SlotInfo                *pSlot = nullptr;
    CERTCertificate             *pInternalClientCert = nullptr;
    SECKEYPrivateKey            *pInternalClientKey = nullptr;
    SECStatus                   status;

    do
    {
        pSlot = InternalPkiManager::GetPk11Slot();
        if (pSlot == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetClientPki] fatal: failed to get PKCS#11 slot w/device info\n");
            ret = ResultErrorLower();
            break;
        }

        //  Traverse the list of all certs in the slot and look
        //  for certs which a) have the desired nickname and b) have a private key.
        SECItem  nickname;
        nickname.len  = PL_strlen(pInNickname);
        nickname.data = (unsigned char *)pInNickname;

        listLock = SECMOD_GetDefaultModuleListLock();
        SECMOD_GetWriteLock(listLock);
        status = PK11_TraverseCertsForNicknameInSlot(&nickname,
                                                     pSlot,
                                                     Pk11ModuleUtil::SslCertTraverseCb,
                                                     &pInternalClientCert);
        if ((status != SECSuccess) || (pInternalClientCert == nullptr))
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetClientPki] unable to find client cert\n");
            ret = ResultErrorLower();
            break;
        }

        pInternalClientKey =
            PK11_FindPrivateKeyFromCert(pSlot, pInternalClientCert, nullptr);
        if (pInternalClientKey == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[GetClientPki] unable to find client key\n");
            ret = ResultErrorLower();
            break;
        }

        NN_DETAIL_SSL_DBG_PRINT("[GetClientPki] returning cert and key\n");
        *pOutCert = pInternalClientCert;
        *pOutPrivKey  = pInternalClientKey;
    } while (NN_STATIC_CONDITION(false));

    if (listLock != nullptr)
    {
        SECMOD_ReleaseWriteLock(listLock);
    }

    if (!ret.IsSuccess())
    {
        if (pInternalClientCert != nullptr)
        {
            CERT_DestroyCertificate(pInternalClientCert);
            pInternalClientCert = nullptr;
        }
    }

    if (pSlot != nullptr)
    {
        PK11_FreeSlot(pSlot);
        pSlot = nullptr;
    }

    return ret;
}


}}}
