﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ssl/ssl_Types.h>
#include <nn/ssl/detail/ssl_Build.h>

#include "ssl_NssCommon.h"
#include "ssl_Util.h"
#include "ssl_PolicyOidBuiltIn.h"
#include "ssl_EvCertUtil.h"
#include "ssl_EvCertUtilManager.h"
#include "debug/ssl_DebugUtil.h"

namespace nn { namespace ssl { namespace detail {


// ------------------------------------------------------------------------------------------------
//
// EvCertUtilManager class
//
// ------------------------------------------------------------------------------------------------
nn::Result EvCertUtilManager::Initialize()
{
    NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER();

    nn::Result result = nn::ResultSuccess();
    NN_DETAIL_SSL_DBG_PRINT("[EvCertUtilManager::Initialize] Start\n");
    do
    {
        result = EvCertUtil::BuiltInPolicyOids::Initialize();
    } while (NN_STATIC_CONDITION(false));

    if (result.IsFailure())
    {
        EvCertUtil::BuiltInPolicyOids::Finalize();
    }
    NN_DETAIL_SSL_DBG_PRINT("[EvCertUtilManager::Initialize] End\n");
    return result;
}


void EvCertUtilManager::Finalize()
{
    NN_DETAIL_SSL_DBG_PRINT("[EvCertUtilManager::Finalize] Start\n");
    EvCertUtil::BuiltInPolicyOids::Finalize();
    NN_DETAIL_SSL_DBG_PRINT("[EvCertUtilManager::Finalize] End\n");
}


// ------------------------------------------------------------------------------------------------
//
// EvCertUtil class
//
// ------------------------------------------------------------------------------------------------
// ------------------------------------------------------------------------------------------------
// EvCertUtil: Public methods
// ------------------------------------------------------------------------------------------------
// Encode policy OID string to SECOidTag. NSS converts the string and assign SECOidTag to each
// string. This is how NSS maintains several objects by ID number and hash.
nn::Result EvCertUtil::EncodePolicyOidString(SECOidTag* pOutPolicyOid, const char* pInOidString)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPolicyOid);
    NN_SDK_REQUIRES_NOT_NULL(pInOidString);

    nn::Result result = nn::ResultSuccess();
    do
    {
        if (pOutPolicyOid == nullptr || pInOidString == nullptr)
        {
            result = ResultInvalidReference();
            break;
        }

        SECItem oidItem;
        oidItem.data = nullptr;
        oidItem.len  = 0;
        SECStatus status = SEC_StringToOID(nullptr, &oidItem, pInOidString, PL_strlen(pInOidString));
        if (status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("[EvCertUtil::EncodePolicyOidString] SEC_StringToOID failed (%s)(pInOidString:%s)\n",
                PR_ErrorToName(PR_GetError()), (pInOidString == nullptr)?"NULL":pInOidString);

            result = ResultErrorLower();
            break;
        }

        SECOidData oidData;
        oidData.oid.len            = oidItem.len;
        oidData.oid.data           = oidItem.data;
        oidData.offset             = SEC_OID_UNKNOWN;
        oidData.desc               = pInOidString;
        oidData.mechanism          = CKM_INVALID_MECHANISM;
        oidData.supportedExtension = INVALID_CERT_EXTENSION;

        *pOutPolicyOid = SECOID_AddEntry(&oidData);
        if (*pOutPolicyOid == SEC_OID_UNKNOWN)
        {
            NN_DETAIL_SSL_DBG_PRINT("[BuiltInPolicyOids::RegisterPolicyOid] SECOID_AddEntry failed (%s)\n",
                PR_ErrorToName(PR_GetError()));
            result = ResultErrorLower();
        }

        SECITEM_FreeItem(&oidItem, PR_FALSE);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


// Returns error detail in pOutError. This function checks:
// - If there is the certificate extension field in the passed certificate
// - If the policy OID in the passed certificate is found in passed PolicyOidInfo
// This function sets below errors
//   SEC_ERROR_EXTENSION_NOT_FOUND      : No certificate extension field is found
//   SEC_ERROR_UNRECOGNIZED_OID         : OID in the certificate is not found in PolicyOidInfo
//   SEC_ERROR_POLICY_VALIDATION_FAILED : Default error when none of above
nn::Result EvCertUtil::GetErrorDetail(
    SECErrorCodes* pOutError,
    PolicyOidInfo* pInOidInfo,
    CERTCertificate* pInCert)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutError);
    NN_SDK_REQUIRES_NOT_NULL(pInOidInfo);
    NN_SDK_REQUIRES_NOT_NULL(pInCert);

    nn::Result result = nn::ResultSuccess();
    do
    {
        if (pOutError == nullptr || pInOidInfo == nullptr || pInCert == nullptr)
        {
            result = ResultInvalidReference();
            break;
        }

        *pOutError = SEC_ERROR_POLICY_VALIDATION_FAILED;

        //  Find certificate extension field
        SECItem policiesExtentionItem;
        SECStatus status = CERT_FindCertExtension(
            pInCert,
            SEC_OID_X509_CERTIFICATE_POLICIES,
            &policiesExtentionItem);
        if (status != SECSuccess)
        {
            NN_DETAIL_SSL_DBG_PRINT("[EvCertUtil::GetErrorDetail] Converting to SEC_ERROR_EXTENSION_NOT_FOUND.\n");
            *pOutError = SEC_ERROR_EXTENSION_NOT_FOUND;
            break;
        }

        CERTCertificatePolicies* pPolicies     = nullptr;
        CERTPolicyInfo*          pPolicyInfo   = nullptr;
        CERTPolicyInfo**         ppPolicyInfos = nullptr;

        //  Decode extension field
        pPolicies = CERT_DecodeCertificatePoliciesExtension(&policiesExtentionItem);
        SECITEM_FreeItem(&policiesExtentionItem, PR_FALSE);

        if (pPolicies == nullptr)
        {
            NN_DETAIL_SSL_DBG_PRINT("[EvCertUtil::GetErrorDetail] Failed to decode extension field.\n");
            *pOutError = SEC_ERROR_EXTENSION_NOT_FOUND;
            break;
        }

        //  Walk-through policies and see if OID of the certificate is found in PolicyOidInfo.
        //  If it is not found, that means EV certificate verification failed because OID is
        //  considered as non-trusted on our platform.
        ppPolicyInfos = pPolicies->policyInfos;
        bool isOidFound = false;
        while (*ppPolicyInfos != nullptr)
        {
            pPolicyInfo = *ppPolicyInfos++;

            if (pPolicyInfo->oid != SEC_OID_UNKNOWN)
            {
                if (pInOidInfo->IsOidSet(pPolicyInfo->oid))
                {
                    isOidFound = true; //  Found match
                }
            }
        }

        if (isOidFound != true)
        {
            NN_DETAIL_SSL_DBG_PRINT("[EvCertUtil::GetErrorDetail] Converting to SEC_ERROR_UNRECOGNIZED_OID.\n");
            *pOutError = SEC_ERROR_UNRECOGNIZED_OID;
        }

        CERT_DestroyCertificatePoliciesExtension(pPolicies);
    } while (NN_STATIC_CONDITION(false));

    return result;
}


// ------------------------------------------------------------------------------------------------
//
// BuiltInPolicyOids class
//
// ------------------------------------------------------------------------------------------------
// ------------------------------------------------------------------------------------------------
// BuiltInPolicyOids: Static members
// ------------------------------------------------------------------------------------------------
EvCertUtil::OidTagListType EvCertUtil::BuiltInPolicyOids::g_DataList;


// ------------------------------------------------------------------------------------------------
// BuiltInPolicyOids: Public static methods
// ------------------------------------------------------------------------------------------------
// Initialize built-in policy OIDs
nn::Result EvCertUtil::BuiltInPolicyOids::Initialize()
{
    nn::Result result = nn::ResultSuccess();
    NN_DETAIL_SSL_DBG_PRINT("[BuiltInPolicyOids::Initialize] Start\n");
    do
    {
        for (int i = 0; i < ELEMENT_COUNT_IN_ARRAY(BuiltInPolicyOid::data); i++)
        {
            for (int j = 0; j < ELEMENT_COUNT_IN_ARRAY(BuiltInPolicyOid::data[i].policyOids); j++)
            {
                if (BuiltInPolicyOid::data[i].policyOids[j][0] == '\0')
                {
                    break; // No policy ID is found
                }

                OidTagListElement* pPolicyListElement = new OidTagListElement;
                if(nullptr == pPolicyListElement)
                {
                    NN_DETAIL_SSL_DBG_PRINT("[BuiltInPolicyOids::Initialize] insufficient memory\n");
                    result = ResultInsufficientMemory();
                    break;
                }

                if (EncodePolicyOidString(&pPolicyListElement->oidTag, BuiltInPolicyOid::data[i].policyOids[j]).IsFailure())
                {
                    NN_DETAIL_SSL_DBG_PRINT("[BuiltInPolicyOids::Initialize] OID registration failed\n");
                    result = ResultErrorLower(); // should not happen

                    delete pPolicyListElement;
                    pPolicyListElement = nullptr;

                    break;
                }

                g_DataList.push_back(*pPolicyListElement);

                NN_DETAIL_SSL_DBG_PRINT("[BuiltInPolicyOids::Initialize] Registered %s (ID:%d)\n",
                    BuiltInPolicyOid::data[i].policyOids[j], pPolicyListElement->oidTag);
            }

            if (result.IsFailure())
            {
                NN_DETAIL_SSL_WARN_PRINT("Built-in policy OID initialization falied (non-fatal)\n");
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    if (result.IsFailure())
    {
        BuiltInPolicyOids::Finalize();
    }

    NN_DETAIL_SSL_DBG_PRINT("[BuiltInPolicyOids::Initialize] End\n");
    return result;
}


void EvCertUtil::BuiltInPolicyOids::Finalize()
{
    NN_DETAIL_SSL_DBG_PRINT("[BuiltInPolicyOids::Finalize] Start\n");

    while (g_DataList.size() > 0)
    {
        auto* pData = &g_DataList.front();
        g_DataList.pop_front();

        delete pData;
    }

    NN_DETAIL_SSL_DBG_PRINT("[BuiltInPolicyOids::Finalize] End\n");
}


size_t EvCertUtil::BuiltInPolicyOids::GetCount()
{
    return g_DataList.size();
}


nn::Result EvCertUtil::BuiltInPolicyOids::GetOidInfo(PolicyOidInfo* pOutOidInfo)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutOidInfo);
    nn::Result result = nn::ResultSuccess();

    do
    {
        if (pOutOidInfo == nullptr)
        {
            result = ResultInvalidReference();
            break;
        }

        for(auto it = g_DataList.begin(); it != g_DataList.end(); ++it)
        {
            SECOidTag* pTmpOidTag = &it->oidTag;
            result = pOutOidInfo->AddOids(pTmpOidTag, 1);
            if (result.IsFailure())
            {
                break;
            }
        }

        NN_DETAIL_SSL_DBG_PRINT("[BuiltInPolicyOids::GetOidInfo] done.\n");
    } while (NN_STATIC_CONDITION(false));

    if (result.IsFailure())
    {
        NN_DETAIL_SSL_DBG_PRINT("[BuiltInPolicyOids::GetOidInfo] failed.\n");
    }

    return result;
}


// ------------------------------------------------------------------------------------------------
//
// PolicyOidInfo class
//
// ------------------------------------------------------------------------------------------------
// ------------------------------------------------------------------------------------------------
// PolicyOidInfo: Public methods
// ------------------------------------------------------------------------------------------------
EvCertUtil::PolicyOidInfo::PolicyOidInfo() : m_pOidsHead(nullptr), m_oidsCount(0), m_listSize(0)
{
}


EvCertUtil::PolicyOidInfo::~PolicyOidInfo()
{
    this->Cleanup();
}

// Allocate the array of SECOidTag
nn::Result EvCertUtil::PolicyOidInfo::Setup(size_t oidCount)
{
    nn::Result result = nn::ResultSuccess();

    do
    {
        if (oidCount == 0)
        {
            break; // Nothing to setup
        }

        m_pOidsHead = new SECOidTag[oidCount];
        if (m_pOidsHead == nullptr)
        {
            result = ResultInsufficientMemory();
            break;
        }

        m_listSize = oidCount;
        NN_DETAIL_SSL_DBG_PRINT("[PolicyOidInfo::Setup] done (size: %d).\n", m_listSize);
    } while (NN_STATIC_CONDITION(false));

    if (result.IsFailure())
    {
        if (m_pOidsHead != nullptr)
        {
            delete[] m_pOidsHead;
        }
        m_oidsCount = 0;
        m_listSize = 0;
    }

    return result;
}


void EvCertUtil::PolicyOidInfo::Cleanup()
{
    if (m_pOidsHead != nullptr)
    {
        delete[] m_pOidsHead;
        m_pOidsHead = nullptr;
    }

    m_oidsCount = 0;
    m_listSize = 0;
}

// Set SECOidTags in the array of SECOidTags (which head is m_pOidsHead).
// This function may be called multiple times.
nn::Result EvCertUtil::PolicyOidInfo::AddOids(SECOidTag* pInOidHead, size_t count)
{
    NN_SDK_REQUIRES_NOT_NULL(pInOidHead);

    nn::Result result      = nn::ResultSuccess();
    size_t     storedCount = 0;
    do
    {
        if (pInOidHead == nullptr)
        {
            result = ResultInvalidReference();
            break;
        }

        if (count == 0)
        {
            NN_DETAIL_SSL_DBG_PRINT("[PolicyOidInfo::AddOids] Nothing to add.\n");
            break;
        }

        if ((m_oidsCount + count) > m_listSize)
        {
            NN_DETAIL_SSL_DBG_PRINT("[PolicyOidInfo::AddOids] The array is already full.\n");
            result = ResultInsufficientMemory();
            break;
        }

        do
        {
            m_pOidsHead[m_oidsCount++] = pInOidHead[storedCount++];
        } while (storedCount < count);
    } while (NN_STATIC_CONDITION(false));

    if (result.IsFailure())
    {
        if (storedCount > 0)
        {
            do
            {
                m_pOidsHead[m_oidsCount--] = SEC_OID_UNKNOWN; // Reset
            } while (--storedCount > 0);
        }
    }

    return result;
}


SECOidTag* EvCertUtil::PolicyOidInfo::GetHead()
{
    return m_pOidsHead;
}


size_t EvCertUtil::PolicyOidInfo::GetCount()
{
    return m_oidsCount;
}


bool EvCertUtil::PolicyOidInfo::IsOidSet(SECOidTag oid)
{
    bool isFound = false;
    do
    {
        if (m_pOidsHead == nullptr || m_listSize == 0 || m_oidsCount == 0)
        {
            break;
        }

        for (size_t i = 0; i < m_oidsCount; i++)
        {
            if (m_pOidsHead[i] == oid)
            {
                isFound = true;
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    return isFound;
}


}}}

