﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/ssl/detail/ssl_Common.h>
#include <nn/ssl/detail/ssl_BuiltinDataInfo.h>

namespace nn { namespace ssl { namespace detail {

/**
 * @brief
 * The BdfReader is a data reader used for extracting built-in data.  It cannot
 * be created directly, use the BdfReaderFactory::Create() method to create an
 * instance of the class for the platform.  This instance can then be used to
 * read the requested BDF.
 */
class BdfReader
{
private:
    BdfMetaHdr                  m_hdr;

protected:
    BdfReader();

public:
    enum BdfId
    {
        BdfId_Invalid       = 0xFFFFFFFF,
        BdfId_TrustedCerts  = 0x00000001,
        BdfId_Crl           = 0x00000002,
        BdfId_TestCrl       = 0x00001000,  //<! Factory instances create this only when initialize_for_test is enabled for ssl.test
    };

    virtual ~BdfReader() { };

    /**
     * @brief       Open and/or prepare to read the data.
     *
     * @return      Result of opening and preparing to read the data
     */
     virtual nn::Result Open() NN_NOEXCEPT = 0;

    /**
     * @brief       Get the number of data entries in the BDF
     *
     * @param[out]  pOutCount  Pointer to a location where the count is to be stored.
     *
     * @return      Result of getting the number of data entries in the BDF.
     */
    nn::Result GetDataCount(uint32_t *pOutCount) NN_NOEXCEPT;

    /**
     * @brief       Get a specific data entry's meta data from the BDF
     *
     * @param[out]  pOutMetaDataEntry  Pointer to the caller provided memory where
     *                                 the meta data should be placed.
     * @param[in]   dataIndex          Index of the data entry to retrieve.
     *
     * @return      Result of getting the requested data entry's meta data
     */
    nn::Result GetMetaDataEntry(BdfMetaDataEntry *pOutMetaDataEntry,
                                uint32_t         dataIndex) NN_NOEXCEPT;

    /**
     * @brief       Get the specific data entry's raw data
     *
     * @param[out]  pOutBuf         Pointer to a caller provide buffer where the data is copied
     * @param[in]   bufLen          The length of the buffer pointed to by pOutBuf, in bytes
     * @param[in]   pMetaDataEntry  Pointer to the meta data for the entry to get
     *
     * @return      Result of getting the requeste data
     */
    nn::Result GetData(uint8_t           *pOutBuf,
                       uint32_t          bufLen,
                       BdfMetaDataEntry  *pMetaDataEntry) NN_NOEXCEPT;

    /**
     * @brief       Read data from the BDF, copying it into the supplied buffer.
     *
     * @param[out]  pOutBytesRead  The amount of data read into the provided buffer.
     * @param[in]   offset         The offset from the start of the BDF in which to read.
     * @param[out]  pOutBuf        Pointer to a buffer were the BDF data should be read.
     * @param[in]   size           The number of bytes to read.
     *
     * @return      Result of reading the data from the TCF.
     */
    virtual nn::Result Read(size_t   *pOutBytesRead,
                            int64_t  offset,
                            uint8_t  *pOutBuf,
                            size_t   size) NN_NOEXCEPT = 0;
};


class BdfReaderFactory
{
public:
    static nn::Result Initialize() NN_NOEXCEPT;
    static nn::Result Finalize() NN_NOEXCEPT;

    /**
     * @brief  Get a new instance of the BdfReader which can be used to
     *         read data from the specific BDF on the platform.
     *
     * @param[in]  id  The identifier of the BDF to load.
     *
     * @return A new instance of a BdfReader, or nullptr if one cannot be
     *         created or the requested BDF is not found.
     */
    static BdfReader *Create(BdfReader::BdfId id) NN_NOEXCEPT;
};

} } }    //  nn::ssl::detail
