﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nnc/nn_Result.h>
#include <nnc/result/result_CConverter.h>

#include <nn/ssl/detail/ssl_Common.h>
#include "ssl_ApiBase.h"

namespace nn { namespace ssl { namespace detail {

SslApiBase::SslApiBase() :
    m_ReferenceCount(0)
{
}

SslApiBase::~SslApiBase()
{
    NN_SDK_ASSERT(
        (m_ReferenceCount == 0),
        "[ssl] The SSL library is not finalized yet (reference count: %d)\n",
        m_ReferenceCount);
}

nn::Result SslApiBase::Initialize() NN_NOEXCEPT
{
    if (m_ReferenceCount++ > 0)
    {
        return nn::ssl::ResultLibraryAlreadyInitialized();
    }
    else
    {
        return nn::ResultSuccess();
    }
}

nn::Result SslApiBase::Finalize() NN_NOEXCEPT
{
    if (m_ReferenceCount == 0)
    {
        return nn::ssl::ResultLibraryNotInitialized();
    }

    if (--m_ReferenceCount == 0)
    {
        return nn::ResultSuccess();
    }

    return nn::ssl::ResultInternalReferenceRemains();
}

// -------------------------------------------------------------------------------------------------
// Static methods
// -------------------------------------------------------------------------------------------------
nn::Result SslApiBase::GetSslResultFromValue(
    nn::Result* pOutResult,
    const char* pInValue,
    uint32_t inValueSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResult);
    NN_SDK_REQUIRES_NOT_NULL(pInValue);

    nn::Result rval = nn::ResultSuccess();
    nnResult   inResult;
    do
    {
        if ((pOutResult == nullptr) || (pInValue == nullptr))
        {
            rval =  nn::ssl::ResultInvalidPointer();
            break;
        }

        if (inValueSize < sizeof(nnResultInnerType))
        {
            rval =  nn::ssl::ResultBufferTooShort();
            break;
        }

        inResult._value = *(reinterpret_cast<const nnResultInnerType *>(pInValue));
        *pOutResult      = nn::result::ConvertFromC(inResult);
    } while (NN_STATIC_CONDITION(false));

    return rval;
}

}}}
