﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// ------------------------------------------------------------------------------------------------
// Debugging features are DISABLED
// ------------------------------------------------------------------------------------------------
#ifndef NN_DETAIL_SSL_ENABLE_PROCESS_DEBUG

#define NN_DETAIL_SSL_DBG_UTIL_INITIALIZE()
#define NN_DETAIL_SSL_DBG_UTIL_FINALIZE()
#define NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER()
#define NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_WITH_NAME(name)
#define NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_FOR_BGTASK()

#else

// ------------------------------------------------------------------------------------------------
// Debugging features are ENABLED
// ------------------------------------------------------------------------------------------------
#include <nn/os.h>
#include <nn/init.h>
#include <nn/mem.h>

#include <nn/nn_SdkLog.h>
#include <nn/ssl/detail/ssl_Common.h>
#include <nn/ssl/detail/ssl_Build.h>
#include <nn/ssl/detail/ssl_Log.h>
#include <nn/ssl/ssl_Types.h>
#include "ssl_DebugSessionCache.h"

#ifdef NN_BUILD_CONFIG_OS_WIN
#error 'Debug util doesn't support Windows, make sure to set ENABLE_SSL_PROCESS_DEBUG to false before building the tree for Windows'
#endif

// ------------------------------------------------------------------------------------------------
// Macros
// ------------------------------------------------------------------------------------------------
#define NN_DETAIL_SSL_DBG_UTIL_PRINT(...)                                                         \
    do {                                                                                          \
        NN_SDK_LOG(__VA_ARGS__);                                                                  \
    }while(NN_STATIC_CONDITION(false))

#define NN_DETAIL_SSL_DBG_UTIL_RETURN_WHEN_NO_ALLOCATOR()                                         \
    do {                                                                                          \
        if (m_pStdAllocator == nullptr)                                                           \
        {                                                                                         \
            return;                                                                               \
        }                                                                                         \
    }while(NN_STATIC_CONDITION(false))

#define NN_DETAIL_SSL_DBG_UTIL_INITIALIZE() g_DebugUtil.Initialize(nn::init::GetAllocator())
#define NN_DETAIL_SSL_DBG_UTIL_FINALIZE()   g_DebugUtil.Finalize()

#ifdef NN_BUILD_CONFIG_COMPILER_CLANG
#define NN_DETAIL_SSL_DBG_UTIL_METHOD_NAME __PRETTY_FUNCTION__
#else
#define NN_DETAIL_SSL_DBG_UTIL_METHOD_NAME __FUNCTION__
#endif // NN_BUILD_CONFIG_COMPILER_CLANG

#ifdef NN_DETAIL_SSL_ENABLE_PROCESS_DEBUG_DUMP_INIT_MEM
#define NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER()                                       \
    DebugUtil::ScopedHeapTracker scopedHeapTracker(&g_DebugUtil,                                  \
                                                   NN_DETAIL_SSL_DBG_UTIL_METHOD_NAME)

#define NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_WITH_NAME(name)                         \
    DebugUtil::ScopedHeapTracker scopedHeapTracker(&g_DebugUtil, name)
#else
#define NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER()
#define NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_WITH_NAME(name)
#endif // NN_DETAIL_SSL_ENABLE_PROCESS_DEBUG_DUMP_INIT_MEM

#ifdef NN_DETAIL_SSL_ENABLE_PROCESS_DEBUG_DUMP_BGTASK_MEM
#define NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_FOR_BGTASK()                            \
    DebugUtil::ScopedHeapTracker scopedHeapTracker(&g_DebugUtil,                                  \
                                                   NN_DETAIL_SSL_DBG_UTIL_METHOD_NAME)
#else
#define NN_DETAIL_SSL_DBG_UTIL_CREATE_SCOPED_HEAP_TRACKER_FOR_BGTASK()
#endif

// ------------------------------------------------------------------------------------------------
// Debug log utils
// ------------------------------------------------------------------------------------------------
namespace nn { namespace ssl { namespace detail {

class DebugUtil
{
public:
    typedef enum TaskType
    {
        TaskType_UpdateHeapStat,   //!< Update heap memory stats
        TaskType_DumpHeapStat,     //!< Dump heap memory stats
        TaskType_DumpSessionCache, //!< Dump session cache info
    } TaskType;

private:
    static const int WorkerMsecInterval = 10; //!< Base timer granularity of the util main thread
    static const int MessageQueueSize   = 32; //!< The number of messages queued for TaskType

    // --------------------------------------------------------------------------------------------
    // Worker
    // --------------------------------------------------------------------------------------------
    class Worker
    {
        static const int ThreadStackSize = (nn::os::StackRegionAlignment < 4096)?(4096):(nn::os::StackRegionAlignment);

        NN_ALIGNAS(nn::os::StackRegionAlignment) char m_ThreadStack[ThreadStackSize];
        uintptr_t                                     m_MessageQueueBuf[MessageQueueSize];
        nn::os::ThreadType                            m_Thread;
        uint32_t                                      m_MsecTimerInterval;
        nn::os::TimerEvent                            m_TimerEvent;
        nn::os::MessageQueue                          m_TaskMessageQueue;

        bool m_IsInitialized;
        bool m_IsReadyToCleanup;

        static void Executer(void* arg);

        bool                  IsReadyToCleanup();
        nn::os::TimerEvent*   GetTimerEvent();
        uint32_t              GetMsecInterval();
        nn::os::MessageQueue* GetMessageQueue();

    public:
        Worker();
        ~Worker();

        nn::Result Initialize(DebugUtil* pInUtilObj);
        nn::Result Finalize();
        void       AddTask(TaskType taskType);
    };

    // --------------------------------------------------------------------------------------------
    // Memory Tracker
    // --------------------------------------------------------------------------------------------
    class MemoryTracker
    {
    private:
        static const int StatsUpdateMsecInterval = 10;   //!< Timer granularity for stats update frequency
        static const int StatsDumpMsecInterval   = 5000; //!< Timer granularity for stats dump

        typedef struct HeapStats
        {
            nn::os::Tick lastDumpTick;
            nn::os::Tick lastUpdateTick;
            size_t       min;
            size_t       max;
        } HeapStats;

        typedef struct OneHeapTrackStat
        {
            OneHeapTrackStat* pNext;
            size_t            hash;
            nn::os::Tick      startTick;
            size_t            startSpace;
        } OneHeapTrackStat;

        bool                        m_IsDumpStats;
        DebugUtil*                  m_pDebugUtilObj;
        HeapStats                   m_HeapStats;
        OneHeapTrackStat*           m_pOneHeapTrackStatHead;
        nn::os::Mutex               m_OneHeapTrackStatLock;
        nn::mem::StandardAllocator* m_pStdAllocator;

        size_t            GetHash(const char* pInTrackName);
        void              AddOneTracker(OneHeapTrackStat* pInTracker);
        void              RemoveOneTracker(OneHeapTrackStat* pInTracker);
        OneHeapTrackStat* FindOneTracker(const char* pInTrackName);

    public:
        typedef struct TrackStats
        {
            uint64_t elapsedMsec;
            int64_t  delta;
            size_t   curHeapSpace;
        } TrackStats;

        MemoryTracker();
        ~MemoryTracker();
        nn::Result Initialize(DebugUtil* pInUtilObj, nn::mem::StandardAllocator* pInStdAllocator);
        nn::Result Finalize();
        void       ConfigurePeriodicDump(bool enable);
        void       DumpHeapStats();
        void       GetHeapStats(size_t* pOutCurSize, size_t* pOutMaxSize, size_t* pOutMinSize);
        void       UpdateHeapStats();
        void       RunScheduler();

        void       StartOneTrack(const char* pInTrackName);
        void       EndOneTrack(const char* pInTrackName);
        bool       EndOneTrack(TrackStats* pOutStats, const char* pInTrackName);
    };

    // --------------------------------------------------------------------------------------------
    // Private members
    // --------------------------------------------------------------------------------------------
    Worker            m_Worker;
    MemoryTracker     m_MemoryTracker;
    DebugSessionCache m_SessionCacheDebugger;

public:
    class ScopedHeapTracker
    {
    public:
        ScopedHeapTracker(DebugUtil* pInDebugUtil,  const char* pInName);
        ~ScopedHeapTracker();

    private:
        DebugUtil*  m_pDebugUtil;
        const char* m_pName;
    };

    void               Initialize(nn::mem::StandardAllocator* pInStdAllocator);
    void               Finalize();
    Worker*            GetWorker();
    MemoryTracker*     GetMemoryTracker();
    DebugSessionCache* GetSessionCacheDebugger();

    void GetHeapStats(size_t* pOutCurSize, size_t* pOutMaxSize, size_t* pOutMinSize);
    void ConfigurePeriodicDump(bool enable);
    void StartHeapTrack(const char* pInTrackName);
    bool EndHeapTrack(
        uint64_t*   pOutElapsedMsec,
        int64_t*    pOutDelta,
        size_t*     pOutCurHeapSpace,
        const char* pInTrackName);
};

extern DebugUtil g_DebugUtil;
}}}

#endif // NN_DETAIL_SSL_ENABLE_PROCESS_DEBUG

